using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using ChaKi.Common;
using ChaKi.Entity.Search;
using System.Diagnostics;

namespace ChaKi.ConditionPanes
{
    public partial class DepSearchPane : UserControl
    {
        private DepSearchCondition m_Model;
        private List<BunsetsuBox> m_Boxes;
        private List<RelationButton> m_Buttons;

        private Point m_DragStartPoint;
        private BunsetsuBox m_DragStartBox;
        private Point m_DragEndPoint;
        private bool m_Dragging = false;

        private static Pen m_Pen;
        private static Pen m_SelPen;

        static DepSearchPane()
        {
            m_Pen = new Pen(Color.Black, 1.5F);
            m_Pen.EndCap = System.Drawing.Drawing2D.LineCap.ArrowAnchor;
            m_SelPen = new Pen(Color.Red, 1.5F);
            m_SelPen.EndCap = System.Drawing.Drawing2D.LineCap.ArrowAnchor;
        }

        public DepSearchPane( DepSearchCondition model )
        {
            InitializeComponent();

            m_Boxes = new List<BunsetsuBox>();
            m_Buttons = new List<RelationButton>();

            m_Model = model;
            m_Model.OnModelChanged += new EventHandler( this.ModelChangedHandler );

            this.UpdateView();
        }

        public void SetCondition(DepSearchCondition cond)
        {
            m_Model = cond;
            m_Model.OnModelChanged += new EventHandler(this.ModelChangedHandler);
            UpdateView();
        }

        public DepSearchCondition GetCondition()
        {
            return m_Model;
        }

        private void UpdateView()
        {
            Debug.WriteLine("DepSearchPane::UpdateView");

//            this.Visible = false;
            this.SuspendLayout();

            m_Boxes.Clear();
            m_Buttons.Clear();
            this.Controls.Clear();
            this.PerformLayout();

            int n = 0;
            foreach (TagSearchCondition bunsetsucond in m_Model.BunsetsuConds)
            {
                // ̃{^
                RelationButton button = new RelationButton(-1,n);
                button.OnCommandEvent += new RelationCommandEventHandler(ButtonCommandEventHandler);
                this.Controls.Add(button);
                m_Buttons.Add(button);

                // OBox
                BunsetsuBox bbox = new BunsetsuBox(n, bunsetsucond.LexemeConds);
                this.Controls.Add(bbox);
                this.m_Boxes.Add(bbox);
                bbox.OnMouseDownEvent += new MouseEventHandler(MouseDownHandler);
                bbox.OnCenterlizedButtonClicked += new EventHandler(CenterlizedButtonClickedEventHandler);
                bbox.OnCommandEvent += new RelationCommandEventHandler(ButtonCommandEventHandler);
                n++;
            }
            // łẼ{^
            RelationButton rightbutton = new RelationButton(-1,n);
            rightbutton.OnCommandEvent += new RelationCommandEventHandler(ButtonCommandEventHandler);
            this.Controls.Add(rightbutton);
            m_Buttons.Add(rightbutton);

            this.ResumeLayout();

            this.RecalcLayout();
//            this.Visible = true;
        }

        /// <summary>
        /// PivotύX
        /// (̂߁AModel->View update Mechanism͎gpAڗŕύX)
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void CenterlizedButtonClickedEventHandler(object sender, EventArgs e)
        {
            PropertyBox pbox = sender as PropertyBox;
            for (int i = 0; i < m_Model.BunsetsuConds.Count; i++)
            {
                TagSearchCondition tagCond = m_Model.BunsetsuConds[i];
                BunsetsuBox bbox = m_Boxes[i];
                for (int j = 0; j < tagCond.LexemeConds.Count; j++) {
                    LexemeCondition lexCond = tagCond.LexemeConds[j];
                    PropertyBox box = bbox.Box[j];
                    if (box == pbox)
                    {
                        box.IsPivot = true;
                    }
                    else
                    {
                        box.IsPivot = false;
                    }
                }
            }
            Invalidate();
        }

        /// <summary>
        /// R}h{^
        /// O(BunsetsuBox)E(PropertyBox)ǂ̃nhŏB
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void ButtonCommandEventHandler(object sender, RelationCommandEventArgs e)
        {
            RelationButton button = sender as RelationButton;

            switch (e.Command)
            {
                case '+':
                    if (button.BunsetsuID < 0)
                    {
                        TagSearchCondition newitem = m_Model.InsertAt(button.ID);
                        if (newitem != null)
                        {
                            newitem.LexemeConds[0].IsPivot = false;
                        }
                    }
                    else
                    {
                        m_Model.InsertLexemeConditionAt(button.BunsetsuID, button.ID);
                    }
                    break;
                default:
                    button.Text = new string(e.Command, 1);
                    if (button.BunsetsuID < 0)
                    {
                        m_Model.SetConnection(button.ID, e.Command);
                    }
                    else
                    {
                        TagSearchCondition tcond = m_Model.BunsetsuConds[button.BunsetsuID];
                        tcond.SetConnection(button.ID, e.Command);
                    }
                    break;
            }
        }

        private void RecalcLayout()
        {
            int x = 5;
            int y = 5;
            int height = 10;
            if (m_Boxes.Count > 0)
            {
                height += (m_Boxes[0].Height+10);
            }
            int maxCount = Math.Min(m_Boxes.Count, m_Buttons.Count - 1);
            for (int i = 0; i < maxCount; i++)
            {
                RelationButton button = m_Buttons[i];
                button.Location = new Point(x, (height-button.Height)/2);
                x += (button.Width + 5);

                BunsetsuBox bbox = m_Boxes[i];
                bbox.Location = new Point(x, y);
                x += (bbox.Width + 5);
            }
            RelationButton rbutton = m_Buttons[m_Buttons.Count-1];
            rbutton.Location = new Point(x, (height - rbutton.Height) / 2);
            x += (rbutton.Width + 5);
            this.Width = x;
            this.Height = height;
        }

        void ModelChangedHandler(object sender, EventArgs e)
        {
            UpdateView();
        }

        /// <summary>
        /// PropertyBoxCenterized ButtoniԂjꂽ̏B
        /// BoxPivot\B
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnPropertyBoxCenterizedButtonClicked(object sender, EventArgs e)
        {
        }

        /// <summary>
        /// PropertyBox̃CxgʒmɂABox폜B
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnPropertyBoxDeleteClicked(object sender, EventArgs e)
        {
        }

        #region ˑhbMO
        // OBoxŃ}EXꂽˑhbOJn
        void MouseDownHandler(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                BunsetsuBox box = sender as BunsetsuBox;
                m_DragStartPoint = new Point((box.Bounds.Left + box.Bounds.Right / 2), box.Bounds.Bottom);
                m_DragStartBox = box;
                m_DragEndPoint = e.Location;
                m_Dragging = true;
                this.Capture = true;
            }
            else if (e.Button == MouseButtons.Right)
            {
                // ReNXgj[\
                this.contextMenu.Show(((Control)sender).PointToScreen(e.Location));
            }
        }

        private void MouseMoveHandler(object sender, MouseEventArgs e)
        {
            if (m_Dragging)
            {
                m_DragEndPoint = e.Location;
                Rectangle r = new Rectangle(0, m_DragStartPoint.Y, this.Width, m_DragStartPoint.Y);
                Invalidate(r);
            }
        }

        private void DepSearchPane_MouseUp(object sender, MouseEventArgs e)
        {
            if (!m_Dragging || m_DragStartBox == null)
            {
                return;
            }
            m_Dragging = false;
            m_DragEndPoint = e.Location;
            Rectangle r = new Rectangle(0, m_DragStartPoint.Y, this.Width, m_DragStartPoint.Y);
            Invalidate(r);
            this.Capture = false;
            Point p = e.Location;
            // MouseUpʒuɂ镶BoxT
            BunsetsuBox hit = null;
            foreach (BunsetsuBox b in m_Boxes)
            {
                if (b.Bounds.Contains(p))
                {
                    hit = b;
                    break;
                }
            }
            if (hit != null)
            {
                LinkCondition lc = new LinkCondition(m_DragStartBox.ID, hit.ID, string.Empty);
                m_Model.AddLinkConstraint(lc);
            }
        }
        #endregion

        private void DepSearchPane_Paint(object sender, PaintEventArgs e)
        {
            Graphics g = e.Graphics;

            if (m_Dragging)
            {
                Pen pen = true ? m_SelPen : m_Pen;
                Point[] pts = new Point[]
                {
                    m_DragStartPoint,
                    new Point(m_DragStartPoint.X,m_DragStartPoint.Y),
                    new Point(m_DragStartPoint.X,m_DragStartPoint.Y+15),
                    new Point(m_DragEndPoint.X,m_DragStartPoint.Y+15),
                    new Point(m_DragEndPoint.X,m_DragStartPoint.Y),
                };

                g.DrawLines(pen, pts);
            }
        }
    }
}
