﻿using System;
using System.Collections.Generic;
using System.Text;
using ChaKi.Entity.Search;
using ChaKi.Entity.Kwic;
using System.Xml.Serialization;

namespace ChaKi.Entity.Search
{
    public class SearchHistoryResetEventArgs : EventArgs
    {
    }

    public class SearchHistoryAddNodeEventArgs : EventArgs
    {
        public SearchHistory Hist;

        public SearchHistoryAddNodeEventArgs(SearchHistory hist)
        {
            this.Hist = hist;
        }
    }

    public class SearchHistory
    {
        public event EventHandler OnUpdateModel;

        // ツリー構造
        public static SearchHistory Root { get; set; }
        [XmlIgnore]
        public List<SearchHistory> Children { get; set; }

        // ノード自身の内容
        public string Name { get; set; }
        public string FilePath { get; set; }
        public SearchConditionsSequence CondSeq { get; set; }
        public KwicList KwicList { get; set; }
        public LexemeCountList LexemeList { get; set; }
        public CommandProgress Progress { get; set; }

        private int m_ShiftPos;     // Centerのシフト値

        // 履歴固有名称の元となる数値
        private static int seqno;

        static SearchHistory()
        {
            Root = new SearchHistory();
            seqno = 1;
        }


        /// <summary>
        /// 条件に応じた検索履歴モデルを生成する
        /// </summary>
        /// <param name="cond"></param>
        /// <returns></returns>
        public static SearchHistory Create(SearchConditionsSequence condSeq)
        {
            SearchHistory obj = new SearchHistory();
            obj.CondSeq = condSeq;
            obj.Progress = new CommandProgress();
            switch (condSeq.Last.ActiveSearch)
            {
                case SearchType.SentenceSearch:
                case SearchType.StringSearch:
                case SearchType.TagSearch:
                case SearchType.DepSearch:
                    obj.KwicList = new KwicList();
                    break;
                case SearchType.TagWordList:
                case SearchType.DepWordList:
                    obj.LexemeList = new LexemeCountList();
                    break;
            }
            obj.Name = string.Format("[{0}]{1}", seqno++, condSeq.Last.ActiveSearch.ToString());
            return obj;
        }

        public SearchHistory()
        {
            this.Children = new List<SearchHistory>();
            this.Name = "/";
            this.FilePath = null;
            this.CondSeq = null;
            this.KwicList = null;
            this.LexemeList = null;
            this.Progress = null;

            m_ShiftPos = 0;
        }

        public SearchHistory(SearchHistory hist)
        {
            this.Children = new List<SearchHistory>();
            this.Name = hist.Name;
            this.FilePath = hist.FilePath;
            this.CondSeq = hist.CondSeq;
            this.KwicList = hist.KwicList;
            this.LexemeList = hist.LexemeList;
            this.Progress = hist.Progress;

            m_ShiftPos = hist.m_ShiftPos;
        }

        public void Reset()
        {
            Children.Clear();

            if (Root.OnUpdateModel != null)
            {
                Root.OnUpdateModel(this, new SearchHistoryResetEventArgs());
            }
        }

        /// <summary>
        /// 自分の持つ子孫からhistと一致するヒストリ項目を探し、削除する.
        /// histが子を持つ場合は一緒に削除される.
        /// </summary>
        /// <param name="hist"></param>
        public void Delete(SearchHistory hist)
        {
            foreach (SearchHistory h in Children)
            {
                if (h.Name.Equals(hist.Name))
                {
                    Children.Remove(h);
                    return;
                }
                h.Delete(hist);
            }
        }

        public void AddChild(SearchHistory hist)
        {
            this.Children.Add(hist);

            if (Root.OnUpdateModel != null)
            {
                Root.OnUpdateModel(this, new SearchHistoryAddNodeEventArgs(hist));
            }
        }

        public SearchHistory FindHistory(string p)
        {
            if (this.Name.Equals(p))
            {
                return this;
            }
            // Search recursively
            foreach (SearchHistory hist in this.Children)
            {
                SearchHistory found = hist.FindHistory(p);
                if (found != null)
                {
                    return found;
                }
            }
            return null;
        }

        public void Shift(int shift)
        {
            m_ShiftPos += shift;
            this.KwicList.Shift(shift);

            TagSearchCondition tcond = this.CondSeq[this.CondSeq.Count - 1].TagCond;
            if (tcond != null)
            {
                tcond.Shift(shift);
            }
        }
    }
}
