﻿using System;
using ChaKi.Entity.Corpora;
using NHibernate;
using System.Collections;

namespace ChaKi.Service.DependencyEdit
{
    enum DOType
    {
        ChangeLinkEnd,
        Split,
        Merge,
        ChangeLinkTag,
        ChangeLexeme,
    }

    class DepOperation
    {
        public DepOperation(DOType t, object[] paramlist)
        {
            m_OpType = t;
            m_ParamList = paramlist;
        }

        private DOType m_OpType;
        private object[] m_ParamList;

        public void Execute(Sentence sen, ISession session)
        {
            switch (m_OpType)
            {
                case DOType.ChangeLinkEnd:
                    {
                        Link link = (Link)m_ParamList[0];
                        Segment seg = (Segment)m_ParamList[2];
                        link.To = seg;
                    }
                    break;
                case DOType.Split:
                    {
                        int bpos = (int)m_ParamList[0];
                        int epos = (int)m_ParamList[1];
                        int spos = (int)m_ParamList[2];
                        ExecuteSplitOperation(bpos, epos, spos, session);
                    }
                    break;
                case DOType.Merge:
                    {
                        int bpos = (int)m_ParamList[0];
                        int epos = (int)m_ParamList[1];
                        ExecuteMergeOperation(bpos, epos, session);
                    }
                    break;
                case DOType.ChangeLinkTag:
                    {
                        Link link = (Link)m_ParamList[0];
                        string newtag = (string)m_ParamList[2];
                        link.Text = newtag;
                    }
                    break;
                case DOType.ChangeLexeme:
                    {
                        int wpos = (int)m_ParamList[0];
                        Lexeme newlex = (Lexeme)m_ParamList[1];
                        Word w = (Word)sen.Words[wpos];
                        m_ParamList[2] = w.Lex;  // oldlex
                        w.Lex = newlex;
                    }
                    break;
            }
        }

        public void UnExecute(Sentence sen, ISession session)
        {
            switch (m_OpType)
            {
                case DOType.ChangeLinkEnd:
                    {
                        Link link = (Link)m_ParamList[0];
                        Segment seg = (Segment)m_ParamList[1];
                        link.To = seg;
                    }
                    break;
                case DOType.Split:
                    {
                        int bpos = (int)m_ParamList[0];
                        int epos = (int)m_ParamList[1];
                        ExecuteMergeOperation(bpos, epos, session);
                    }
                    break;
                case DOType.Merge:
                    {
                        int bpos = (int)m_ParamList[0];
                        int epos = (int)m_ParamList[1];
                        int spos = (int)m_ParamList[2];
                        ExecuteSplitOperation(bpos, epos, spos, session);
                    }
                    break;
                case DOType.ChangeLinkTag:
                    {
                        Link link = (Link)m_ParamList[0];
                        string oldtag = (string)m_ParamList[1];
                        link.Text = oldtag;
                    }
                    break;
                case DOType.ChangeLexeme:
                    {
                        int wpos = (int)m_ParamList[0];
                        Lexeme oldlex = (Lexeme)m_ParamList[2];
                        Word w = (Word)sen.Words[wpos];
                        w.Lex = oldlex;
                    }
                    break;
            }
        }

        private void ExecuteSplitOperation(int bpos, int epos, int spos, ISession session)
        {
            // Split範囲にあるSegmentを取得
            Segment left = FindSegmentBetween(bpos, epos, session);
            if (left == null)
            {
                throw new InvalidOperationException("Splitting: Missing segment");
            }
            // leftセグメントの終端
            left.EndChar = spos;
            // 新しいセグメント(right)
            Segment right = new Segment();
            right.StartChar = spos;
            right.EndChar = epos;
            right.Text = "Bunsetsu";
            // rightをDBに保存
            session.Save(right);
            // leftから出ているLinkをrightから出るように変更
            Link oldlink = FindLinkFrom(left, session);
            if (oldlink == null)
            {
                throw new InvalidOperationException("Splitting: Missing link");
            }
            oldlink.From = right;
            // left（既存）からright（新規）SegmentへLinkを張る
            Link newlink = new Link();
            newlink.From = left;
            newlink.To = right;
            newlink.Text = "D";
            session.Save(newlink);
        }

        private void ExecuteMergeOperation(int bpos, int epos, ISession session)
        {
            Segment bseg = FindSegmentStartingAt(bpos, session);
            if (bseg == null)
            {
                throw new InvalidOperationException("Merging: Missing first segment");
            }
            Segment eseg = FindSegmentEndingAt(epos, session);
            if (eseg == null)
            {
                throw new InvalidOperationException("Merging: Missing second segment");
            }

            // bsegから出るLink
            Link blink = FindLinkFrom(bseg, session);
            if (blink == null)
            {
                throw new InvalidOperationException("Merging: Missing link");
            }
            // esegから出るLink
            Link elink = FindLinkFrom(eseg, session);
            if (elink == null)
            {
                throw new InvalidOperationException("Merging: Missing link");
            }
            // esegに入るLink
            IList links = FindLinksTo(eseg, session);

            // leftセグメントを右に拡張することでマージする
            bseg.EndChar = epos;
            // 削除された文節をDBから切り離し、Transientなオブジェクトにする。
            session.Delete(eseg);
            // esegから出るリンクをDBから切り離す
            session.Delete(elink);
            // bsegから出るリンクの係り先をelinkの係り先に変更
            blink.To = elink.To;
            // esegに入っていたリンクの係り先をすべてbsegに変更
            foreach (object o in links)
            {
                Link lk = o as Link;
                if (lk.From != bseg)
                {
                    lk.To = bseg;
                }
            }
        }

        private Segment FindSegmentBetween(int bpos, int epos, ISession session)
        {
            IQuery query = session.CreateQuery(
                string.Format("from Segment seg where seg.StartChar = {0} and seg.EndChar = {1} and seg.Text = 'Bunsetsu'", bpos, epos));
            IList result = query.List();
            if (result.Count != 1)
            {
                return null;
            }
            return (Segment)result[0];
        }

        private Segment FindSegmentStartingAt(int pos, ISession session)
        {
            // 文の開始終了位置に、長さ0のDummy Segmentが存在する。Segmentを同定するクエリでこれをヒットしないように注意。
            IQuery query = session.CreateQuery(
                string.Format("from Segment seg where seg.StartChar = {0} and seg.EndChar > {0} and seg.Text = 'Bunsetsu'", pos));
            IList result = query.List();
            if (result.Count != 1)
            {
                return null;
            }
            return (Segment)result[0];
        }

        private Segment FindSegmentEndingAt(int pos, ISession session)
        {
            IQuery query = session.CreateQuery(
                string.Format("from Segment seg where seg.EndChar = {0} and seg.StartChar < {0} and seg.Text = 'Bunsetsu'", pos));
            IList result = query.List();
            if (result.Count != 1)
            {
                return null;
            }
            return (Segment)result[0];
        }

        private Link FindLinkFrom(Segment seg, ISession session)
        {
            IQuery query = session.CreateQuery(string.Format("select lk from Link lk, Segment seg where seg.ID={0} and lk.From=seg", seg.ID));
            IList result = query.List();
            if (result.Count != 1)
            {
                return null;
            }
            return (Link)result[0];
        }

        private IList FindLinksTo(Segment seg, ISession session)
        {
            IQuery query = session.CreateQuery(string.Format("select lk from Link lk, Segment seg where seg.ID={0} and lk.To=seg", seg.ID));
            return query.List();
        }

        public override string ToString()
        {
            string s = "{" + this.m_OpType + ":";
            foreach (object o in m_ParamList)
            {
                s += o;
                s += ", ";
            }
            s += "}";
            return s;
        }
    }
}
