﻿using System.Text;
using System.Collections;
using ChaKi.Entity.Corpora;
using ChaKi.Entity.Search;
using NHibernate;
using NHibernate.Criterion;
using System.Diagnostics;
using System;

namespace ChaKi.Service.Search
{
    /// <summary>
    /// 検索条件EntityよりHQLクエリ文字列を作成するためのユーティリティークラス
    /// </summary>
    internal class QueryBuilder
    {

        public static string BuildLexemeQuery(LexemeCondition lexcond)
        {
            StringBuilder sb = new StringBuilder();
            StringConnector conn = new StringConnector(" and ");

            foreach (PropertyPair item in lexcond.PropertyPairs)
            {
                string op = (item.Value.IsRegEx)? " regexp ": "=";

                if (item.Value.Target is PartOfSpeech)
                {
                    sb.Append(conn.Get());
                    sb.AppendFormat("l.PartOfSpeech.Name{0}'{1}'", op, item.Value.StrVal);
                }
                else if (item.Value.Target is CType)
                {
                    sb.Append(conn.Get());
                    sb.AppendFormat("l.CType.Name{0}'{1}'", op, item.Value.StrVal);
                }
                else if (item.Value.Target is CForm)
                {
                    sb.Append(conn.Get());
                    sb.AppendFormat("l.CForm.Name{0}'{1}'", op, item.Value.StrVal);
                }
                else
                {
                    sb.Append(conn.Get());
                    sb.AppendFormat("l.{0}{1}'{2}'", item.Key, op, item.Value.StrVal);
                }
            }
            if (sb.Length > 0)
            {
                return "from Lexeme l where " + sb.ToString();
            }
            return "from Lexeme";
        }
        
        /// <summary>
        /// 与えられたLexemeのリストからLexeme IDを抜き出して
        /// INクエリ条件の右辺："(id1,id2,..)" 形式に変換する。
        /// </summary>
        /// <param name="lexemeList"></param>
        /// <returns></returns>
        public static string BuildLexemeIDList(IList lexemeList)
        {
            StringBuilder sb = new StringBuilder("(");
            StringConnector connector = new StringConnector(",");
            foreach (Lexeme lex in lexemeList)
            {
                sb.Append(connector.Get());
                sb.Append(lex.ID);
            }
            sb.Append(")");
            return sb.ToString();
        }

        /// <summary>
        /// TagSearch用のクエリ文字列構築
        /// </summary>
        /// <param name="lexemeResultSet"></param>
        /// <param name="tagCond"></param>
        /// <returns></returns>
        public static string BuildTagSearchQuery(LexemeResultSet lexemeResultSet, SearchConditions cond)
        {
            Debug.Assert(lexemeResultSet != null);
            Debug.Assert(lexemeResultSet.Count > 0);

            TagSearchCondition tagCond = cond.TagCond;

            StringBuilder sb = new StringBuilder();
            int iPivot = tagCond.GetPivotPos();
            if (iPivot < 0)
            {
                throw new Exception("Pivot not found.");
            }
            sb.AppendFormat("select w{0} ", iPivot);
            sb.Append("from ");

            StringConnector connector = new StringConnector(",");
            foreach (LexemeResult res in lexemeResultSet)
            {
                sb.Append(connector.Get());
                sb.AppendFormat("Word w{0}", res.No);
            }

            sb.Append(" where ");
            sb.Append(QueryBuilder.BuildTagSearchQueryWhereClause(iPivot, tagCond, lexemeResultSet));

            return sb.ToString();
        }

        /// <summary>
        /// TagSearchクエリのwhere節を作成する関数
        /// </summary>
        /// <param name="iPivot"></param>
        /// <param name="tagCond"></param>
        /// <param name="lexResults"></param>
        /// <returns></returns>
        private static string BuildTagSearchQueryWhereClause(int iPivot, TagSearchCondition tagCond, LexemeResultSet lexResults)
        {
            // 検索の効率のため、lexResultsをLexemeのヒット数の少ない順に並べ替える。
            lexResults.Sort();

            StringBuilder sb = new StringBuilder();
            StringConnector connector = new StringConnector(" and ");
            foreach (LexemeResult result in lexResults)
            {
                // Pivotと同じ文に出現していて、相対位置が指定範囲内であること。
                if (!result.Cond.IsPivot)
                {
                    sb.Append(connector.Get());
                    sb.AppendFormat("w{0}.Sen.ID = w{1}.Sen.ID", result.No, iPivot);
                    Range range = result.Cond.RelativePosition;
                    if (range.Start == range.End)
                    {
                        sb.Append(connector.Get());
                        sb.AppendFormat("w{0}.Pos = w{1}.Pos + ({2})", result.No, iPivot, range.Start);
                    }
                    else if (range.Start < range.End)
                    {
                        sb.Append(connector.Get());
                        sb.AppendFormat("w{0}.Pos >= w{1}.Pos + ({2})", result.No, iPivot, range.Start);
                        sb.Append(connector.Get());
                        sb.AppendFormat("w{0}.Pos <= w{1}.Pos + ({2})", result.No, iPivot, range.End);
                    }
                }
                sb.Append(connector.Get());
                // かつ、候補LexemeのどれかにIDが一致していること。
                sb.AppendFormat("w{0}.Lex.ID in {1}", result.No, QueryBuilder.BuildLexemeIDList(result.LexemeList));
            }
            return sb.ToString();
        }

        /// <summary>
        /// WordList用のクエリ文字列構築
        /// </summary>
        /// <param name="lexemeResultSet"></param>
        /// <param name="tagCond"></param>
        /// <returns></returns>
        public static string BuildWordListQuery(LexemeResultSet lexemeResultSet, SearchConditions cond)
        {
            Debug.Assert(lexemeResultSet != null);
            Debug.Assert(lexemeResultSet.Count > 0);

            TagSearchCondition tagCond = cond.TagCond;

            StringBuilder sb = new StringBuilder();
            int iPivot = tagCond.GetPivotPos();
            if (iPivot < 0)
            {
                throw new Exception("Pivot not found.");
            }
            sb.AppendFormat("select w{0}.Lex, count(w{0}) ", iPivot);
            sb.Append("from ");

            StringConnector connector = new StringConnector(",");
            foreach (LexemeResult res in lexemeResultSet)
            {
                sb.Append(connector.Get());
                sb.AppendFormat("Word w{0}", res.No);
            }

            sb.Append(" where ");
            sb.Append(QueryBuilder.BuildTagSearchQueryWhereClause(iPivot, tagCond, lexemeResultSet));

            sb.AppendFormat(" group by w{0}.Lex.ID", iPivot);

            return sb.ToString();
        }

        /// <summary>
        /// StringSearch用のクエリ文字列構築
        /// </summary>
        /// <param name="lexemeResultSet"></param>
        /// <param name="tagCond"></param>
        /// <returns></returns>
        public static string BuildStringSearchQuery(SearchConditions cond)
        {
            StringSearchCondition strCond = cond.StringCond;
            if (strCond.Pattern.Length == 0)
            {
                return "from Sentence";
            }
            string pattern = string.Format("^.*{0}.*$", strCond.Pattern);
            return string.Format("from Sentence s where s.Text regexp '{0}'", pattern);
        }

        public static string BuildSentenceListQuery(SearchConditions cond)
        {
            return "from Sentence";
        }

        public static string BuildSentenceContextQuery(int index, int contextCount)
        {
            int minID = Math.Max(0, index - contextCount);
            int maxID = index + contextCount;

            return string.Format("from Sentence s where s.ID >= {0} and s.ID <= {1} order by s.ID", minID, maxID);
        }

        /// <summary>
        /// Dependency Search用のクエリを作成する
        /// </summary>
        /// <param name="lexemeResultSet"></param>
        /// <param name="cond"></param>
        /// <returns></returns>
        public static string BuildDepSearchQuery(LexemeResultSet lexemeResultSet, SearchConditions cond)
        {
            Debug.Assert(lexemeResultSet != null);
            Debug.Assert(lexemeResultSet.Count > 0);

            DepSearchCondition depCond = cond.DepCond;

            StringBuilder sb = new StringBuilder();
            int iPivot = depCond.GetPivotPos();
            if (iPivot < 0)
            {
                throw new Exception("Pivot not found.");
            }
            sb.AppendFormat("select w{0} ", iPivot);
            sb.Append("from ");

            StringConnector connector = new StringConnector(",");
            foreach (LexemeResult res in lexemeResultSet)
            {
                sb.Append(connector.Get());
                sb.AppendFormat("Word w{0}", res.No);
            }
            sb.Append(connector.Get());
            sb.AppendFormat("Sentence sen");
            for (int i = 0; i < depCond.BunsetsuConds.Count; i++)
            {
                sb.Append(connector.Get());
                sb.AppendFormat("Bunsetsu b{0}", i);
            }

            sb.Append(" where ");
            sb.Append(QueryBuilder.BuildDepSearchQueryWhereClause(iPivot, depCond, lexemeResultSet));

            return sb.ToString();
        }

        private static string BuildDepSearchQueryWhereClause(int iPivot, DepSearchCondition depCond, LexemeResultSet lexResults)
        {
            // 検索の効率のため、lexResultsをLexemeのヒット数の少ない順に並べ替える。
            lexResults.Sort();

            StringBuilder sb = new StringBuilder();
            StringConnector connector = new StringConnector(" and ");
            foreach (LexemeResult result in lexResults)
            {
                // 同じ文に出現していること。
                sb.Append(connector.Get());
                sb.AppendFormat("w{0}.Sen.ID = sen.ID", result.No);
                // かつ、候補LexemeのどれかにIDが一致していること。
                sb.Append(connector.Get());
                sb.AppendFormat("w{0}.Lex.ID in {1}", result.No, QueryBuilder.BuildLexemeIDList(result.LexemeList));
                // かつ、その語がBunsetsu(i)に属していること
                sb.Append(connector.Get());
                sb.AppendFormat("w{0}.Bunsetsu.ID = b{1}.ID", result.No, result.BunsetsuNo);
                // 語の間の順序関係
                if (result.Cond.LeftConnection == '-')
                {
                    sb.Append(connector.Get());
                    sb.AppendFormat("w{0}.Pos+1 = w{1}.Pos", result.No-1, result.No);
                }
                else if (result.Cond.LeftConnection == '<')
                {
                    sb.Append(connector.Get());
                    sb.AppendFormat("w{0}.Pos < w{1}.Pos", result.No-1, result.No);
                }
                else if (result.Cond.LeftConnection == '^')
                {
                    sb.Append(connector.Get());
                    sb.AppendFormat("w{0}.Pos = 0", result.No);
                }
                else if (result.Cond.RightConnection == '$')
                {
                    throw new NotImplementedException("'$' is not implemented yet");
//                    sb.Append(connector.Get());
//                    sb.AppendFormat("w{0}.EndChar = s{1}.EndChar", result.No, result.BunsetsuNo);
                }
            }
            // Segment間の順序関係
            for (int i = 0; i < depCond.BunsetsuConds.Count; i++) {
                TagSearchCondition tcond = depCond.BunsetsuConds[i];
                if (tcond.LeftConnection == '^')
                {
                    sb.Append(connector.Get());
                    sb.AppendFormat("s{0}.Pos=0", i);
                }
                else if (tcond.LeftConnection == '-')
                {
                    sb.Append(connector.Get());
                    sb.AppendFormat("s{0}.Pos = s{1}.Pos-1", i-1, i);
                }
                else if (tcond.LeftConnection == '<')
                {
                    sb.Append(connector.Get());
                    sb.AppendFormat("s{0}.Pos < s{1}.Pos", i-1, i);
                }
                else if (tcond.RightConnection == '$')
                {
                    throw new NotImplementedException("'$' is not implemented yet");
//                    sb.Append(connector.Get());
//                    sb.AppendFormat("s{0}.EndChar = sen.EndChar", i);
                }
            }
            // Segment間のLink条件
            for (int i = 0; i < depCond.LinkConds.Count; i++)
            {
                /*
                LinkCondition kcond = depCond.LinkConds[i];
                sb.Append(connector.Get());
                sb.AppendFormat("k{0}.From = s{1} and k{0}.To = s{2}", i, kcond.SegidFrom, kcond.SegidTo);
                if (kcond.Text.Length > 0)
                {
                    sb.Append(connector.Get());
                    sb.AppendFormat("k{0}.Text = '{1}'", i, kcond.Text);
                }
                 */
            }
            return sb.ToString();
        }
    }
}
