﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using ChaKi.Common.Settings;
using ChaKi.Entity.Corpora.Annotations;
using PopupControl;
using ChaKi.Common.Properties;

namespace ChaKi.Common
{
    /// <summary>
    /// DependencyEdit PanelでAnnotation Tagを選択するためのポップアップウィンドウとして
    /// 使用されるユーザコントロール.
    /// Segment, Link, Groupの3種のTagについてあらかじめ準備されたインスタンスを提供するので、
    /// newする必要はない.
    /// </summary>
    public partial class TagSelector : UserControl
    {
        public string TagType { get; set; }
        public Tag Selection { get; private set; }
        private List<Tag> m_Source;
        public event EventHandler TagSelected;
        private View m_ListViewMode;

        public static Dictionary<string, TagSelector> PreparedSelectors = new Dictionary<string, TagSelector>();
        public static Dictionary<string, Popup> PreparedPopups = new Dictionary<string, Popup>();

        static TagSelector()
        {
            string type;
            TagSelector ts;
            type = ChaKi.Entity.Corpora.Annotations.Tag.SEGMENT;
            ts = new TagSelector() { TagType = type };
            PreparedSelectors.Add(type, ts);
            PreparedPopups.Add(type, new Popup(ts) { Resizable = true });
            type = ChaKi.Entity.Corpora.Annotations.Tag.LINK;
            ts = new TagSelector() { TagType = type };
            PreparedSelectors.Add(type, ts);
            PreparedPopups.Add(type, new Popup(ts) { Resizable = true });
            type = ChaKi.Entity.Corpora.Annotations.Tag.GROUP;
            ts = new TagSelector() { TagType = type };
            PreparedSelectors.Add(type, ts);
            PreparedPopups.Add(type, new Popup(ts) { Resizable = true });
        }

        public static void ClearAllTags()
        {
            PreparedSelectors[ChaKi.Entity.Corpora.Annotations.Tag.SEGMENT].Reset();
            PreparedSelectors[ChaKi.Entity.Corpora.Annotations.Tag.LINK].Reset();
            PreparedSelectors[ChaKi.Entity.Corpora.Annotations.Tag.GROUP].Reset();
        }

        public TagSelector()
        {
            InitializeComponent();
            this.ResizeRedraw = true;

            listView1.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
                new ColumnHeader() { Name = "name", Text = "Name", Width=100 },
                new ColumnHeader() { Name = "description", Text="Description", Width=200}
            });
            listView1.FullRowSelect = true;
            this.Selection = null;
            m_Source = new List<Tag>();
        }

        public void Reset()
        {
            this.listView1.Items.Clear();
            if (this.listView1.LargeImageList == null)
            {
                this.listView1.LargeImageList = new ImageList()
                {
                    ImageSize = new Size(32, 32),
                    ColorDepth = ColorDepth.Depth32Bit
                };
                this.listView1.SmallImageList = new ImageList()
                {
                    ImageSize = new Size(16, 16),
                    ColorDepth = ColorDepth.Depth32Bit
                };
            }
            else
            {
                this.listView1.LargeImageList.Images.Clear();
            }
            m_Source.Clear();
        }

        public View DisplayMode
        {
            set
            {
                m_ListViewMode = value;
                this.toolStripButton1.Checked = (m_ListViewMode == View.LargeIcon);
                this.toolStripButton2.Checked = (m_ListViewMode == View.List);
                this.toolStripButton3.Checked = (m_ListViewMode == View.Details);
                this.listView1.View = m_ListViewMode;
            }
        }

        public void AddTag(Tag t)
        {
            Image img = Resources.Gear.ToBitmap();
            int i = listView1.LargeImageList.Images.Count;
            listView1.LargeImageList.Images.Add(img);
            listView1.SmallImageList.Images.Add(img);
            ListViewItem item = new ListViewItem(t.Name, i);
            item.SubItems.Add(t.Description);
            listView1.Items.Add(item);
            m_Source.Add(t);
        }

        private void listView1_ItemActivate(object sender, EventArgs e)
        {
            DetermineResult();
        }

        private void listView1_Click(object sender, EventArgs e)
        {
            DetermineResult();
        }

        private void DetermineResult()
        {
            int index = (this.listView1.SelectedIndices.Count > 0) ? this.listView1.SelectedIndices[0] : -1;
            if (m_Source != null && index >= 0 && index < m_Source.Count)
            {
                this.Selection = m_Source[index];
                if (TagSelected != null)
                {
                    TagSelected(this, null);
                }
            }
            Popup p = this.Parent as Popup;
            if (p != null)
            {
                p.Close(ToolStripDropDownCloseReason.ItemClicked);
            }
        }

        // PopupでResizableを有効にした場合は必ず必要
        protected override void WndProc(ref Message m)
        {
            Popup p = Parent as Popup;
            if (p != null && p.ProcessResizing(ref m))
            {
                return;
            }
            base.WndProc(ref m);
        }

        // 表示時に親(Popup)のサイズを保存値に設定.
        private void TagSelector_Load(object sender, EventArgs e)
        {
            Popup p = Parent as Popup;
            if (p != null)
            {
                TagSelectorSettingItem setting = TagSelectorSettings.Current.GetByType(this.TagType);
                p.Width = setting.Size.Width;
                p.Height = setting.Size.Height;
                this.DisplayMode = (View)setting.Style;
            }
            this.SizeChanged += new EventHandler(TagSelector_SizeChanged);
        }

        private void TagSelector_SizeChanged(object sender, EventArgs e)
        {
            Popup p = Parent as Popup;
            if (p != null)
            {
                TagSelectorSettingItem setting = TagSelectorSettings.Current.GetByType(this.TagType);
                setting.Size = new Size(p.Width, p.Height);
            }
        }

        private void toolStripButton1_Click(object sender, EventArgs e)
        {
            DisplayMode = View.LargeIcon;
            TagSelectorSettingItem setting = TagSelectorSettings.Current.GetByType(this.TagType);
            setting.Style = (int)this.listView1.View;
        }

        private void toolStripButton2_Click(object sender, EventArgs e)
        {
            DisplayMode = View.List;
            TagSelectorSettingItem setting = TagSelectorSettings.Current.GetByType(this.TagType);
            setting.Style = (int)this.listView1.View;
        }

        private void toolStripButton3_Click(object sender, EventArgs e)
        {
            DisplayMode = View.Details;
            TagSelectorSettingItem setting = TagSelectorSettings.Current.GetByType(this.TagType);
            setting.Style = (int)this.listView1.View;
        }
    }
}
