﻿using System;
using System.Collections.Generic;
using System.Text;
using ChaKi.Entity.Kwic;
using ChaKi.Entity.Corpora;
using ChaKi.Entity.Search;
using ChaKi.Entity.Settings;
using ChaKi.Service.Search;
using ChaKi.Common;
using ChaKi.Service.Database;

namespace ChaKi
{
    public class ChaKiModel
    {
        /// <summary>
        /// CurrentCorpusが変化したときの通知イベント
        /// </summary>
        public static event CurrentChangedDelegate OnCurrentChanged;
        
        static ChaKiModel()
        {
            CurrentCorpus = null;
        }

        public ChaKiModel()
        {
            History = SearchHistory.Root;
            CurrentSearchConditions = new SearchConditions();
        }

        /// <summary>
        /// システムで唯一保持される、カレントコーパスの参照
        /// この値はさまざまな場合に変更される
        /// 変更要因：
        /// (1) Search Conditions/Corpusタブでユーザがコーパスのリストボックスの選択状態を変更したとき
        /// (2) 検索実行中に、検索対象のコーパスが変化したとき
        /// (3) KwicViewのカレント選択行が変更されたとき（選択行の元コーパスに変更される）
        /// (4) KwicViewでマウスHoverイベントが発生したとき（Hover行の元コーパスに変更される）
        /// 参照するクラス：
        /// (a) このクラス自身が、CurrentCorpusの設定により非表示属性を決定する
        /// (b) PropertyBoxが、CurrentCorpusの設定により非表示属性を決定する
        /// (c) GuidePanelが、表示更新前にCurrentCorpusの設定により非表示属性を決定する
        /// なお、CurrentCorpus == nullの場合は、デフォルトの属性がすべて表示される。
        /// </summary>
        public static Corpus CurrentCorpus
        {
            get
            {
                return m_currentCorpus;
            }
            set
            {
                m_currentCorpus = value;
                if (OnCurrentChanged != null)
                {
                    OnCurrentChanged(m_currentCorpus, -1);
                }
            }
        }

        private static Corpus m_currentCorpus;

        public SearchHistory History { get; private set; }
        public SearchConditions CurrentSearchConditions { get; set; }

        public void Reset()
        {
            History.Reset();
            CurrentSearchConditions.Reset();
        }

        public void DeleteHistory(SearchHistory hist)
        {
            History.Delete(hist);
        }

        public void Initialize()
        {
            Reset();

            ChaKiModel.CurrentCorpus = null;
            foreach (Corpus c in UserSettings.GetInstance().LastCorpus)
            {
                if (ChaKiModel.CurrentCorpus == null)
                {
                    try
                    {
                        DBService dbs = DBService.Create(c.DBParam);
                        dbs.LoadMandatoryCorpusInfo(c);
                        ChaKiModel.CurrentCorpus = c;
                    }
                    catch (Exception ex)
                    {
                        ErrorReportDialog err = new ErrorReportDialog("Cannot load default corpus", ex);
                        err.ShowDialog();
                        continue;
                    }
                }
                CurrentSearchConditions.CorpusCond.Corpora.Add(c);
            }
        }
    }
}
