﻿using System;
using System.Collections.Generic;
using System.Data.Common;
using ChaKi.Entity.Corpora;
using ChaKi.Entity.Corpora.Annotations;
using ChaKi.Service.Database;
using ChaKi.Service.Readers;
using Iesi.Collections.Generic;
using MySql.Data.MySqlClient;

namespace CreateCorpusSLA
{
    public class CreateCorpus
    {
        public string path = null;
        public string corpusName = null;
        public string textEncoding = "SHIFT_JIS";
        public string textType = "auto";

        private DBService m_Service = null;
        private Corpus m_Corpus = null;
        private string m_DefaultString;

        private DocumentSet m_DocumentSet;
        private Project m_Project;

        public void InitializeDocumentSet()
        {
            m_DocumentSet = new DocumentSet();

            // Create Default Project for the DocumentSet
            m_Project = new Project();
            m_DocumentSet.AddProject(m_Project);
            
            // Create Default TagSet for the Project
            m_Project.TagSetList = new List<TagSet>();
        }

        public bool ParseArguments(string[] args)
        {
            int n = 0;
            foreach (string arg in args)
            {
                if (arg.Length > 1 && arg.StartsWith("-"))
                {
                    string[] tokens = arg.Substring(1).Split(new char[] { '=' });
                    if (tokens.Length < 2 || (!tokens[0].Equals("e") && !tokens[0].Equals("t")))
                    {
                        Console.WriteLine("Invalid option: {0}", arg);
                        return false;
                    }
                    if (tokens[0].Equals("e"))
                    {
                        this.textEncoding = tokens[1];
                    }
                    else if (tokens[0].Equals("t"))
                    {
                        string token = tokens[1].ToLower();
                        string[] subtypes = token.Split(new char[] { '|' });
                        if (subtypes.Length > 0)
                        {
                            this.textType = subtypes[0];
                        }
                        else
                        {
                            this.textType = token;
                        }
                    }
                }
                else
                {
                    if (n == 0)
                    {
                        this.path = arg;
                    }
                    else if (n == 1)
                    {
                        this.corpusName = arg;
                    }
                    n++;
                }
            }
            if (n < 2)
            {
                return false;
            }
            return true;
        }

        public void ParseInput()
        {
            // Corpusの作成
            m_Corpus = Corpus.CreateFromFile(corpusName);
            if (m_Corpus == null)
            {
                Console.WriteLine("Error: Cannot create Corpus. Maybe <out file>'s extension is neither .db nor .def");
                return;
            }

            // DB初期化のためのサービスを得る
            m_Service = DBService.Create(m_Corpus.DBParam);
            m_DefaultString = m_Service.GetDefault(); // INSERT文のDEFAULT文字列
            try
            {
                m_Service.DropDatabase();
                m_Service.CreateDatabase();
            }
            catch (Exception ex)
            {
                Console.Write("Info: ");
                Console.WriteLine(ex.Message);
                // Continue.
            }
            try
            {
                m_Service.DropAllTables();
            }
            catch (Exception ex)
            {
                PrintException(ex);
                // continue
            }
            try
            {
                m_Service.CreateAllTables();
                m_Service.CreateAllIndices();
            }
            catch (Exception ex)
            {
                PrintException(ex);
                // maybe some trouble in creating indices; continue anyway...
            }

            // CaboChaファイル読込み
            Console.WriteLine("Reading {0}", path);
            try
            {
                CorpusSourceReader reader = null;
                if (textType.Equals("auto"))
                {
                    reader = CorpusSourceReaderFactory.Create(path, textEncoding, m_Corpus);
                    Console.WriteLine("Using {0}", reader.GetType().Name);
                }
                else if (textType.Equals("chasen"))
                {
                    reader = new CabochaChasenReader(m_Corpus);
                }
                else if (textType.Equals("mecab"))
                {
                    reader = new CabochaMecabReader(m_Corpus);
                }
                else
                {
                    Console.WriteLine("Invalid Reader Type: {0}", textType);
                    return;
                }
                Document newdoc = reader.ReadFromFileSLA(path, textEncoding);
                m_DocumentSet.AddDocument(newdoc);

                // Readerに定義されるTagSetをProjectに結びつける
                TagSet ts = reader.GetTagSet();
                if (ts != null)
                {
                    m_Project.AddTagSet(ts);
                }

            }
            catch (Exception ex)
            {
                PrintException(ex);
                return;
            }
        }

        public void SaveProject()
        {
            Console.Write("Saving Project...");
            try
            {
                using (DbConnection conn = m_Service.GetDbConnnection())
                {
                    conn.Open();
                    DbCommand cmd = conn.CreateCommand();
                    DbTransaction trans = conn.BeginTransaction();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    cmd.CommandText = string.Format("INSERT INTO projects VALUES({0},{1},{2},{3})",
                        m_Project.ID, m_DocumentSet.ID, 0, 0);
                    cmd.ExecuteNonQuery();

                    TagSetVersion ver = m_Project.TagSetList[0].Version;
                    cmd.CommandText = string.Format("INSERT INTO tagset_versions VALUES({0},'{1}',{2},{3})",
                        ver.ID, ver.Version, ver.Revision, m_Service.GetBooleanString(ver.IsCurrent));
                    cmd.ExecuteNonQuery();

                    int n = 0;
                    foreach (TagSet ts in m_Project.TagSetList)
                    {
                        ts.ID = n;
                        cmd.CommandText = string.Format("INSERT INTO tagset_definitions VALUES({0},'{1}',{2})",
                            ts.ID, ts.Name, ts.Version.ID);
                        cmd.ExecuteNonQuery();

                        int m = 0;
                        foreach (Tag t in ts.Tags)
                        {
                            t.ID = m;
                            cmd.CommandText = string.Format("INSERT INTO tag_definition VALUES({0},{1},'{2}','{3}','{4}',{5})",
                                t.ID, ts.ID, t.Type, t.Name, t.Description, t.Version.ID);
                            cmd.ExecuteNonQuery();
                            m++;
                        }

                        cmd.CommandText = string.Format("INSERT INTO projects_tagsets VALUES({0},{1})",
                            m_Project.ID, ts.ID);
                        cmd.ExecuteNonQuery();
                        n++;
                    }

                    cmd.CommandText = string.Format("INSERT INTO document_sets VALUES({0},'{1}')",
                        m_DocumentSet.ID, m_DocumentSet.Name);
                    cmd.ExecuteNonQuery();
                    n = 0;
                    foreach (Document doc in m_DocumentSet.Documents)
                    {
                        cmd.CommandText = string.Format("INSERT INTO documents VALUES({0},{1},{2},'{3}','{4}')",
                            doc.ID, m_DocumentSet.ID, doc.Order, doc.FileName, doc.Text.Replace("'", "''"));
                        cmd.ExecuteNonQuery();
                        n++;
                    }

                    trans.Commit();
                    cmd.Dispose();
                }
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
            Console.WriteLine("written.");
        }

        public void SaveLexicon()
        {
            // Native DB Connection
            Console.WriteLine("\n\nSaving to database {0}", corpusName);
            try
            {
                using (DbConnection conn = m_Service.GetDbConnnection())
                {
                    int n;
                    conn.Open();
                    DbCommand cmd = conn.CreateCommand();
                    DbTransaction trans = conn.BeginTransaction();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    Console.WriteLine("Saving Lexicon...");
                    Console.WriteLine("Saving PartsOfSpeech...");
                    n = 0;
                    foreach (PartOfSpeech pos in m_Corpus.Lex.PartsOfSpeech)
                    {
                        pos.ID = n++;
                        cmd.CommandText = string.Format("INSERT INTO parts_of_speech VALUES({0},'{1}','{2}','{3}','{4}','{5}')",
                            pos.ID, pos.Name1, pos.Name2, pos.Name3, pos.Name4, pos.Name);
                        cmd.ExecuteNonQuery();
                        Console.Write("> {0}\r", pos.ID+1);
                    }
                    Console.WriteLine("\x0a Saving CTypes...");
                    n = 0;
                    foreach (CType ctype in m_Corpus.Lex.CTypes)
                    {
                        ctype.ID = n++;
                        cmd.CommandText = string.Format("INSERT INTO ctypes VALUES({0},'{1}','{2}','{3}')",
                            ctype.ID, ctype.Name1, ctype.Name2, ctype.Name);
                        cmd.ExecuteNonQuery();
                        Console.Write("> {0}\r", ctype.ID+1);
                    }
                    Console.WriteLine("\x0a Saving CForms...");
                    n = 0;
                    foreach (CForm cform in m_Corpus.Lex.CForms)
                    {
                        cform.ID = n++;
                        cmd.CommandText = string.Format("INSERT INTO cforms VALUES({0},'{1}')",
                            cform.ID, cform.Name);
                        cmd.ExecuteNonQuery();
                        Console.Write("> {0}\r", cform.ID+1);
                    }
                    Console.WriteLine("\x0a Saving Lexemes...");
                    n = 0;
                    foreach (Lexeme lex in m_Corpus.Lex.Entries)
                    {
                        lex.ID = n++;   // Lexiconは後方参照もあるので、先に全部にIDを振っておく
                    }
                    foreach (Lexeme lex in m_Corpus.Lex.Entries)
                    {
                        cmd.CommandText = string.Format("INSERT INTO lexemes VALUES ({0},'{1}','{2}','{3}',{4},{5},{6},{7},{8},{9})",
                            lex.ID,
                            lex.Surface.Replace("'", "''").Replace("\\", ""),
                            lex.Reading.Replace("'", "''"),
                            lex.Pronunciation.Replace("'", "''"),
                            lex.BaseLexeme.ID,
                            lex.PartOfSpeech.ID, lex.CType.ID, lex.CForm.ID, m_DefaultString, lex.Frequency);
                        cmd.ExecuteNonQuery();
                        Console.Write("> {0}\r", lex.ID+1);
                    }
                    trans.Commit();
                    cmd.Dispose();
                }
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
            Console.WriteLine("\x0aLexicon written.");
        }

        public void SaveDocumentTags()
        {
            Console.WriteLine("\nSaving DocumentTags ({0})...", m_Corpus.DocumentTags.Count);
            try
            {
                using (DbConnection conn = m_Service.GetDbConnnection())
                {
                    conn.Open();
                    DbCommand cmd = conn.CreateCommand();
                    DbTransaction trans = conn.BeginTransaction();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    foreach (DocumentTag tag in m_Corpus.DocumentTags)
                    {
                        cmd.CommandText = string.Format("INSERT INTO documenttags VALUES({0},'{1}','{2}',{3})",
                            tag.ID, tag.Tag, tag.Description.Replace("'", "''"), m_DefaultString);
                        try
                        {
                            cmd.ExecuteNonQuery();
                        }
                        catch
                        {
                            Console.WriteLine("\nInvalid data: query={0}", cmd.CommandText); ;
                        }
                        Console.Write("> {0}\r", tag.ID + 1);
                    }
                    trans.Commit();
                    cmd.Dispose();
                    Console.WriteLine();
                }
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        public void SaveSentenceTags()
        {
            Console.WriteLine("\nSaving SentenceTags ({0})...", m_Corpus.Sentences.Count);
            try
            {
                using (DbConnection conn = m_Service.GetDbConnnection())
                {
                    conn.Open();
                    DbCommand cmd = conn.CreateCommand();
                    DbTransaction trans = conn.BeginTransaction();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    foreach (Sentence sen in m_Corpus.Sentences)
                    {
                        ISet<DocumentTag> list = sen.DocumentTags;
                        foreach (DocumentTag tag in list)
                        {
                            cmd.CommandText = string.Format("INSERT INTO sentences_documenttags VALUES({0},{1})",
                                sen.ID, tag.ID);
                            cmd.ExecuteNonQuery();
                        }
                        Console.Write("> {0}\r", sen.ID + 1);
                    }
                    trans.Commit();
                    cmd.Dispose();
                    Console.WriteLine();
                }
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
        }

        public void SaveSentences()
        {
            // Sentenceをセーブする
            Console.WriteLine("\nSaving Sentences...");
            SimpleTransactionManager trans = null;
            DbCommand cmd = null;

            try
            {
                trans = new SimpleTransactionManager(m_Service);
                trans.Begin();
                cmd = trans.Cmd;

                int senid = 0;
                int wordid = 0;
                foreach (Sentence sen in m_Corpus.Sentences)
                {
                    sen.ID = senid++;
                    cmd.CommandText = string.Format("INSERT INTO sentences VALUES({0},{1},{2},{3})",
                            sen.ID, sen.StartChar, sen.EndChar, sen.ParentDoc.ID);
                    cmd.ExecuteNonQuery();
                    foreach (Word word in sen.Words)
                    {
                        word.ID = wordid++;
                        cmd.CommandText = string.Format("INSERT INTO words VALUES ({0},{1},{2},{3},{4},{5},{6},{7})",
                            word.ID, word.Sen.ID, word.StartChar, word.EndChar, word.Lex.ID,
                            0, m_DefaultString, word.Pos );  // word.Bunsetsu.ID=0
                        cmd.ExecuteNonQuery();
                    }
                    if (senid > 0 && senid % 500 == 0)
                    {
                        trans.CommitAndContinue();
                        cmd = trans.Cmd;
                        Console.Write("> {0} Committed.\r", sen.ID + 1);
                    }
                }
                Console.WriteLine("> {0} Committed.", m_Corpus.Sentences.Count);
                trans.CommitAndContinue();
                cmd = trans.Cmd;

                // Segment, Linkをセーブする
                int segid = 0;
                Console.WriteLine("\nSaving Segments...");
                foreach (Segment seg in m_Corpus.Segments)
                {
                    seg.ID = segid++;
                    cmd.CommandText = string.Format("INSERT INTO segments VALUES({0},{1},{2},{3},{4},{5},{6},{7},'{8}',{9})",
                       seg.ID, seg.Tag.ID, 0/*document_id*/, seg.StartChar, seg.EndChar,
                       m_Project.ID, 0/*user_id*/,
                       m_Service.GetDefault(), seg.Comment, seg.Sentence.ID);
                    cmd.ExecuteNonQuery();
                    if (segid > 0 && segid % 500 == 0)
                    {
                        trans.CommitAndContinue();
                        cmd = trans.Cmd;
                        Console.Write("> {0} Committed.\r", seg.ID + 1);
                    }
                }
                Console.WriteLine("> {0} Committed.", m_Corpus.Segments.Count);
                trans.CommitAndContinue();
                cmd = trans.Cmd;

                Console.WriteLine("\nSaving Links...");
                int linkid = 0;
                foreach (Link lnk in m_Corpus.Links)
                {
                    lnk.ID = linkid++;
                    cmd.CommandText = string.Format("INSERT INTO links VALUES({0},{1},{2},{3},{4},{5},{6},{7},{8},'{9}',{10},{11})",
                        lnk.ID, lnk.Tag.ID, lnk.From.ID, lnk.To.ID,
                        m_Service.GetBooleanString(lnk.IsDirected), m_Service.GetBooleanString(lnk.IsTransitive),
                        m_Project.ID, 0/*user_id*/, m_Service.GetDefault(), lnk.Comment,
                        lnk.FromSentence.ID, lnk.ToSentence.ID);
                    cmd.ExecuteNonQuery();
                    if (linkid > 0 && linkid % 500 == 0)
                    {
                        trans.CommitAndContinue();
                        cmd = trans.Cmd;
                        Console.Write("> {0} Committed.\r", lnk.ID + 1);
                    }
                }
                Console.WriteLine("> {0} Committed.", m_Corpus.Links.Count);
                trans.Commit();
                Console.Write("\n");
            }
            catch (Exception ex)
            {
                PrintException(ex);
            }
            finally
            {
                if (trans != null)
                {
                    trans.Dispose();
                }
            }
        }

#if false
            NHibernate.Cfg.Configuration cfg = new Configuration();
            cfg.SetProperties(new Dictionary<string, string>());
            svc.SetupConnection(cfg);
            cfg.AddAssembly("ChaKiEntity");
#endif

        public void UpdateIndex()
        {
            Console.WriteLine("\nUpdating Index...");
            try
            {
                m_Service.CreateFinalIndices();
            }
            catch (Exception ex)
            {
                PrintException(ex);
                // maybe some trouble in creating indices; continue anyway...
            }
            Console.WriteLine();
            Console.WriteLine("Done.");
        }

        public static void PrintException(Exception ex)
        {
            Console.WriteLine("Exception: {0}", ex.ToString());
        }
    }
}
