﻿using System.Collections.Generic;
using System.Drawing;
using Crownwood.DotNetMagic.Common;
using System;
using System.IO;
using System.Xml.Serialization;
using ChaKi.Common;
using ChaKi.Views.KwicView;

namespace ChaKi
{
    public class GUISetting : ICloneable
    {
        public static GUISetting Instance { get; private set; }

        public VisualStyle Visual { get; set; }
        public FontDictionary Fonts { get; set; }

        public KwicViewSettings KwicViewSettings { get; set; }

        public event EventHandler FontChanged;
        public event EventHandler VisualStyleChanged;

        static GUISetting()
        {
            Instance = new GUISetting();
        }

        private GUISetting()
        {
            this.Fonts = new FontDictionary();

            // Defaults
            this.Visual = VisualStyle.Office2007Silver;

            this.Fonts.Add("BaseText", new Font("ＭＳ 明朝", 11.0F));
            this.Fonts.Add("BaseAnsi", new Font("Lucida Sans Unicode", 7.0F));

            this.KwicViewSettings
               = new KwicViewSettings{ LineMargin = 5, WordMargin = 0, ShowSegments = false, ShowLinks = false, ShowGroups = false };
        }

        public void Set(GUISetting src)
        {
            Instance.Visual = src.Visual;
            if (this.VisualStyleChanged != null)
            {
                this.VisualStyleChanged(this, null);
            }
            this.Fonts = src.Fonts;
            if (this.FontChanged != null)
            {
                this.FontChanged(this, null);
            }
        }

        #region ICloneable メンバ
        public object Clone()
        {
            GUISetting obj = new GUISetting();
            obj.Visual = this.Visual;
            obj.Fonts = new FontDictionary();
            foreach (KeyValuePair<string, Font> pair in this.Fonts)
            {
                obj.Fonts.Add(string.Copy(pair.Key), (Font)pair.Value.Clone());
            }
            return obj;
        }
        #endregion

        public Font GetBaseTextFont()
        {
            return this.Fonts["BaseText"];
        }

        public Font GetBaseAnsiFont()
        {
            return this.Fonts["BaseAnsi"];
        }

        public void SetFont(string tag, Font font)
        {
            this.Fonts[tag] = font;
            if (this.FontChanged != null)
            {
                this.FontChanged(this, new EventArgs());
            }
        }

        /// <summary>
        /// 設定をXMLファイルから読み込む
        /// </summary>
        /// <param name="file"></param>
        public static void Load(string file)
        {
            using (StreamReader rd = new StreamReader(file))
            {
                XmlSerializer ser = new XmlSerializer(typeof(GUISetting));
                Instance = (GUISetting)ser.Deserialize(rd);
            }
            KwicViewPanel.Settings = Instance.KwicViewSettings;
        }

        /// <summary>
        /// 設定をXMLファイルへ書き出す
        /// </summary>
        /// <param name="file"></param>
        public static void Save(string file)
        {
            Instance.KwicViewSettings = KwicViewPanel.Settings;
            using (StreamWriter wr = new StreamWriter(file))
            {
                XmlSerializer ser = new XmlSerializer(typeof(GUISetting));
                ser.Serialize(wr, Instance);
            }
        }
    }
}
