﻿using System;
using System.Windows.Forms;
using ChaKi.Common;
using ChaKi.Entity.Kwic;
using ChaKi.Entity.Corpora;
using System.Drawing;
using System.Collections.Generic;

namespace ChaKi.Views
{

    public partial class WordListView : UserControl, IChaKiView
    {
        private List<Corpus> m_CorpusList;
        private LexemeCountList m_Model;
        private int m_FirstCountColIndex;   // corpusごとのカウント列の開始列index
        private int m_TotalColIndex;        // "All"列の列index
        private int m_RatioColIndex;        // "Ratio"列の列index
        private LexemeFilter m_LexFilter;


        public LexemeCountList Model
        {
            get { return m_Model; }
        }

        public void SetCorpusList(List<Corpus> corpusList)
        {
            m_CorpusList = corpusList;
            InitializeGrid();
        }

        public void SetModel(object model)
        {
            if (model != null && !(model is LexemeCountList))
            {
                throw new ArgumentException("Assigning invalid model to WordListView");
            }
            m_Model = (LexemeCountList)model;
            if (m_Model != null)
            {
                m_Model.OnLexemeCountAdded += new AddLexemeCountEventHandler(this.AddLexemeCountHandler);
            }
            UpdateView();
        }

        public void SetVisible(bool f)
        {
            this.Visible = f;
        }

        public WordListView()
        {
            m_CorpusList = new List<Corpus>();
            m_Model = null;
            m_LexFilter = new LexemeFilter();

            InitializeComponent();

            InitializeGrid();
        }

        public WordListView( List<Corpus> corpusList )
        {
            m_CorpusList = corpusList;
            m_Model = new LexemeCountList();
            m_Model.OnLexemeCountAdded += new AddLexemeCountEventHandler(this.AddLexemeCountHandler);
            m_LexFilter = new LexemeFilter();

            InitializeComponent();

            InitializeGrid();
        }

        protected void InitializeGrid()
        {
            this.dataGridView1.GridColor = Color.DarkGray;
            this.dataGridView1.RowHeadersWidth = 100;
            this.dataGridView1.RowTemplate.Height = 18;
            this.dataGridView1.InitializeGrid();

            // 基本属性の表示列を生成する
            this.dataGridView1.Columns.Clear();
            int ncol = 0;
            foreach (string tag in Lexeme.PropertyName.Values)
            {
                DataGridViewColumn col = new DataGridViewTextBoxColumn();
                col.Name = tag;
                col.SortMode = DataGridViewColumnSortMode.Automatic;
                col.Width = 70; //TODO: to be part of Persistent Setting
                col.DefaultCellStyle.BackColor = Color.Ivory;
                this.dataGridView1.Columns.Add(col);
                ncol++;
            }
            m_FirstCountColIndex = ncol;
            // 検索するコーパスの数分だけ、カウント列を生成する
            foreach (Corpus c in m_CorpusList)
            {
                DataGridViewColumn col = new DataGridViewTextBoxColumn();
                col.Name = c.Name;
                col.SortMode = DataGridViewColumnSortMode.Automatic;
                col.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
                col.Width = 50; //TODO: to be part of Persistent Setting
                col.DefaultCellStyle.BackColor = Color.LightGray;
                this.dataGridView1.Columns.Add(col);
                ncol++;
            }
            m_TotalColIndex = ncol;
            // TOTAL列を生成する
            {
                DataGridViewColumn col = new DataGridViewTextBoxColumn();
                col.Name = "All";
                col.SortMode = DataGridViewColumnSortMode.Automatic;
                col.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
                col.Width = 50; //TODO: to be part of Persistent Setting
                col.DefaultCellStyle.BackColor = Color.LightGray;
                this.dataGridView1.Columns.Add(col);
                ncol++;
            }
            m_RatioColIndex = ncol;
            // TOTAL列を生成する
            {
                DataGridViewColumn col = new DataGridViewTextBoxColumn();
                col.Name = "Ratio(%)";
                col.SortMode = DataGridViewColumnSortMode.Automatic;
                col.DefaultCellStyle.Alignment = DataGridViewContentAlignment.MiddleRight;
                col.Width = 50; //TODO: to be part of Persistent Setting
                col.DefaultCellStyle.BackColor = Color.LightGray;
                this.dataGridView1.Columns.Add(col);
                ncol++;
            }

            this.dataGridView1.ResumeUpdatingTotal();
        }

        private void WordListForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            m_Model.OnLexemeCountAdded -= new AddLexemeCountEventHandler(this.AddLexemeCountHandler);
        }

        private void AddLexemeCountHandler(object sender, AddLexemeCountEventArgs e)
        {
            this.Invoke(new AddLexemeCountDelegate(AddLexemeCount), new object[] { e });
        }

        delegate void AddLexemeCountDelegate( AddLexemeCountEventArgs e );

        private int AddNewRow(Lexeme lex)
        {
            int rowindex = this.dataGridView1.Rows.Add();
            // 行にLexemeデータをセット
            int ncol = 0;
            foreach (LP tag in Lexeme.PropertyName.Keys)
            {
                string val = "*";
                if (!m_LexFilter.IsFiltered(tag))
                {
                    val = lex.GetStringProperty(tag);
                }
                this.dataGridView1[ncol++, rowindex].Value = val;
            }
            this.dataGridView1.Rows[rowindex].HeaderCell.Value = rowindex;
            this.dataGridView1[m_TotalColIndex, rowindex].Value = (long)0;
            this.dataGridView1[m_RatioColIndex, rowindex].Value = (double)0.0;
            // 最初のデータ追加時に、TOTAL行を初期化
            if (rowindex == 1)
            {
                for (int i = 0; i < m_CorpusList.Count; i++)
                {
                    this.dataGridView1[m_FirstCountColIndex + i, 0].Value = (long)0;
                }
                this.dataGridView1[m_TotalColIndex, 0].Value = (long)0;
                this.dataGridView1[m_RatioColIndex, 0].Value = (double)0.0;
//                this.timer1.Start();
            }
            return rowindex;
        }

        private void AddLexemeCount( AddLexemeCountEventArgs e )
        {
            int rowindex = e.Index;
            if (rowindex < 0)
            {
                // 新しいLexeme行を追加する必要がある場合
                rowindex = AddNewRow(e.Lex);
            }
            // 行のカウントを更新
            int corpusindex = m_CorpusList.IndexOf(e.Cps);
            if (corpusindex < 0)
            {
                throw new Exception("Invalid Corpus in WordList Result.");
            }
            this.dataGridView1[m_FirstCountColIndex + corpusindex, rowindex].Value = e.Count;
        }


        /// <summary>
        /// Modelの内容からGridを生成しなおす。
        /// </summary>
        private void UpdateView()
        {
            this.dataGridView1.SuspendLayout();
            this.dataGridView1.SuspendUpdatingTotal();
            Cursor oldCursor = this.Cursor;
            this.Cursor = Cursors.WaitCursor;
            LexemeCountList tmpModel = new LexemeCountList(m_LexFilter);

            this.dataGridView1.ResetRows();
            if (m_Model != null)
            {
                foreach (LexemeCount lc in m_Model)
                {
                    LexemeCount lc0 = null;
                    try
                    {
                        lc0 = tmpModel[lc.Lex];
                        int rowindex = tmpModel.IndexOf(lc0) + 1;
                        // 既に存在している行にカウントをマージする。
                        for (int i = 0; i < m_CorpusList.Count; i++)
                        {
                            long c = (long)(this.dataGridView1[m_FirstCountColIndex + i, rowindex].Value);
                            c += lc.Counts[m_CorpusList[i]];
                            this.dataGridView1[m_FirstCountColIndex + i, rowindex].Value = c;
                        }
                    }
                    catch
                    {
                    }
                    if (lc0 == null)
                    {
                        // 新しい行を追加する
                        tmpModel.Add(lc);
                        int rowindex = AddNewRow(lc.Lex);
                        for (int i = 0; i < m_CorpusList.Count; i++)
                        {
                            this.dataGridView1[m_FirstCountColIndex + i, rowindex].Value = lc.Counts[m_CorpusList[i]];
                        }
                    }
                }
            }
            dataGridView1.ResumeLayout();
            this.dataGridView1.ResumeUpdatingTotal();
            this.Cursor = oldCursor;
        }

        private void dataGridView1_ColumnHeaderMouseClick(object sender, DataGridViewCellMouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                if (e.ColumnIndex < m_FirstCountColIndex)
                {
                    dataGridView1.CurrentCell = dataGridView1[e.ColumnIndex,0];
                    Point p = dataGridView1.GetCellDisplayRectangle(e.ColumnIndex, 0, true).Location;
                    this.contextMenuStrip1.Show(dataGridView1.PointToScreen(p));
                }
            }
        }

        /// <summary>
        /// 列の内容を縮退させる
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void compactRowToolStripMenuItem_Click(object sender, EventArgs e)
        {
            int col = dataGridView1.CurrentCell.ColumnIndex;
            m_LexFilter.SetFiltered((LP)col);
            UpdateView();
        }

        /// <summary>
        /// 縮退していた列の内容を元に戻す
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void expandRowToolStripMenuItem_Click(object sender, EventArgs e)
        {
            int col = dataGridView1.CurrentCell.ColumnIndex;
            if (col < 0 || col >= sizeof(LP))
            {
                return;
            }
            m_LexFilter.ResetFiltered((LP)col);
            UpdateView();
        }

        private void dataGridView1_RowHeaderMouseClick(object sender, DataGridViewCellMouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                if (e.RowIndex > 0)
                {
                    dataGridView1.CurrentCell = dataGridView1[0, e.RowIndex];
                    Point p = dataGridView1.GetCellDisplayRectangle(0, e.RowIndex, true).Location;
                    this.contextMenuStrip2.Show(dataGridView1.PointToScreen(p));
                }
            }
        }

        private void listOccurrenceToolStripMenuItem_Click(object sender, EventArgs e)
        {

        }

        /// <summary>
        /// TOTAL欄をupdateする
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void dataGridView1_UpdatingTotal(object sender, EventArgs e)
        {
            // TOTALを更新
            int rowmax = this.dataGridView1.Rows.Count;
            if (rowmax == 0)
            {
                return;
            }
            // 全体のTOTAL
            long total = 0;
            // 各コーパスごとの縦のTOTAL
            for (int i = 0; i < m_CorpusList.Count; i++)
            {
                long ctotal = 0;
                for (int j = 1; j < rowmax; j++)
                {
                    ctotal += (long)(this.dataGridView1[m_FirstCountColIndex + i, j].Value);
                }
                this.dataGridView1[m_FirstCountColIndex + i, 0].Value = ctotal;
                total += ctotal;
            }

            // 全体のTOTAL
            this.dataGridView1[m_TotalColIndex, 0].Value = total;
            this.dataGridView1[m_RatioColIndex, 0].Value = (double)100.0;

            // 各行横のTOTAL
            for (int j = 1; j < rowmax; j++)
            {
                long sum = 0;
                for (int i = 0; i < m_CorpusList.Count; i++)
                {
                    sum += (long)(this.dataGridView1[m_FirstCountColIndex + i, j].Value);
                }
                this.dataGridView1[m_TotalColIndex, j].Value = sum;
                double ratio = (double)sum * 100.0 / (double)total;
                this.dataGridView1[m_RatioColIndex, j].Value = ratio;//string.Format("{0:F1}", ratio);
            }
        }
    }
}
