<?php
/**
 * 1ファイルの最大のコンテンツサイズ
 */
const MAX_CONTENT_SIZE =  800000;

/**
 * PNGファイルの最大サイズ
 */
const MAX_PNG_SIZE     = 1000000;

/**
 * データベース名
 */
const DB_NAME = __DIR__ . '/mydb.db';

/**
 * ログファイル名
 */
const LOGFILE = __DIR__ . '/entries.log';

/**
 * ZIPファイル格納先
 */
const ZIPDIR = __DIR__ . '/../zip/';

/**
 * トリップに使うソルト
 */
const TRIPSALT = '#TRIPSALT#';

/**
 * ホストアドレスのハッシュに使うソルト
 */
const HOSTADDR_SALT = '@@IP@@';

/**
 * 最大の失敗回数
 */
const MAX_FAIL_COUNT = 5;

/**
 * データベースオブジェクトを返す。
 * FETCH_ASSOC, ERRMODE_EXCEPTIONが設定される。
 */
function create_pdo() {
    // sqlite3のdbファイルをグループでの読み書き可能にする
    $db_path = DB_NAME;
    touch($db_path);
    chmod($db_path, 0664);

    $pdo = new PDO('sqlite:' . $db_path);
    // SQL実行時にもエラーの代わりに例外を投げるように設定
    // (毎回if文を書く必要がなくなる)
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);

    // テーブル作成

    // レポート(通報)
    // 通報内容を記録する。
    $pdo->exec("CREATE TABLE IF NOT EXISTS REPORT_ENTRIES(
        zipid INTEGER NOT NULL,
        remote_addr VARCHAR NOT NULL,
        report_type VARCHAR,
        comment VARCHAR,
        primary key (zipid, remote_addr)
    )");

    // リモートアドレス
    // アップロードしてきたリモートアドレスを管理する
    $pdo->exec("CREATE TABLE IF NOT EXISTS REMOTE_ENTRIES(
        remote_addr VARCHAR PRIMARY KEY,
        regdate1 TIMESTAMP,
        regdate2 TIMESTAMP,
        regdate3 TIMESTAMP,
        regdate4 TIMESTAMP,
        regdate5 TIMESTAMP,
        faildate TIMESTAMP,
        failcount INTEGER
    )");

    // ZIPファイル一覧
    $pdo->exec("CREATE TABLE IF NOT EXISTS ZIP_ENTRIES(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        fname VARCHAR UNIQUE NOT NULL,
        fsize INTEGER,
        href VARCHAR,
        title VARCHAR,
        author VARCHAR,
        delkey VARCHAR,
        license VARCHAR,
        comment VARCHAR,
        regdate TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        hostaddr VARCHAR,
        downloaded INTEGER DEFAULT 0
    )");
    
    // タグ一覧
    $pdo->exec("CREATE TABLE IF NOT EXISTS TAG_ENTRIES(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        tagname VARCHAR UNIQUE NOT NULL,
        sticky INTEGER DEFAULT 0,
        regdate TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        hostaddr VARCHAR
    )");

    // ZIPファイルが保持するパーツ一覧
    $pdo->exec("CREATE TABLE IF NOT EXISTS PARTS_ENTRIES(
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        zipid INTEGER NOT NULL,
        partsname VARCHAR
    )");
    $pdo->exec("CREATE INDEX IF NOT EXISTS PARTS_ENTRIES_IX ON PARTS_ENTRIES(zipid)");
    
    // パーツとタグとの関連テーブル
    $pdo->exec("CREATE TABLE IF NOT EXISTS TAG_PARTS_REL(
        tagid INTEGER NOT NULL,
        partsid INTEGER NOT NULL,
        sticky INTEGER DEFAULT 0
    )");
    $pdo->exec("CREATE UNIQUE INDEX IF NOT EXISTS TAG_PARTS_REL_IX1 ON TAG_PARTS_REL(tagid, partsid)");
    $pdo->exec("CREATE UNIQUE INDEX IF NOT EXISTS TAG_PARTS_REL_IX2 ON TAG_PARTS_REL(partsid, tagid)");
    $pdo->exec("CREATE TABLE IF NOT EXISTS TAG_ZIP_REL(
        tagid INTEGER NOT NULL,
        zipid INTEGER NOT NULL,
        sticky INTEGER DEFAULT 0
    )");
    $pdo->exec("CREATE UNIQUE INDEX IF NOT EXISTS TAG_ZIP_REL_IX1 ON TAG_ZIP_REL(tagid, zipid)");
    $pdo->exec("CREATE UNIQUE INDEX IF NOT EXISTS TAG_ZIP_REL_IX1 ON TAG_ZIP_REL(zipid, tagid)");
    return $pdo;
}

/**
 * リモートアドレスを取得する。
 * プロキシ系経由である場合は、その双方が@で接続されて表記される。
 * @return string リモートアドレス
 */
function get_hostaddr() {
    if (array_key_exists('HTTP_X_FORWARDED_FOR', $_SERVER)) {
        return $_SERVER['HTTP_X_FORWARDED_FOR'] . '@' . $_SERVER['REMOTE_ADDR'];
    }
    return $_SERVER['REMOTE_ADDR'];
}

/**
 * ログファイルにメッセージを追記する。
 * 行の先頭には日付(GMT)とリモートアドレスが記録される。
 * 書き込みは排他的に行われる。
 * @param string メッセージ
 */
function append_log($msg) {
    $hostaddr = get_hostaddr();
    $date = gmdate('Y-m-d H:i:s');
    $path = LOGFILE;
    file_put_contents($path, $date . ' ' . $hostaddr . ' ' . $msg . PHP_EOL , FILE_APPEND | LOCK_EX);
}

/**
 * URLセーフなBASE64変換.
 * https://qiita.com/YamasakiKenta/items/7ea2579a5e99477f950a
 * @param string $data
 * @return URLセーフなbase64文字列
 */ 
function base64url_encode($data) { 
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '='); 
} 

/**
 * URLセーフなBASE64のデコード
 * @param string $data URLセーフなbase64文字列
 * @return デコードされたデータ
 */
function base64url_decode($data) { 
    return base64_decode(str_pad(strtr($data, '-_', '+/'), strlen($data) % 4, '=', STR_PAD_RIGHT)); 
} 
?>