<?php
////////////////////////////////////
// zipファイルのアップロードのCGI
////////////////////////////////////

ini_set('display_errors', 1);
require_once __DIR__ . '/common.php';
require_once __DIR__ . '/access_ctrl.php';
require_once __DIR__ . '/calc_trip.php';
require_once __DIR__ . '/read_catalog.php';

header("Content-type: application/json; charset=utf-8");

// パラメータの取り出し

$title = isset($_POST['title']) ? $_POST['title'] : '';
$author = isset($_POST['author']) ? $_POST['author'] : '';
$delkey = isset($_POST['delkey']) ? $_POST['delkey'] : '';
$license = isset($_POST['license']) ? $_POST['license'] : '';
$comment = isset($_POST['comment']) ? $_POST['comment'] : '';
$validation = isset($_POST['validation']) ? ($_POST['validation'] == 'true') : false;

$result = ['result' => 'error'];

// フォームの入力データが大きすぎればエラーとして何もしない。
if (strlen($title) > 5000 || strlen($author) > 500 ||
    strlen($delkey) > 500 || strlen($license) > 500 ||
    strlen($comment) > 20000) {
    echo json_encode($result);
    exit;
}

$hostaddr = get_hostaddr();

// 一時ファイルができているか（アップロードされているか）チェック
if (isset($_FILES['file'])) {
    $file = $_FILES['file'];
    if ($file['error'] == 0) {
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $fsize = $file['size'];
        if (($ext != 'zip' && $ext != 'cmj') || $fsize < 1000) {
            // 拡張子がzip, cmjでない、もしくは異常に小さい(1kb以下はありえない)
            $result = ['result' => 'error', 'message' => '不正なZIPファイルです。'];
    
        } else {
            // zipの中身のチェック
            $result = check_zip($file['tmp_name'], $validation);
            if ($result['result'] == 'ok') {
                // 一時ファイルから所定フォルダへの退避
                $fname = sha1_file($file['tmp_name']) . '.' . $ext;
                $destzip = ZIPDIR . $fname;
                if (move_uploaded_file($file['tmp_name'], $destzip)) {
                    // データベースへの登録
                    $result = append_entry($fname, $fsize, $title, $author, $delkey, $license, $comment, $hostaddr);
                    if ($result['result'] == 'ok') {
                        // catalog.txtの解析とデータベースへの登録
                        $zipid = $result['id'];
                        search_zip_catalog($destzip, $zipid, $hostaddr);
                    }
                } else {
                    $result = ['result' => 'error', 'message' => 'ファイルの保存に失敗しました。'];
                }
            }
        }
    } else {
        $result = ['result' => 'error', 'message' => 'ZIPファイルのアップロードでエラーが発生しました。err=' . $file['error']];
    }
} else {
    $result = ['result' => 'error', 'message' => 'ZIPファイルがアップロードされていません。'];
}

echo json_encode($result);

/**
 * ZIPの中身を検証する
 * 
 * zipファイルが正しく、png, jpeg, txt以外を含んでいないこと
 * 各サイズが800kb以下のこと。
 * 
 * @param string $zipfile ZIPファイルのパス
 * @return array 結果を格納したarray
 */
function check_zip($zipfile, $validation) {
    $error_files = [];
    $large_files = [];

    $za = new ZipArchive();
    $res = $za->open($zipfile);
    if ($res === TRUE) {
        $filecnt = ['png' => 0, 'txt' => 0, 'jpeg' => 0, 'jpg' => 0,
            'ini' => 0, 'xml' => 0, 'gif' => 0, 'root_png' => 0];
        $hasCatalog = false;
        $hasReadme = false;
        for ($i = 0; $i < $za->numFiles; $i++) { 
            $stat = $za->statIndex($i); 
            $entryname = $stat['name'];
            $entryname = str_replace('\\', '/', $entryname); // バックスラッシュで入っているzipもあり
            if (!preg_match('/\/$/', $entryname)) {
                // フォルダ以外なら、ファイル名と拡張子、サイズを取り出す
                $dirname = pathinfo($entryname, PATHINFO_DIRNAME);
                $fname = pathinfo($entryname, PATHINFO_FILENAME);
                $ext = strtolower(pathinfo($entryname, PATHINFO_EXTENSION));
                $lcname = strtolower(pathinfo($entryname, PATHINFO_BASENAME));
                $fsize = $stat['size'];
                if ($fname != 'license' && $fname != 'readme' && $fname != 'read me' &&
                    $lcname != 'thumbs.db' && $lcname != '.ds_store' &&
                    $ext != 'txt' && $ext != 'ini' && $ext != 'xml' && $ext != 'cpd' &&
                    $ext != 'png' && $ext != 'jpg' && $ext != 'jpeg' && $ext != 'gif') {
                    // 対象外のファイル拡張子、またはファイル名
                    $error_files[] = $entryname;
                } else {
                    // ファイル数のカウント
                    if ($ext == 'png' && $dirname == '.') {
                        @$filecnt['root_png']++; // 親ディレクトリ上にあるPNGの数(パーツではない)
                    } else {
                        @$filecnt[$ext]++; // @で初回undefuned警告を握りつぶす
                    }

                    // カタログファイルか？
                    if ($lcname == 'catalog.txt') {
                        $hasCatalog = true;
                    }
                    // readmeか？
                    if ($fname == 'readme' || $fname == 'read me') {
                        $hasReadme = true;
                    }

                    // ファイルサイズが大きすぎる
                    if (!(($ext == 'png' && $fsize <= MAX_PNG_SIZE) || 
                          ($ext != 'xml' && $fsize <= MAX_CONTENT_SIZE))) {
                        // pngならMAX_PNG_SIZE以下、xml以外ならMAX_CONTENT_SIZE以下のこと。
                        // そうでなければ大きすぎる.
                        $large_files[] = $entryname;
                    }

                    // validationの場合は、隠しファイル, gifもエラーにする
                    if ($validation) {
                        if ($lcname == 'thumbs.db' || $lcname == '.ds_store') {
                            $error_files[] = $entryname;
                        }
                        if ($ext == 'gif') {
                            $error_files[] = $entryname;
                        }
                    }
                }
            }
        }
        $za->close();
        if ($validation && (!$hasCatalog || !$hasReadme)) {
            return ['result' => 'error',
                'message' => 'ZIPファイル内にreadme.txt、またはcatalog.txtが含まれていません。',
                'error_files' => $error_files, 'large_files' => $large_files];
        } else if (count($error_files) > 0 || count($large_files) > 0) {
            return ['result' => 'error',
                'message' => 'ZIPファイル内に許可されない、もしくは大きすぎるファイルが含まれています。',
                'error_files' => $error_files, 'large_files' => $large_files];
        } else if ($filecnt['png'] == 0) {
            return ['result' => 'error',
                'message' => 'ZIPファイル内に有効なパーツ画像がありません'];
        } else if (($filecnt['root_png'] + $filecnt['jpg'] + $filecnt['jpeg'] +
             $filecnt['gif'] + $filecnt['txt'] + $filecnt['xml']) > 5) {
            return ['result' => 'error',
                'message' => 'ZIPファイル内にパーツ画像以外のファイルが多数あります。'];
        }
        return ['result' => 'ok', 'message' => 'valid zip'];
    }
    return ['result' => 'error', 'message' => 'ZIPファイルが不正です', 'error_code' => $res];
}

/**
 * データベースにエントリを追加または更新する
 * 
 * @param string $fname ファイル名
 * @param integer $fsize ファイルサイズ
 * @param string $title タイトル
 * @param string $author 作者名
 * @param string $delkey 削除キー
 * @param integer $license ライセンスタイプ
 * @param string $fncommentame コメント
 * @param string $hostaddr リモートアドレス
 */
function append_entry($fname, $fsize, $title, $author, $delkey, $license, $comment, $hostaddr) {
    $pdo = create_pdo();
    
    // アップロード回数制限中か？
    $accessCtrl = new AccessCtrl($pdo);
    if (!$accessCtrl->register($hostaddr) || $accessCtrl->getFailCount($hostaddr) > MAX_FAIL_COUNT) {
        return [
            'result' => 'error',
            'message' => 'アップロード制限中です。しばらくお待ちください。'
        ];
    }

    // エントリの確認
    $fetch = $pdo->prepare('SELECT id, author, delkey FROM ZIP_ENTRIES WHERE fname = ?');
    $fetch->execute([$fname]);
    $row = $fetch->fetch();
    $fetch = null;

    if ($row === FALSE) {
        // 新規挿入
        $stmt = $pdo->prepare('INSERT INTO ZIP_ENTRIES(fname, fsize, title, author, delkey,
            license, comment, hostaddr)
            VALUES (:fname, :fsize, :title, :author, :delkey, :license, :comment, :hostaddr)');
        $stmt->execute(['fname' => $fname, 'fsize' => $fsize, 'title' => $title, 'author' => $author,
            'delkey' => $delkey, 'license' => $license, 'comment' => $comment,
            'hostaddr' => $hostaddr]);
        $id = $pdo->lastInsertId();
        append_log("insert ${id} ${fname} ${fsize} ${author} ${delkey} ${title}");
        $stmt = null;

    } else {
        // 更新
        $id = $row['id'];
        $prev_author = $row['author'];
        $prev_delkey = $row['delkey'];

        $passphrase = get_trip_pass($author);
        $prev_passphrase = get_trip_pass($prev_author);
        if (($prev_passphrase == '' && $delkey == $prev_delkey) ||
            ($prev_passphrase != '' && $prev_passphrase == $passphrase)) {
            // トリップ指定がない場合はdelkeyの一致、トリップの指定がある場合はトリップの一致をもって書き換え可とする
            append_log("update ${id}");
            $stmt = $pdo->prepare('UPDATE ZIP_ENTRIES
                SET title = :title, author = :author, delkey = :delkey,
                    license = :license, comment = :comment, hostaddr = :hostaddr
                WHERE id = :id');
            $stmt->execute(['id' => $id, 'title' => $title, 'author' => $author,
                'delkey' => $delkey, 'license' => $license, 'comment' => $comment,
                'hostaddr' => $hostaddr]);
            $stmt = null;
        } else {
            append_log("unmatch delkey. zipid:${id}");
            $accessCtrl->failIncrement($hostaddr);
            $accessCtrl = null;
            $pdo = null;
            return ['result' => 'error', 'message' => 'トリップまたはdelkeyが一致しません'];
        }
    }
    $accessCtrl = null;
    $pdo = null;
    return ['result' => 'ok', 'id' => $id, 'hostaddr' => $hostaddr];
}
?>