package charactermanaj.clipboardSupport;

import java.awt.Color;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.image.BufferedImage;
import java.io.IOException;


/**
 * クリップボード用ユーテリティクラス.<br>
 * @author seraphy
 */
public final class ClipboardUtil {

	private ClipboardUtil() {
		super();
	}
	
	/**
	 * クリップボードにイメージを設定する.<br>
	 * JDKのクリップボード経由の画像転送では透過色を表現できないので、背景色を指定する必要がある.<br>
	 * (ただし、このアプリケーション内であれば透過色を維持したままコピー可能.)<br>
	 * @param img イメージ
	 * @param bgColor 背景色
	 */
	public static void setImage(BufferedImage img, Color bgColor) {
		if (img == null || bgColor == null) {
			throw new IllegalArgumentException();
		}

		Toolkit tk  = Toolkit.getDefaultToolkit();
		Clipboard cb = tk.getSystemClipboard();
		
		ImageSelection imageSelection = new ImageSelection(img, bgColor);
		cb.setContents(imageSelection, null);
	}

	/**
	 * クリップボード内にイメージがあるか?
	 * @return イメージがあればtrue
	 */
	public static boolean hasImage() {
		Toolkit tk = Toolkit.getDefaultToolkit();
		Clipboard cb = tk.getSystemClipboard();
		return cb.isDataFlavorAvailable(DataFlavor.imageFlavor)
				|| cb.isDataFlavorAvailable(ImageSelection.bufferedImageDataFlavor);
	}
	
	/**
	 * クリップボードからイメージを取得する.
	 * @return 
	 * @throws IOException
	 */
	public static BufferedImage getImage() throws IOException {
		Toolkit tk = Toolkit.getDefaultToolkit();
		Clipboard cb = tk.getSystemClipboard();

		try {
			if (cb.isDataFlavorAvailable(ImageSelection.bufferedImageDataFlavor)) {
				// JavaVM内でのコピーを試みる。
				// コピー元が透過画像であれば取得できるのも透過画像である
				return (BufferedImage) cb.getData(ImageSelection.bufferedImageDataFlavor);
			}
			
			if (cb.isDataFlavorAvailable(DataFlavor.imageFlavor)) {
				// 汎用の画像形式で取得を試みる。
				// 透過画像は使えないため、ここで取得されるものは非透過画像である。
				return (BufferedImage) cb.getData(DataFlavor.imageFlavor);
			}

		} catch (UnsupportedFlavorException ex) {
			// 直前にisDataFlavorAvailableで確認しているので、
			// よほどタイミングが悪くなければエラーは発生しないはず。
			throw new IOException(ex.getMessage());
		}
		
		return null;
	}
	
}
