package charactermanaj.util;

import java.io.File;
import java.net.URI;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * ユーザーデータの保存先を生成するファクトリ
 * @author seraphy
 */
public class UserDataFactory {
	
	/**
	 * ロガー
	 */
	private static final Logger logger = Logger.getLogger(UserDataFactory.class.getName());

	/**
	 * シングルトン
	 */
	private static UserDataFactory inst = new UserDataFactory();
	
	/**
	 * インスタンスを取得する.
	 * @return インスタンス
	 */
	public static UserDataFactory getInstance() {
		return inst;
	}
	
	/**
	 * プライベートコンストラクタ
	 */
	private UserDataFactory() {
		super();
	}

	/**
	 * 拡張子を含むファイル名を指定し、そのファイルが保存されるべきユーザディレクトリを判定して返す.<br>
	 * nullまたは空の場合、もしくは拡張子がない場合はユーザディレクトリのルートを返します.<br>
	 * フォルダがなければ作成されます.<br>
	 * @param name ファイル名、もしくはnull
	 * @return ファィルの拡張子に対応したデータ保存先フォルダ
	 */
	public File getSpecialDataDir(String name) {
		File userDataDir = ConfigurationDirUtilities.getUserDataDir();
		
		if (name != null && name.length() > 0) {
			int pos = name.lastIndexOf('.');
			if (pos >= 0) {
				String ext = name.substring(pos + 1);
				if (ext.length() > 0) {
					if ("ser".equals(ext)) {
						userDataDir = new File(userDataDir, "caches");
					} else {
						userDataDir = new File(userDataDir, ext + "s");
					}
				}
			}
		}

		// フォルダがなければ作成する.
		if (!userDataDir.exists()) {
			boolean result = userDataDir.mkdirs();
			logger.log(Level.INFO, "makeDir: " + userDataDir + " /succeeded=" + result);
		}
		
		return userDataDir;
	}
	
	/**
	 * 指定した名前のユーザーデータ保存先を作成する.
	 * @param name ファイル名
	 * @return 保存先
	 */
	public UserData getUserData(String name) {
		if (name == null || name.trim().length() == 0) {
			throw new IllegalArgumentException();
		}
		return new FileUserData(new File(getSpecialDataDir(name), name));
	}

	/**
	 * docBaseごとにのハッシュ値を文字列表現化したプレフィックスをもつユーザーデータ保存先を作成する.<br>
	 * docBaseのURIの圧縮を目的としており、等しいdocBaseは等しいプレフィックスによるようにしている.(暗号化が目的ではない).<br>
	 * ハッシュ値はmd5の5バイトで生成されるため、nameを工夫して衝突の確率を軽減するか衝突しても問題ないように考慮することが望ましい.<Br>
	 * @param docBase URI、null可
	 * @param name ファイル名
	 * @return 保存先
	 */
	public UserData getMangledNamedUserData(URI docBase, String name) {
		String prefix = getMangledNamedPrefix(docBase);
		return getUserData(prefix + "-" + name);
	}

	/**
	 * docBaseごとにのハッシュ値を文字列表現化したプレフィックスを返す.<br>
	 * docBaseがnullの場合は空文字とみなす.<br>
	 * @param docBase URI、null可
	 * @return ハッシュ値の文字列表現
	 */
	public String getMangledNamedPrefix(URI docBase) {
		String docBaseStr;
		if (docBase == null) {
			docBaseStr = "";
		} else {
			docBaseStr = docBase.toString();
		}
		String prefix = UUID.nameUUIDFromBytes(docBaseStr.getBytes()).toString();
		return prefix;
	}
}

