package charactermanaj.util;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * ユーザーデータの保存先を生成するファクトリ
 * @author seraphy
 */
public class UserDataFactory {
	
	/**
	 * ロガー
	 */
	private static final Logger logger = Logger.getLogger(UserDataFactory.class.getName());

	/**
	 * MANGLED管理ファイル名
	 */
	private static final String META_FILE = "mangled_info.xml";
	
	/**
	 * シングルトン
	 */
	private static UserDataFactory inst = new UserDataFactory();
	
	/**
	 * MangledNameのキャッシュ
	 */
	private HashMap<URI, String> mangledNameMap = new HashMap<URI, String>();
	
	/**
	 * インスタンスを取得する.
	 * @return インスタンス
	 */
	public static UserDataFactory getInstance() {
		return inst;
	}
	
	/**
	 * プライベートコンストラクタ
	 */
	private UserDataFactory() {
		super();
	}

	/**
	 * 拡張子を含むファイル名を指定し、そのファイルが保存されるべきユーザディレクトリを判定して返す.<br>
	 * nullまたは空の場合、もしくは拡張子がない場合はユーザディレクトリのルートを返します.<br>
	 * フォルダがなければ作成されます.<br>
	 * @param name ファイル名、もしくはnull
	 * @return ファィルの拡張子に対応したデータ保存先フォルダ
	 */
	public File getSpecialDataDir(String name) {
		File userDataDir = ConfigurationDirUtilities.getUserDataDir();
		
		if (name != null && name.length() > 0) {
			int pos = name.lastIndexOf('.');
			if (pos >= 0) {
				String ext = name.substring(pos + 1);
				if (ext.length() > 0) {
					if ("ser".equals(ext)) {
						userDataDir = new File(userDataDir, "caches");
					} else {
						userDataDir = new File(userDataDir, ext + "s");
					}
				}
			}
		}

		// フォルダがなければ作成する.
		if (!userDataDir.exists()) {
			boolean result = userDataDir.mkdirs();
			logger.log(Level.INFO, "makeDir: " + userDataDir + " /succeeded=" + result);
		}
		
		return userDataDir;
	}
	
	/**
	 * 指定した名前のユーザーデータ保存先を作成する.
	 * @param name ファイル名
	 * @return 保存先
	 */
	public UserData getUserData(String name) {
		if (name == null || name.trim().length() == 0) {
			throw new IllegalArgumentException();
		}
		return new FileUserData(new File(getSpecialDataDir(name), name));
	}

	/**
	 * docBaseごとにのハッシュ値を文字列表現化したプレフィックスをもつユーザーデータ保存先を作成する.<br>
	 * docBaseのURIの圧縮を目的としており、等しいdocBaseは等しいプレフィックスによるようにしている.(暗号化が目的ではない).<br>
	 * ハッシュ値はmd5の5バイトで生成されるため、既存のものと衝突した場合は末尾に数値が付与される.<br>
	 * @param docBase URI、null可
	 * @param name ファイル名
	 * @return 保存先
	 */
	public UserData getMangledNamedUserData(URI docBase, String name) {
		String mangledName = mangledNameMap.get(docBase);
		if (mangledName == null) {
			File storeDir = getSpecialDataDir(name);
			mangledName = registerMangledName(docBase, storeDir);
			mangledNameMap.put(docBase, mangledName);
		}
		return getUserData(mangledName + "-" + name);
	}
	
	/**
	 * ハッシュ化文字列のデータベースファイルをオープンし、現在の登録をすべて読み込む.<br>
	 * クローズされるまでデータベースはロックされた状態となる.<br>
	 * @param storeDir データベースファイルの格納フォルダ
	 * @return ハッシュ化文字列データベース
	 * @throws IOException 失敗
	 */
	protected FileMappedProperties openMetaFile(File storeDir) throws IOException {
		File metaFile = new File(storeDir, META_FILE);

		FileMappedProperties mangledProps = new FileMappedProperties(metaFile);

		// ロードされていなければロードする.
		// (ロードに失敗した場合は空の状態とみなして継続する.)
		try {
			mangledProps.load();

		} catch (Exception ex) {
			logger.log(Level.WARNING, "mangled database is broken."
					+ mangledProps.getFile(), ex);
		}
		
		return mangledProps;
	}
	
	private static final String URI_KEY_PREFIX = "uri.";
	private static final String MANGLED_NAME_PREFIX = "mangled.";


	/**
	 * DocBaseのURIを辞書に登録し、そのハッシュ化文字列を返す.<br>
	 * ハッシュの衝突を回避するための辞書ファイル「mangled_info.xml」を使用して、
	 * 衝突した場合は末尾に連番をふることで補正を行う.<br>
	 * すでに登録済みの同じuriの場合は同じハッシュ化文字列を返す.<br>
	 * @param docBase URI
	 * @param mangledProps 辞書ファイル
	 * @return ハッシュ化文字列(衝突した場合は連番が振られる)
	 */
	protected String registerMangledName(URI docBase, FileMappedProperties mangledProps) {
		final String noAdjustedMangledName = getNoAdjustedMangledName(docBase);
		String adjustedMangledName = noAdjustedMangledName;

		// ハッシュ化文字列に対応するURIのリストを取得する.
		// (通常は一個、まれにハッシュが衝突した場合に複数になる.)
		final String mangledLookupKey = "mangled_base." + noAdjustedMangledName;
		String sameMangledURIList = mangledProps.get(mangledLookupKey);
		List<String> uris;
		if (sameMangledURIList != null && sameMangledURIList.length() > 0) {
			// ハッシュ化文字列に対するURIのリストは空白区切りであるとみなしてリストに変換する.
			uris = Arrays.asList(sameMangledURIList.split("\\s+"));
		
		} else {
			// 新規のハッシュ化文字列になる場合は空のリストとする.
			uris = Collections.emptyList();
			sameMangledURIList = "";
		}
		
		final String uri = docBase.toASCIIString();
		final String registeredMangledName = mangledProps.get(URI_KEY_PREFIX + uri);
		if (!uris.contains(uri) || registeredMangledName == null || registeredMangledName.length() == 0) {
			// まだハッシュ化文字列に、そのURIが未登録の場合、
			// もしくは、そのURIに対するハッシュ化文字列の索引がない場合
			int pos = uris.indexOf(uri);
			if (pos < 0) {
				if (!uris.isEmpty()) {
					// まだ未登録である場合、同じUUIDであれば末尾に数値をつける.
					// (同一のUUIDがなければ末尾は付与しない)
					adjustedMangledName += "_" + uris.size();
				}

				// 登録済みUUIDリストに追加する.
				if (sameMangledURIList.length() > 0) {
					sameMangledURIList += " "; // 空白区切り (URIの文字列表現では空白は含まれないため問題なし)
				}
				sameMangledURIList += uri;

			} else {
				// すでに登録済みであれば、再度、そのインデックスを使用する.
				if (pos > 0) {
					adjustedMangledName += "_" + pos;
				}
			}
			
			// 登録
			mangledProps.put(MANGLED_NAME_PREFIX + adjustedMangledName, uri); // 補正後UUIDからURIへの索引
			mangledProps.put(URI_KEY_PREFIX + uri, adjustedMangledName); // URIから補正語UUIDへの索引
			mangledProps.put(mangledLookupKey, sameMangledURIList); // 補正前UUIDを使用するURIリスト

		} else {
			// 登録済みの場合
			adjustedMangledName = registeredMangledName;
		}
		
		return adjustedMangledName;
	}

	/**
	 * DocBaseのURIを辞書に登録し、そのハッシュ化文字列を返す.<br>
	 * ハッシュの衝突を回避するための辞書ファイル「mangled_info.xml」を使用して、
	 * 衝突した場合は末尾に連番をふることで補正を行う.<br>
	 * @param docBase URI
	 * @param storeDir 格納先ディレクトリ(格納先単位で辞書ファイルが作成される)
	 * @return ハッシュ化文字列(衝突した場合は連番が振られる)
	 */
	protected String registerMangledName(URI docBase, File storeDir) {
		final String noAdjustedMangledName = getNoAdjustedMangledName(docBase);
		String adjustedMangledName = noAdjustedMangledName;
		FileMappedProperties mangledProps = null;
		try {
			mangledProps = openMetaFile(storeDir);
			try {
				// 名前を登録する.
				adjustedMangledName = registerMangledName(docBase, mangledProps);
	
				// 追加・変更されていたら保存する.
				if (mangledProps.isModified()) {
					mangledProps.save();
				}

			} finally {
				mangledProps.close();
			}

		} catch (Exception ex) {
			logger.log(
					Level.WARNING,
					"mangled database is broken." + ((mangledProps == null) ? storeDir
							: mangledProps.getFile()), ex);
		}
		
		return adjustedMangledName;
	}

	/**
	 * 登録されている、すべてのハッシュ化文字列に対するURIを取得する.<br>
	 * @param mangledNames ハッシュ化文字列のコレクション
	 * @param storeDir 格納先ディレクトリ(格納先単位で辞書ファイルが作成される)
	 * @param ope ハッシュ化文字列に対応するURIが発見された場合のオペレーション
	 */
	public Map<String, URI> getMangledNameMap(File storeDir) {
		if (storeDir == null) {
			throw new IllegalArgumentException();
		}
		
		HashMap<String, URI> uris = new HashMap<String, URI>();
		
		FileMappedProperties mangledProps = null;
		try {
			mangledProps = openMetaFile(storeDir);
			try {
				for (Map.Entry<String, String> propsEntry : mangledProps.entrySet()) {
					String key = propsEntry.getKey();
					String value = propsEntry.getValue();
					if (key.startsWith("mangled.")) {
						try {
							String mangledName = key.substring(8);
							URI uri = new URI(value);
							
							if (logger.isLoggable(Level.FINEST)) {
								logger.log(Level.FINEST, "registered mangled name: " + mangledName + "=" + uri);
							}
							uris.put(mangledName, uri);

							// 現在のすべての登録をキャッシュにも格納する.
							// (ただし変換できない場合は無視する)
							try {
								// URIはASCII変換されているため、一旦、ファイル位置に変換する.
								URI docBase = new File(uri).toURI();
								mangledNameMap.put(docBase, mangledName);
								if (logger.isLoggable(Level.FINE)) {
									logger.log(Level.FINE, "mangled-name cached. " + docBase + "=" + mangledName);
								}
							} catch (Exception ex) {
								logger.log(Level.FINE, "mangled-name decode error." + uri, ex);
							}

						} catch (Exception ex) {
							logger.log(Level.WARNING,
									"mangled database is broken."
											+ mangledProps.getFile(), ex);
						}
					}
				}

			} finally {
				mangledProps.close();
			}

		} catch (Exception ex) {
			logger.log(
					Level.WARNING,
					"mangled database is broken." + ((mangledProps == null) ? storeDir
							: mangledProps.getFile()), ex);
		}
		
		return uris;
	}
	
	/**
	 * URIのコレクションを指定して、それぞれの登録済みのハッシュ化文字列をマップとして返す.<br>
	 * 登録フラグがfalseの場合、まだ登録されていないものはnullとなる.<br>
	 * そうでない場合は新規に登録され、その値が設定される.<br>
	 * @param uris URIのコレクション
	 * @param storeDir 格納先ディレクトリ(格納先単位で辞書ファイルが作成される)
	 * @param register 検索時に存在しなければ登録する場合はtrue
	 * @return URIをキーとし、ハッシュ化文字列を値とするマップ。登録されていないURIはnullが値となる.<br>
	 */
	public Map<URI, String> getMangledNameMap(Collection<URI> uris, File storeDir, boolean register) {
		if (storeDir == null) {
			throw new IllegalArgumentException();
		}
		if (uris == null || uris.isEmpty()) {
			// nullまたは空の場合は空を返す.
			return Collections.emptyMap();
		}
		
		HashMap<URI, String> results = new HashMap<URI, String>();
		
		FileMappedProperties mangledProps = null;
		try {
			mangledProps = openMetaFile(storeDir);
			try {
				for (URI uri : uris) {
					if (uri == null) {
						continue; // 不正uri
					}

					String mangledName = mangledProps.get(uri.toASCIIString());
					if (mangledName == null) {
						// 未登録の場合
						if (register && uri != null) {
							// 登録する場合
							mangledName = registerMangledName(uri, mangledProps);
						}
					}
					results.put(uri, mangledName);
				}

				// 登録する場合で変更があれば保存する.
				if (register && mangledProps.isModified()) {
					mangledProps.save();
				}
				
			} finally {
				mangledProps.close();
			}
		} catch (Exception ex) {
			logger.log(
					Level.WARNING,
					"mangled database is broken." + ((mangledProps == null) ? storeDir
							: mangledProps.getFile()), ex);
		}
		return results;
	}

	/**
	 * docBaseをハッシュ値化文字列にした、補正前の文字列を返す.<br>
	 * docBaseがnullの場合は空文字とみなして変換する.<br>
	 * @param docBase URI、null可
	 * @return ハッシュ値の文字列表現
	 */
	private String getNoAdjustedMangledName(URI docBase) {
		String docBaseStr;
		if (docBase == null) {
			docBaseStr = "";
		} else {
			docBaseStr = docBase.toString();
		}
		String mangledName = UUID.nameUUIDFromBytes(docBaseStr.getBytes()).toString();

		if (logger.isLoggable(Level.FINEST)) {
			logger.log(Level.FINEST, "mangledName " + docBase + "=" + mangledName);
		}
		
		return mangledName;
	}
}

