package charactermanaj.ui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.dnd.DropTarget;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.List;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DefaultListModel;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButton;
import javax.swing.JRootPane;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import charactermanaj.Main;
import charactermanaj.clipboardSupport.ClipboardUtil;
import charactermanaj.graphics.io.FileImageResource;
import charactermanaj.graphics.io.ImageCachedLoader;
import charactermanaj.graphics.io.LoadedImage;
import charactermanaj.model.AppConfig;
import charactermanaj.model.CharacterData;
import charactermanaj.model.io.CharacterDataPersistent;
import charactermanaj.model.io.PartsImageDirectoryWatchAgent;
import charactermanaj.model.io.PartsImageDirectoryWatchAgentFactory;
import charactermanaj.ui.util.FileDropTarget;
import charactermanaj.util.DesktopUtilities;
import charactermanaj.util.ErrorMessageHelper;
import charactermanaj.util.LocalizedResourcePropertyLoader;
import charactermanaj.util.UIHelper;


/**
 * プロファイルを選択するためのダイアログ、およびデフォルトプロファイルを開く
 * @author seraphy
 */
public class ProfileSelectorDialog extends JDialog {

	private static final long serialVersionUID = -6883202891172975022L;
	
	private static final Logger logger = Logger.getLogger(ProfileSelectorDialog.class.getName());


	protected static final String STRINGS_RESOURCE = "languages/profileselectordialog";
	
	/**
	 * サンプルイメージをロードするためのローダー
	 */
	private ImageCachedLoader imageLoader = new ImageCachedLoader();

	
	/**
	 * サンプルイメージファイルが保存可能であるか?<br>
	 * 有効なキャラクターデータが選択されており、サンプルイメージの更新が許可されていればtrue.<br>
	 */
	private boolean canWriteSamplePicture;
	
	/**
	 * サンプルイメージを表示するパネル
	 */
	private SamplePicturePanel sampleImgPanel;

	
	private JButton btnOK;
	
	private JButton btnProfileNew;
	
	private JButton btnProfileEdit;

	private JButton btnProfileRemove;
	
	private JButton btnProfileBrowse;

	private JButton btnProfileImport;

	private JButton btnProfileExport;

	/**
	 * プロファイル一覧を表示するリストコンポーネント
	 */
	private JList characterList;
	
	/**
	 * プロファイル一覧のリストモデル
	 */
	private DefaultListModel characterListModel;
	
	/**
	 * プロファイルの説明用テキストエリア
	 */
	private JTextArea descriptionArea;
	
	

	/**
	 * ダイアログをOKで閉じた場合に選択していたキャラクターデータを示す.<br>
	 * nullの場合はキャンセルを意味する.
	 */
	private CharacterData selectedCharacterData;

	
	/**
	 * プロファイルの選択ダイアログを構築する.
	 * @param parent 親フレーム、もしくはnull
	 * @param characterDatas プロファイルのリスト
	 */
	public ProfileSelectorDialog(JFrame parent, List<CharacterData> characterDatas) {
		super(parent, true);
		if (characterDatas == null) {
			throw new IllegalArgumentException();
		}

		setDefaultCloseOperation(JDialog.DO_NOTHING_ON_CLOSE);
		addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				onClosing();
			}
		});
		
		final Properties strings = LocalizedResourcePropertyLoader
				.getInstance().getLocalizedProperties(STRINGS_RESOURCE);

		setTitle(strings.getProperty("title"));

		JPanel centerPanel = new JPanel();
		GridLayout gridLayout = new GridLayout(2, 1);
		centerPanel.setLayout(gridLayout);
		
		JPanel pnlProfiles = new JPanel(new BorderLayout());
		
		characterListModel = new DefaultListModel();
		for (CharacterData characterData : characterDatas) {
			characterListModel.addElement(characterData);
		}
		
		characterList = new JList(characterListModel);
		characterList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		characterList.setCellRenderer(new DefaultListCellRenderer() {
			private static final long serialVersionUID = 1L;
			@Override
			public Component getListCellRendererComponent(JList list,
					Object value, int index, boolean isSelected,
					boolean cellHasFocus) {
				CharacterData characterData = (CharacterData) value;
				StringBuilder displayNameBuf = new StringBuilder();
				displayNameBuf.append(characterData.getName());
				if (ProfileListManager.isUsingCharacterData(characterData)) {
					displayNameBuf.append(strings.getProperty("profile.opend"));
				}
				if (!characterData.canWrite()) {
					displayNameBuf.append(strings.getProperty("profile.noteditable"));
				}
				return super.getListCellRendererComponent(list, displayNameBuf
						.toString(), index, isSelected, cellHasFocus);
			}
		});
		characterList.addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent e) {
				updateUIState();
			}
		});
		characterList.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				if (e.getClickCount() == 2) {
					// 正確に2回
					onOK();
				}
			}
		});
		
		pnlProfiles.add(characterList, BorderLayout.CENTER);
		
		JPanel pnlProfileEditButtons = new JPanel();
		pnlProfileEditButtons.setBorder(BorderFactory.createEmptyBorder(0, 3, 0, 3));
		GridBagLayout pnlProfileEditButtonsLayout = new GridBagLayout();
		GridBagConstraints gbc = new GridBagConstraints();
		pnlProfileEditButtons.setLayout(pnlProfileEditButtonsLayout);
		
		btnProfileNew = new JButton(new AbstractAction(strings.getProperty("profile.new")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				boolean makeDefault = ((e.getModifiers() & ActionEvent.SHIFT_MASK) != 0);
				onProfileNew(makeDefault);
			}
		});
		btnProfileEdit = new JButton(new AbstractAction(strings.getProperty("profile.edit")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onProfileEdit();
			}
		});
		btnProfileRemove = new JButton(new AbstractAction(strings.getProperty("profile.remove")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onProfileRemove();
			}
		});
		btnProfileBrowse = new JButton(new AbstractAction(strings.getProperty("profile.browse")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onProfileBrowse();
			}
		});
		btnProfileImport = new JButton(new AbstractAction(strings.getProperty("profile.import")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onProfileImport();
			}
		});
		btnProfileExport = new JButton(new AbstractAction(strings.getProperty("profile.export")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onProfileExport();
			}
		});
		
		btnProfileNew.setToolTipText(strings.getProperty("profile.new.tooltip"));
		
		gbc.gridx = 0;
		gbc.gridy = 0;
		gbc.gridwidth = 1;
		gbc.gridheight = 1;
		gbc.weightx = 0.;
		gbc.weighty = 0.;
		gbc.anchor = GridBagConstraints.WEST;
		gbc.fill = GridBagConstraints.BOTH;
		gbc.ipadx = 0;
		gbc.ipady = 0;
		gbc.insets = new Insets(0, 3, 0, 3);
		pnlProfileEditButtons.add(btnProfileNew,gbc);

		gbc.gridx = 0;
		gbc.gridy = 1;
		pnlProfileEditButtons.add(btnProfileEdit, gbc);

		gbc.gridx = 0;
		gbc.gridy = 2;
		pnlProfileEditButtons.add(btnProfileRemove, gbc);

		gbc.gridx = 0;
		gbc.gridy = 3;
		gbc.weighty = 1.;
		pnlProfileEditButtons.add(Box.createGlue(), gbc);

		gbc.gridx = 0;
		gbc.gridy = 4;
		gbc.weighty = 0.;
		pnlProfileEditButtons.add(btnProfileBrowse, gbc);
		
		gbc.gridx = 0;
		gbc.gridy = 5;
		gbc.weighty = 0.;
		pnlProfileEditButtons.add(btnProfileImport, gbc);

		gbc.gridx = 0;
		gbc.gridy = 6;
		pnlProfileEditButtons.add(btnProfileExport, gbc);
		
		JPanel pnlProfilesGroup = new JPanel(new BorderLayout());
		pnlProfilesGroup.setBorder(BorderFactory.createCompoundBorder(BorderFactory
				.createEmptyBorder(3, 3, 3, 3), BorderFactory
				.createTitledBorder(strings.getProperty("profiles"))));
		pnlProfilesGroup.add(pnlProfiles, BorderLayout.CENTER);
		pnlProfilesGroup.add(pnlProfileEditButtons, BorderLayout.EAST);
		

		
		JPanel infoPanel = new JPanel(new GridLayout(1, 2));
		JPanel descriptionPanel = new JPanel(new BorderLayout());
		descriptionPanel.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createEmptyBorder(3, 3, 3, 3), BorderFactory
						.createTitledBorder(strings.getProperty("description"))));
		
		descriptionArea = new JTextArea();
		descriptionArea.setEditable(false);
		descriptionPanel.add(new JScrollPane(descriptionArea), BorderLayout.CENTER);

		// サンプルピクャパネル
		sampleImgPanel = new SamplePicturePanel();
		
		// サンプルピクチャファイルのドラッグアンドドロップ
		new DropTarget(sampleImgPanel, new FileDropTarget() {
			@Override
			protected void onDropFiles(final List<File> dropFiles) {
				if (dropFiles == null || dropFiles.isEmpty()) {
					return;
				}
				// インポートダイアログを開く.
				// ドロップソースの処理がブロッキングしないように、
				// ドロップハンドラの処理を終了して呼び出す.
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						onDrop(dropFiles);
					}
				});
			}
			@Override
			protected void onException(Exception ex) {
				ErrorMessageHelper.showErrorDialog(ProfileSelectorDialog.this, ex);
			}
		});
		
		// サンプルピクチャのコンテキストメニュー
		final JPopupMenu popupMenu = new JPopupMenu();
		final JMenuItem popupMenuCut = popupMenu.add(new AbstractAction(strings.getProperty("samplepicture.cut")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onSamplePictureCut();
			}
		});
		final JMenuItem popupMenuPaste = popupMenu.add(new AbstractAction(strings.getProperty("samplepicture.paste")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onSamplePicturePaste();
			}
		});
		sampleImgPanel.addMouseListener(new MouseAdapter() {
			@Override
			public void mousePressed(MouseEvent e) {
				evaluatePopup(e);
			}
			@Override
			public void mouseReleased(MouseEvent e) {
				evaluatePopup(e);
			}
			private void evaluatePopup(MouseEvent e) {
				if (e.isPopupTrigger()) {
					popupMenuCut.setEnabled(sampleImgPanel.getSamplePictrue() != null && canWriteSamplePicture);
					popupMenuPaste.setEnabled(canWriteSamplePicture && ClipboardUtil.hasImage());
					popupMenu.show(sampleImgPanel, e.getX(), e.getY());
				}
			}
		});

		JScrollPane sampleImgPanelSp = new JScrollPane(sampleImgPanel);
		sampleImgPanelSp.setBorder(null);
		JPanel sampleImgTitledPanel = new JPanel(new BorderLayout());
		sampleImgTitledPanel.add(sampleImgPanelSp, BorderLayout.CENTER);
		sampleImgTitledPanel.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createEmptyBorder(3, 3, 3, 3), BorderFactory
						.createTitledBorder(strings.getProperty("sample-image"))));
		
		infoPanel.add(descriptionPanel);
		infoPanel.add(sampleImgTitledPanel);
		
		centerPanel.add(pnlProfilesGroup);
		centerPanel.add(infoPanel);

		
		Container contentPane = getContentPane();
		contentPane.setLayout(new BorderLayout());
		contentPane.add(centerPanel, BorderLayout.CENTER);
		
		// OK/CANCEL ボタンパネル

		JPanel btnPanel = new JPanel();
		BoxLayout boxLayout = new BoxLayout(btnPanel, BoxLayout.LINE_AXIS);
		btnPanel.setLayout(boxLayout);
		btnPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 42));
		
		AbstractAction actOK = new AbstractAction(strings.getProperty("btn.select")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onOK();
			}
		};
		AbstractAction actCancel = new AbstractAction(strings.getProperty("btn.cancel")) {
			private static final long serialVersionUID = 1L;
			public void actionPerformed(ActionEvent e) {
				onCancel();
			}
		};
		
		btnOK = new JButton(actOK);
		JButton btnCancel = new JButton(actCancel);
		if (Main.isLinuxOrMacOSX()) {
			btnPanel.add(btnCancel);
			btnPanel.add(btnOK);
		} else {
			btnPanel.add(btnOK);
			btnPanel.add(btnCancel);
		}
		
		JPanel btnPanel2 = new JPanel(new BorderLayout());
		btnPanel2.add(btnPanel, BorderLayout.EAST);
		
		contentPane.add(btnPanel2, BorderLayout.SOUTH);
		
		Toolkit tk = Toolkit.getDefaultToolkit();
		JRootPane rootPane = getRootPane();
		rootPane.setDefaultButton(btnOK);
		
		InputMap im = rootPane.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
		ActionMap am = rootPane.getActionMap();
		im.put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), "closeProfileSelectorDialog");
		im.put(KeyStroke.getKeyStroke(KeyEvent.VK_W, tk.getMenuShortcutKeyMask()), "closeProfileSelectorDialog");
		am.put("closeProfileSelectorDialog", actCancel);
		
		pack();

		setSize(500, 500);
		setLocationRelativeTo(parent);
		
		characterList.requestFocus();
		updateUIState();
	}
	
	public CharacterData getSelectedCharacterData() {
		return selectedCharacterData;
	}
	
	/**
	 * キャラクターデータの選択変更に伴い、ボタンやサンプルピクチャなどを切り替える.
	 */
	protected void updateUIState() {
		CharacterData characterData = (CharacterData) characterList.getSelectedValue();
		
		final Properties strings = LocalizedResourcePropertyLoader
				.getInstance().getLocalizedProperties(STRINGS_RESOURCE);

		boolean selected = (characterData != null);
		boolean enableEdit = (characterData != null)
				&& characterData.canWrite();

		btnOK.setEnabled(selected);
	
		btnProfileNew.setEnabled(true);
		btnProfileEdit.setEnabled(selected);
		btnProfileRemove.setEnabled(selected && enableEdit);
		btnProfileImport.setEnabled(true);
		btnProfileExport.setEnabled(selected);
		btnProfileBrowse.setEnabled(selected);
		
		if (enableEdit) {
			btnProfileEdit.setText(strings.getProperty("profile.edit"));
		} else {
			btnProfileEdit.setText(strings.getProperty("profile.edit.readonly"));
		}
		
		// 有効なキャラクターデータであり、且つ、書き込み可能であり、且つ、使用中でなければ削除可能
		boolean removable = characterData != null && characterData.isValid()
				&& !ProfileListManager.isUsingCharacterData(characterData)
				&& characterData.canWrite();

		btnProfileRemove.setEnabled(removable);

		boolean canWriteSamplePicture = false;
		BufferedImage sampleImage = null;

		if (characterData != null && characterData.isValid()) {
			// description
			StringWriter sw = new StringWriter();
			PrintWriter descriptionBuf = new PrintWriter(sw);
			URI docBase = characterData.getDocBase();
			String author = characterData.getAuthor();
			String description = characterData.getDescription();
			if (docBase != null) {
				descriptionBuf.println("configuration: " + docBase);
			}
			if (author != null && author.length() > 0) {
				descriptionBuf.println("author: " + author);
			}
			Dimension imageSize = characterData.getImageSize();
			if (imageSize != null) {
				descriptionBuf.println("size: "
						+ imageSize.width + "x"	+ imageSize.height);
			}
			if (description != null) {
				description = description.replace("\r\n", "\n");
				description = description.replace("\r", "\n");
				description = description.replace("\n", System.getProperty("line.separator"));
				descriptionBuf.println(description);
			}
			descriptionArea.setText(sw.toString());
			descriptionArea.setSelectionStart(0);
			descriptionArea.setSelectionEnd(0);
			
			// sample picture
			try {
				CharacterDataPersistent persist = CharacterDataPersistent.getInstance();
				sampleImage = persist.loadSamplePicture(characterData, imageLoader);
				canWriteSamplePicture = persist.canSaveSamplePicture(characterData);

			} catch (Exception ex) {
				// サンプルピクチャの読み込みに失敗したら、サンプルピクチャを表示しないだけで処理は継続する.
				logger.log(Level.WARNING, "sample picture loading failed. " + characterData , ex);
				sampleImage = null;
			}
		}
		
		this.canWriteSamplePicture = canWriteSamplePicture;

		String dropHere = strings.getProperty("dropHere");
		String noPicture = strings.getProperty("nopicture");
		sampleImgPanel.setSamplePicture(sampleImage);
		sampleImgPanel.setAlternateText(canWriteSamplePicture ? dropHere : noPicture);
	}
	

	/**
	 * サンプルピクチャのファイルを削除し、表示されている画像をクリップボードに保存する
	 */
	protected void onSamplePictureCut() {
		CharacterData characterData = (CharacterData) characterList.getSelectedValue();
		BufferedImage img = sampleImgPanel.getSamplePictrue();

		Toolkit tk = Toolkit.getDefaultToolkit();
		if (characterData == null || !characterData.isValid() || !canWriteSamplePicture || img == null) {
			tk.beep();
			return;
		}

		try {
			// クリップボードにイメージをセット
			Color bgColor = AppConfig.getInstance().getSampleImageBgColor();
			ClipboardUtil.setImage(img, bgColor);
	
			// サンプルピクチャを削除
			CharacterDataPersistent persist = CharacterDataPersistent.getInstance();
			persist.saveSamplePicture(characterData, null);
			
			// プレビューを更新
			sampleImgPanel.setSamplePicture(null);

		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
		}
	}
	
	/**
	 * サンプルピクチャをクリップボードから貼り付け、それをファイルに保存する
	 */
	protected void onSamplePicturePaste() {
		CharacterData characterData = (CharacterData) characterList.getSelectedValue();
		Toolkit tk = Toolkit.getDefaultToolkit();
		if (characterData == null || !characterData.isValid() || !canWriteSamplePicture) {
			tk.beep();
			return;
		}

		try {
			BufferedImage img = ClipboardUtil.getImage();
			if (img != null) {
				// 画像が読み込まれた場合、それを保存する.
				CharacterDataPersistent persist = CharacterDataPersistent.getInstance();
				persist.saveSamplePicture(characterData, img);
	
				sampleImgPanel.setSamplePicture(img);
	
			} else {
				// サンプルピクチャは更新されなかった。
				tk.beep();
			}

		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
		}
	}
	
	/**
	 * サンプルピクチャパネルにドロップされた画像ファイルをサンプルピクチャとしてコピーします.<br>
	 * @param dtde ドロップイベント
	 */
	protected void onDrop(Collection<File> dropFiles) {
		Toolkit tk = Toolkit.getDefaultToolkit();
		CharacterData characterData = (CharacterData) characterList.getSelectedValue();
		if (dropFiles == null || dropFiles.isEmpty()
				|| !canWriteSamplePicture || characterData == null
				|| !characterData.isValid() || !canWriteSamplePicture) {
			tk.beep();
			return;
		}

		try {
			// 最初のファィルのみ取得する.
			File dropFile = dropFiles.iterator().next();

			// ドロップファイルがあれば、イメージとして読み込む
			BufferedImage img = null;
			if (dropFile != null && dropFile.isFile() && dropFile.canRead()) {
				try {
					LoadedImage loadedImage = imageLoader.load(new FileImageResource(dropFile)); 
					img = loadedImage.getImage();
					
				} catch (IOException ex) {
					// イメージのロードができない = 形式が不正であるなどの場合は
					// 読み込みせず、継続する.
					img = null;
				}
			}
			if (img != null) {
				// 画像が読み込まれた場合、それを保存する.
				CharacterDataPersistent persist = CharacterDataPersistent.getInstance();
				persist.saveSamplePicture(characterData, img);

				sampleImgPanel.setSamplePicture(img);

			} else {
				// サンプルピクチャは更新されなかった。
				tk.beep();
			}
			
		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
		}
	}
	
	/**
	 * 閉じる場合
	 */
	protected void onClosing() {
		dispose();
	}
	
	/**
	 * OKボタン押下
	 */
	protected void onOK() {
		selectedCharacterData = (CharacterData) characterList.getSelectedValue();
		if (selectedCharacterData == null) {
			Toolkit tk = Toolkit.getDefaultToolkit();
			tk.beep();
			return;
		}
		dispose();
	}
	
	/**
	 * キャンセルボタン押下
	 */
	protected void onCancel() {
		selectedCharacterData = null;
		onClosing();
	}
	
	/**
	 * プロファイルの作成
	 * @param makeDefault デフォルトのプロファイルで作成する場合
	 */
	protected void onProfileNew(boolean makeDefault) {
		CharacterDataPersistent persist = CharacterDataPersistent.getInstance();

		CharacterData cd = (CharacterData) characterList.getSelectedValue();

		if (makeDefault || cd == null) {
			// デフォルトのプロファイルを使用します.
			cd = persist.createDefaultCharacterData();
		}
		
		// 基本情報をコピーします。
		CharacterData newCd = cd.duplicateBasicInfo();
		// DocBaseはnullにする。これにより新規作成と判断される.
		newCd.setDocBase(null);

		// 新規なのでパーツセット情報はリセットする
		newCd.clearPartsSets(false);

		ProfileEditDialog editDlg = new ProfileEditDialog(this, newCd);
		editDlg.setVisible(true);
		
		newCd = editDlg.getResult();
		if (newCd == null) {
			// キャンセルされた場合
			return;
		}
		
		// 新規プロファイルを保存する.
		try {
			persist.createProfile(newCd);
			persist.saveFavorites(newCd);

		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
			return;
		}

		// 作成されたプロファイルを一覧に追加する.
		characterListModel.addElement(newCd);
	}

	/**
	 * プロファィルの編集
	 */
	protected void onProfileEdit() {
		CharacterData cd = (CharacterData) characterList.getSelectedValue();
		int rowIndex = characterList.getSelectedIndex();
		if (cd == null || !cd.isValid() || rowIndex < 0) {
			return;
		}

		try {
			// プロファイル編集ダイアログを開き、その結果を取得する.
			CharacterData newCd = ProfileListManager.editProfile(this, cd);
			if (newCd == null) {
				// キャンセルされた場合
				return;
			}

			// 現在開いているメインフレームに対してキャラクター定義が変更されたことを通知する.
			MainFrame.notifyChangeCharacterData(cd, newCd, this);

			// プロファイル一覧画面も更新する.
			characterListModel.set(rowIndex, newCd);
			characterList.repaint();

		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
			return;
		}
	}

	/**
	 * プロファイルの削除
	 */
	protected void onProfileRemove() {
		CharacterData cd = (CharacterData) characterList.getSelectedValue();
		if (cd == null || !cd.isValid() || ProfileListManager.isUsingCharacterData(cd) || !cd.canWrite()) {
			// 無効なキャラクター定義であるか、使用中であるか、書き込み不可であれば削除は実行できない.
			Toolkit tk = Toolkit.getDefaultToolkit();
			tk.beep();
			return;
		}
		
		final Properties strings = LocalizedResourcePropertyLoader
				.getInstance().getLocalizedProperties(STRINGS_RESOURCE);

		String msgTempl = strings.getProperty("profile.remove.confirm");
		MessageFormat fmt = new MessageFormat(msgTempl);
		String msg = fmt.format(new Object[]{cd.getName()});
		
		JPanel msgPanel = new JPanel(new BorderLayout(5, 5));
		msgPanel.add(new JLabel(msg), BorderLayout.CENTER);
		JCheckBox chkRemoveForce = new JCheckBox(strings.getProperty("profile.remove.force"));
		msgPanel.add(chkRemoveForce, BorderLayout.SOUTH);

		JOptionPane optionPane = new JOptionPane(msgPanel, JOptionPane.QUESTION_MESSAGE, JOptionPane.YES_NO_OPTION) {
			private static final long serialVersionUID = 1L;
			@Override
			public void selectInitialValue() {
				String noBtnCaption = UIManager.getString("OptionPane.noButtonText");
				for (JButton btn : UIHelper.getInstance().getDescendantOfClass(JButton.class, this)) {
					if (btn.getText().equals(noBtnCaption)) {
						// 「いいえ」ボタンにフォーカスを設定
						btn.requestFocus();
					}
				}
			}
		};
		JDialog dlg = optionPane.createDialog(this, strings.getProperty("confirm.remove"));
		dlg.setVisible(true);
		Object ret = optionPane.getValue();
		if (ret == null || ((Number) ret).intValue() != JOptionPane.YES_OPTION) {
			return;
		}
		
		if (!cd.canWrite() || cd.getDocBase() == null) {
			JOptionPane.showMessageDialog(this, strings.getProperty("profile.remove.cannot"));
			return;
		}
		
		boolean forceRemove = chkRemoveForce.isSelected();
		
		try {
			CharacterDataPersistent persiste = CharacterDataPersistent.getInstance();
			persiste.remove(cd, forceRemove);

		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
			return;
		}
	
		// モデルから該当キャラクターを削除して再描画
		characterListModel.removeElement(cd);
		characterList.repaint();
		updateUIState();
	}
	
	/**
	 * 場所を開く
	 */
	protected void onProfileBrowse() {
		CharacterData cd = (CharacterData) characterList.getSelectedValue();
		if (cd == null || !cd.isValid()) {
			Toolkit tk = Toolkit.getDefaultToolkit();
			tk.beep();
			return;
		}
		try {
			URI docBase = cd.getDocBase();
			if (!DesktopUtilities.browseBaseDir(docBase)) {
				JOptionPane.showMessageDialog(this, docBase);
			}
		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
		}
	}
	
	/**
	 * インポート
	 */
	protected void onProfileImport() {
		try {
			CharacterData selCd = (CharacterData) characterList.getSelectedValue();

			// 選択したプロファイルを更新するか、新規にプロファイルを作成するか選択できるようにする
			if (selCd != null) {
				final Properties strings = LocalizedResourcePropertyLoader
					.getInstance().getLocalizedProperties(STRINGS_RESOURCE);

				JPanel radioPanel = new JPanel(new BorderLayout());
				JRadioButton btnUpdate = new JRadioButton(strings.getProperty("importToUpdateProfile"));
				JRadioButton btnNew = new JRadioButton(strings.getProperty("importToCreateProfile"));
				ButtonGroup radios = new ButtonGroup();
				radios.add(btnUpdate);
				radios.add(btnNew);
				btnUpdate.setSelected(true);
				radioPanel.add(btnUpdate, BorderLayout.NORTH);
				radioPanel.add(btnNew, BorderLayout.SOUTH);
				
				int ret = JOptionPane.showConfirmDialog(this, radioPanel,
						strings.getProperty("confirmUpdateProfile"),
						JOptionPane.OK_CANCEL_OPTION);
				if (ret != JOptionPane.OK_OPTION) {
					return;
				}
			
				if (btnNew.isSelected()) {
					// 選択されていないことにする.
					selCd = null;
				}
			}
			
			// キャラクターデータをロードし直す.
			CharacterData cd;
			if (selCd != null) {
				cd = selCd.duplicateBasicInfo();
				try {
					ProfileListManager.loadCharacterData(cd);
					ProfileListManager.loadFavorites(cd);
					
				} catch (IOException ex) {
					ErrorMessageHelper.showErrorDialog(this, ex);
					// 継続する.
				}
			} else {
				cd = null;
			}
			
			// ディレクトリ監視エージェントの停止
			PartsImageDirectoryWatchAgentFactory agentFactory = PartsImageDirectoryWatchAgentFactory.getFactory();
			PartsImageDirectoryWatchAgent agent = agentFactory.getAgent(cd);
			agent.suspend();
			try {
				// インポートウィザードの実行
				ImportWizardDialog importWizDialog = new ImportWizardDialog(this, cd);
				importWizDialog.setVisible(true);
				
				CharacterData newCd = importWizDialog.getImportedCharacterData();
				if (importWizDialog.getExitCode() == ImportWizardDialog.EXIT_PROFILE_CREATED) {
	
					// 作成されたプロファイルを一覧に追加する.
					characterListModel.addElement(newCd);
	
				} else if (importWizDialog.getExitCode() == ImportWizardDialog.EXIT_PROFILE_UPDATED) {
	
					// 更新されたプロファイルを通知する
					MainFrame.notifyImportedPartsOrFavorites(cd, newCd, this);
				}

			} finally {
				agent.resume();
			}
			
		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
		}
	}

	/**
	 * エクスポート
	 */
	protected void onProfileExport() {
		CharacterData cd = (CharacterData) characterList.getSelectedValue();
		if (cd == null || !cd.isValid()) {
			Toolkit tk = Toolkit.getDefaultToolkit();
			tk.beep();
			return;
		}
		
		try {
			// コピーした情報に対してパーツデータをロードする.
			final CharacterData newCd = cd.duplicateBasicInfo();
			setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
			try {
				ProfileListManager.loadCharacterData(newCd);

			} finally {
				setCursor(Cursor.getDefaultCursor());
			}

			// エクスポートウィザードを表示
			BufferedImage sampleImage = sampleImgPanel.getSamplePictrue();
			ExportWizardDialog exportWizDialog = new ExportWizardDialog(this, newCd, sampleImage);
			exportWizDialog.setVisible(true);

		} catch (Exception ex) {
			ErrorMessageHelper.showErrorDialog(this, ex);
			return;
		}
	}
}
