package charactermanaj.graphics.filters;

import java.awt.Color;

/**
 * 色変換フィルタ.<br>
 * @author seraphy
 */
public class ColorConvertFilter extends AbstractFilter {

	/**
	 * 色置換用インターフェイス.<br>
	 * @author seraphy
	 */
	public interface ColorReplace {

		/**
		 * R,G,Bの順に格納されている色データに対して編集を行う.<br> 
		 * @param rgb 編集対象
		 */
		void convert(int[] rgb);
	}

	/**
	 * 色置換オブジェクト
	 */
	private final ColorReplace colorReplace;
	
	/**
	 * HSB値の各オフセット(3要素)
	 */
	private final float[] hsbOffsets;
	
	/**
	 * 淡色化率.<br>
	 * 0に近づくほどグレースケールに近づく.<br>
	 * 0で完全なグレースケール、1の場合は色の変更はなし。
	 */
	private final float grayLevel;

	/**
	 * ガンマ値補正用テーブル.<br>
	 */
	private final int[][] gammaTbl;
	
	/**
	 * 色変換フィルタを構築する.<br>
	 * @param colorReplace 色置換オブジェクト、不要ならばnull
	 * @param hsbOffsets HSBオフセット(3要素)、不要ならばnull
	 * @param grayLevel 淡色化率、1でそのまま、0でグレースケール化。
	 * @param gammaTableFactory ガンマ補正値ファクトリ、不要ならばnull
	 */
	public ColorConvertFilter(ColorReplace colorReplace, float[] hsbOffsets, float grayLevel, GammaTableFactory gammaTableFactory) {
		if (gammaTableFactory == null) {
			gammaTableFactory = new GammaTableFactory(1.f);
		}
		if (hsbOffsets != null && hsbOffsets.length < 3) {
			throw new IllegalArgumentException("hsbOffset too short.");
		}
		if (hsbOffsets != null) {
			if (hsbOffsets[0] == 0 && hsbOffsets[1] == 0 && hsbOffsets[2] == 0) {
				hsbOffsets = null;
			} else {
				hsbOffsets = (float[]) hsbOffsets.clone();
			}
		}
		if (grayLevel < 0) {
			grayLevel = 0;
		} else if (grayLevel > 1) {
			grayLevel = 1.f;
		}
		this.grayLevel = grayLevel;
		this.gammaTbl = gammaTableFactory.createGammaTable();
		this.hsbOffsets = hsbOffsets;
		this.colorReplace = colorReplace;
	}
	
	/**
	 * ピクセルデータに対して色変換を行う.<br>
	 * @param pixcels ピクセルデータ
	 */
	protected void filter(int[] pixcels) {
		final float grayLevel = this.grayLevel;
		final float negGrayLevel = 1.f - grayLevel;

		// グレースケール変換テーブル
		int[] precalc = new int[256];
		int[] negPrecalc = new int[256];
		for (int i = 0; i < 256; i++) {
			precalc[i] = (int)(i * grayLevel) & 0xff;
			negPrecalc[i] = (int)(i * negGrayLevel) & 0xff;
		}

		// 全ピクセルに対して計算を行う
		final ColorReplace colorReplace = this.colorReplace;
		int[] rgbvals = new int[3];
		final float[] hsbOffsets = this.hsbOffsets;
		final float[] hsbvals = new float[3];
		final int[][] gammaTbl = this.gammaTbl;
		final int mx = pixcels.length;
		for (int i = 0; i < mx; i++) {
			int argb = pixcels[i];
			
			// ガンマ変換
			int a = gammaTbl[0][(argb >> 24) & 0xff];
			int r = gammaTbl[1][(argb >> 16) & 0xff];
			int g = gammaTbl[2][(argb >> 8) & 0xff];
			int b = gammaTbl[3][(argb) & 0xff];
			
			// 色交換
			if (colorReplace != null) {
				rgbvals[0] = r;
				rgbvals[1] = g;
				rgbvals[2] = b;

				colorReplace.convert(rgbvals);
				
				r = rgbvals[0];
				g = rgbvals[1];
				b = rgbvals[2];
			}
			
			// 輝度
			int br = ((77 * r + 150 * g + 29 * b) >> 8) & 0xff;

			// 輝度(グレースケール)に近づける
			r = ((int)(precalc[r] + negPrecalc[br])) & 0xff;
			g = ((int)(precalc[g] + negPrecalc[br])) & 0xff;
			b = ((int)(precalc[b] + negPrecalc[br])) & 0xff;
			
			// 色調変換
			if (hsbOffsets != null) {
				Color.RGBtoHSB(r, g, b, hsbvals);
				for (int l = 0; l < 3; l++) {
					hsbvals[l] += hsbOffsets[l];
				}
				for (int l = 1; l < 3; l++) {
					if (hsbvals[l] < 0) {
						hsbvals[l] = 0;
					} else if (hsbvals[l] > 1.f) {
						hsbvals[l] = 1.f;
					}
				}
				int rgb = Color.HSBtoRGB(hsbvals[0], hsbvals[1], hsbvals[2]);
				argb = (a << 24) | (rgb & 0xffffff);

			} else {
				argb = (a << 24) | (r << 16) | (g << 8) | b;
			}
			pixcels[i] = argb;
		}
	}
}
