package charactermanaj.model.io;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import charactermanaj.model.AppConfig;
import charactermanaj.model.CharacterData;
import charactermanaj.model.RecentData;
import charactermanaj.util.DirectoryConfig;
import charactermanaj.util.UserData;
import charactermanaj.util.UserDataFactory;

/**
 * 最後に使用したデータの使用状況を保存・復元するためのクラス
 * @author seraphy
 */
public final class RecentDataPersistent {

	/**
	 * ロガー
	 */
	private static final Logger logger = Logger.getLogger(RecentDataPersistent.class.getName());

	/**
	 * シングルトン
	 */
	private static final RecentDataPersistent inst = new RecentDataPersistent();
	
	/**
	 * 最後に使用したプロファイルのコートベースを保存するファイル名
	 */
	private static final String RECENT_CHARACTER_SER = "recent-character.ser";
	

	/**
	 * プライベートコンストラクタ
	 */
	private RecentDataPersistent() {
		super();
	}
	
	/**
	 * インスタンスを取得する
	 * @return インスタンス
	 */
	public static RecentDataPersistent getInstance() {
		return inst;
	}
	
	/**
	 * 最後に使用したキャラクターデータとして記録する.
	 * @param characterData
	 * @throws IOException 保存できなった場合
	 */
	public void saveRecent(CharacterData characterData) throws IOException {
		if (characterData == null) {
			throw new IllegalArgumentException();
		}
		if (!characterData.isValid()) {
			return;
		}
		
		AppConfig appConfig = AppConfig.getInstance();
		
		RecentData recentData = new RecentData();
		recentData.setAppVersion(appConfig.getSpecificationVersion());
		recentData.setDocBase(characterData.getDocBase());
		
		UserData recentCharacterStore = getRecentCharacterStore();

		// 他のキャラクターディレクトリ上のデータを取得しマージする.
		Map<File, RecentData> recentDataMap = new HashMap<File, RecentData>();
		try {
			if (recentCharacterStore.exists()) {
				Object rawRecentData = recentCharacterStore.load();
				if (rawRecentData instanceof Map) {
					@SuppressWarnings("unchecked")
					Map<File, RecentData> prevRecentDataMap = (Map<File, RecentData>) rawRecentData;
					for (Map.Entry<File, RecentData> entry : prevRecentDataMap.entrySet()) {
						File dir = entry.getKey();
						if (dir.exists() && dir.isDirectory()) {
							// 現在も有効なものだけを保存する.
							// (すでに存在しなくなっているものは除外する.)
							RecentData prevRecentData = entry.getValue();
							recentDataMap.put(dir, prevRecentData);
						}
					}
				}
			}

		} catch (Exception ex) {
			logger.log(Level.WARNING, "old recentDataFile load failed.", ex);
			// 古い情報が取得できない場合でも最新のものだけは保存できるようにしておく。
		}

		// 保存する.
		File currentCharactersDir = DirectoryConfig.getInstance().getCharactersDir();
		recentDataMap.put(currentCharactersDir, recentData);
		recentCharacterStore.save(recentDataMap);
	}
	
	/**
	 * 最後に使用したキャラクターデータを取得する.
	 * @return キャラクターデータ。最後に使用したデータが存在しない場合はnull
	 * @throws IOException 読み込みに失敗した場合
	 */
	public CharacterData loadRecent() throws IOException {
		UserData recentCharacterStore = getRecentCharacterStore();
		if (!recentCharacterStore.exists()) {
			return null;
		}

		RecentData recentData;
		try {
			File currentCharactersDir = DirectoryConfig.getInstance().getCharactersDir();

			Object rawRecentData = recentCharacterStore.load();
			if (rawRecentData instanceof RecentData) {
				// 旧形式 (単一)
				recentData = (RecentData) rawRecentData;
				logger.log(Level.INFO, "old-recentdata-type: " + recentData);
				
				// 旧形式で保存されているURIが、現在選択しているキャラクターデータと同じ親ディレクトリ
				// でなければ復元しない.
				URI uri = recentData.getDocBase();
				File parentDir = new File(uri).getParentFile().getParentFile(); // 2段上
				if (!currentCharactersDir.equals(parentDir)) {
					logger.log(Level.INFO,
							"unmatched characters-dir. current="
									+ currentCharactersDir + "/recent="
									+ parentDir);
					recentData = null;
				}

			} else if (rawRecentData instanceof Map) {
				// 新形式 (複数のキャラクターディレクトリに対応)
				@SuppressWarnings("unchecked")
				Map<File, RecentData> recentDataMap = (Map<File, RecentData>) rawRecentData;
				recentData = recentDataMap.get(currentCharactersDir);
				logger.log(Level.INFO, "recent-data: " + currentCharactersDir + "=" + recentData);

			} else {
				// 不明な形式
				logger.log(Level.SEVERE,
						"invalid file format. " + recentCharacterStore
								+ "/class=" + rawRecentData.getClass());
				recentData = null;
			}

		} catch (Exception ex) {
			// RecentData情報の復元に失敗した場合は最後に使用したデータが存在しないものとみなす.
			logger.log(Level.WARNING, "recent data loading failed. " + recentCharacterStore, ex);
			recentData = null;
		}

		if (recentData != null) {
			CharacterDataPersistent persist = CharacterDataPersistent.getInstance();
			return persist.loadProfile(recentData.getDocBase());
		}

		// 履歴がない場合、もしくは読み取れなかった場合はnullを返す.
		return null;
	}
	
	/**
	 * 最後に使用したキャラクタデータの保存先を取得する
	 * @return 保存先
	 */
	protected UserData getRecentCharacterStore() {
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		UserData recentCharacterStore = userDataFactory.getUserData(RECENT_CHARACTER_SER);
		return recentCharacterStore;
	}
	
	
}
