package pcc.chemicraft.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;

import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import pcc.chemicraft.ChemiCraftData;
import pcc.chemicraft.core.system.ChemiCraftCraftingManager;
import pcc.chemicraft.util.ChemicalNBTRecipe;
import pcc.chemicraft.util.Formula;
import pcc.chemicraft.util.ICompoundHandler;
import pcc.chemicraft.util.ListHash;
import pcc.chemicraft.util.MaterialRecipe;


/**
 * ChemiCraftのAPI
 * 基本的にAddonはこのクラスのインスタンスを使う
 * @author mozipi
 *
 */
public class ChemiCraftAPI {

	/**
	 * Instance of the ChemiCraftAPI.
	 */
	private static ChemiCraftAPI instance = new ChemiCraftAPI();


	public static ChemiCraftAPI instance(){
		return instance;
	}


	/**
	 * 化合台の原子の数のリスト
	 */
	private ArrayList<Integer[]> chemicalCombinationAmounts = new ArrayList<Integer[]>();



	/**
	 * 化合台の原子の種類のリスト
	 */
	private ArrayList<String[]> chemicalCombinationAtoms = new ArrayList<String[]>();


	/**
	 * 化合台の結果のリスト
	 */
	private ArrayList<ItemStack> chemicalCombinationResult = new ArrayList<ItemStack>();


	/**
	 * ChemiCraftの化学作業台類のレシピのマネージャー
	 */
	private ChemiCraftCraftingManager chemiCraftCraftingManager = new ChemiCraftCraftingManager();


	/**
	 * List of item name of handler to compounds.
	 */
	private ArrayList<String> compoundHandlerItemNames = new ArrayList<String>();



	/**
	 * List of compounds handlers.
	 */
	private ArrayList<ICompoundHandler> compoundHandlers = new ArrayList<ICompoundHandler>();



	/**
	 * 化合物の文字列をダメージ値に変換します。
	 */
	private HashMap<String, Integer> compoundHash = new HashMap<String, Integer>();


	/**
	 * List of compounds names.
	 */
	private ListHash<String, String> compoundsNameListHash = new ListHash<String, String>();


	/**
	 * 電気分解燃料のリスト。
	 */
	private HashMap<ItemStack, Integer> electrolysisFuelList = new HashMap<ItemStack, Integer>();


	/**
	 * 電気分解レシピのリスト
	 */
	private HashMap<ItemStack, ItemStack[]> electrolysisRecipeList = new HashMap<ItemStack, ItemStack[]>();


	/**
	 * 素材製作台のレシピクラス
	 */
	private ArrayList<MaterialRecipe> materialRecipe = new ArrayList<MaterialRecipe>();


	/**
	 * 熱分解燃料のリスト。
	 */
	private HashMap<ItemStack, Integer> pyrolysisFuelList = new HashMap<ItemStack, Integer>();


	/**
	 * 熱分解レシピのリスト
	 */
	private HashMap<ItemStack, ItemStack[]> pyrolysisRecipeList = new HashMap<ItemStack, ItemStack[]>();


	/**
	 * ツール&武器作成台の素材一覧のリスト
	 */
	private ArrayList<ItemStack[]> toolAndWeaponMaterials = new ArrayList<ItemStack[]>();


	/**
	 * ツール&武器作成台の結果のリスト
	 */
	private ArrayList<ItemStack> toolAndWeaponResult = new ArrayList<ItemStack>();



	/**
	 * ツール&武器作成台の不定形であるか
	 */
	private ArrayList<Boolean> toolAndWeaponSharpless = new ArrayList<Boolean>();



	/**
	 * 化合レシピを追加します。materialの要素数は0<= n <= 16にしてください。
	 * @param material 素材
	 * @param result 結果
	 */
	public void addChemicalCombinationRecipe(ItemStack result, Formula formula){
		chemicalCombinationAtoms.add(formula.getAtoms());
		chemicalCombinationAmounts.add(formula.getAmonts());
		chemicalCombinationResult.add(result);
	}



	/**
	 * add compound.
	 * @param name compound name.
	 */
	public void addCompound(String name){
		compoundsNameListHash.add("en_US", name);
		compoundHash.put(name, compoundHash.size());
	}


	/**
	 * add compound corresponding to the language.
	 * @param lang Language to the corresponding
	 * @param englishName compound name
	 * @param langName compound name(specified language)
	 */
	public void addCompound(String lang, String englishName, String langName){
		addCompound(englishName);
		addCompoundLanguage(lang, langName);
	}



	public void addCompoundLanguage(String lang, String langName){
		compoundsNameListHash.add(
				lang,
				langName);
	}



	/**
	 * 電気分解台の燃料を追加します
	 * @param itemstack 燃料のItemStack
	 * @param burnTime 燃焼時間(tick * rate)
	 */
	public void addElectrolysisDecompositionFuel(ItemStack itemstack, int burnTime) {
		this.electrolysisFuelList.put(
				itemstack,
				burnTime);
	}



	public void addElectrolysisDecompositionRecipe(ArrayList<ItemStack> material, Formula formula) {
		for (ItemStack item : material)
        {
			if (item != null){
				addElectrolysisDecompositionRecipe(item, formula);
			}
        }
	}



	public void addElectrolysisDecompositionRecipe(ItemStack material, Formula formula) {
		ItemStack[] itemstacks =
				new ItemStack[formula.getAtoms().length];
		for (int i = 0; i < itemstacks.length; i++) {
			itemstacks[i] =
					new ItemStack(
							ChemiCraftCore.instance.itemAtoms,
							formula.getAmonts()[i],
							ChemiCraftData.toAtoms(formula.getAtoms()[i]));
		}
		this.electrolysisRecipeList.put(
				material,
				itemstacks);
	}



	/**
	 * 素材作成台のレシピを追加します
	 * @param materials 素材
	 * @param result 結果
	 * @param nbtRecipe NBT(Nullの場合はなし)
	 */
	public void addMaterialRecipe(ItemStack[] materials, ItemStack result, ChemicalNBTRecipe nbtRecipe){
		materialRecipe.add(
				new MaterialRecipe(
						result,
						materials,
						nbtRecipe,
						false));
	}



	/**
	 * 素材作成台の不定形レシピを追加します
	 * @param materials 素材
	 * @param result 結果
	 * @param nbtRecipe NBT(Nullの場合はなし)
	 */
	public void addSharplessMaterialRecipe(ItemStack[] materials, ItemStack result, ChemicalNBTRecipe nbtRecipe){
		materialRecipe.add(
				new MaterialRecipe(
						result,
						materials,
						nbtRecipe,
						true));
	}



	/**
	 * 電気分解台の燃料を追加します
	 * @param itemstack 燃料のItemStack
	 * @param burnTime 燃焼時間(tick * rate)
	 */
	public void addPyrolysisDecompositionFuel(ItemStack itemstack, int burnTime) {
		this.pyrolysisFuelList.put(
				itemstack,
				burnTime);
	}



	public void addPyrolysisDecompositionRecipe(ArrayList<ItemStack> material, Formula formula) {
		for (ItemStack item : material)
        {
			if (item != null){
				addPyrolysisDecompositionRecipe(item, formula);
				return;
			}
        }
	}



	/**
	 * 電気分解台のレシピを追加します
	 * @param material 素材
	 * @param integers 原子の元素番号の配列
	 * @param integers2 原子のできる数の配列
	 */
	public void addPyrolysisDecompositionRecipe(ItemStack material, Formula formula) {
		ItemStack[] itemstacks =
				new ItemStack[formula.getAtoms().length];
		for (int i = 0; i < itemstacks.length; i++) {
			itemstacks[i] =
					new ItemStack(
							ChemiCraftCore.instance.itemAtoms,
							formula.getAmonts()[i],
							ChemiCraftData.toAtoms(formula.getAtoms()[i]));
		}
		this.pyrolysisRecipeList.put(
				material,
				itemstacks);
	}



	public void addReversible(ItemStack result, Formula formula){
		addChemicalCombinationRecipe(result, formula);
		addPyrolysisDecompositionRecipe(result, formula);
		addElectrolysisDecompositionRecipe(result, formula);
	}


	public void addReversibleOfElectrolysis(ItemStack result, Formula formula){
		addChemicalCombinationRecipe(result, formula);
		addElectrolysisDecompositionRecipe(result, formula);
	}



	public void addReversibleOfPyrolysis(ItemStack result, Formula formula){
		addChemicalCombinationRecipe(result, formula);
		addPyrolysisDecompositionRecipe(result, formula);
	}



	/**
	 * ツール&武器作成台の不定形レシピを追加します
	 * @param materials 素材
	 * @param result 結果
	 */
	public void addSharplessToolAndWeaponRecipe(ItemStack[] materials, ItemStack result) {
		toolAndWeaponMaterials.add(materials);
		toolAndWeaponResult.add(result);
		toolAndWeaponSharpless.add(true);
	}



	/**
	 * ツール&武器作成台のレシピを追加します
	 * @param materials 素材
	 * @param result 結果
	 */
	public void addToolAndWeaponRecipe(ItemStack[] materials, ItemStack result) {
		toolAndWeaponMaterials.add(materials);
		toolAndWeaponResult.add(result);
		toolAndWeaponSharpless.add(false);
	}



	//以下システム関連//////////////////////////////////////////////////////

	public ArrayList<Integer[]> getChemicalCombinationAmounts(){
		return chemicalCombinationAmounts;
	}



	public ArrayList<String[]> getChemicalCombinationAtoms(){
		return chemicalCombinationAtoms;
	}



	public ArrayList<ItemStack> getChemicalCombinationResult(){
		return chemicalCombinationResult;
	}



	public int getCompound(String key){
		if(compoundHash.get(key) != null){
			return compoundHash.get(key);
		} else {
			return -1;
		}
	}



	public ArrayList<ICompoundHandler> getCompoundHandler(){
		compoundHandlers.trimToSize();
		return compoundHandlers;

	}



	public ArrayList<String> getCompoundHandlerItemName(){
		compoundHandlerItemNames.trimToSize();
		return compoundHandlerItemNames;
	}



	public ListHash<String, String> getCompoundsName(){
		return compoundsNameListHash;
	}



	public ChemiCraftCraftingManager getCraftingManager(){
		return chemiCraftCraftingManager;
	}



	public HashMap<ItemStack, Integer> getElectrolysisFuelList()
	{
		return electrolysisFuelList;
	}



	public HashMap<ItemStack, ItemStack[]> getElectrolysisRecipeList()
	{
		return electrolysisRecipeList;
	}



	public ArrayList<MaterialRecipe> getMaterialRecipe(){
		return materialRecipe;
	}



	public HashMap<ItemStack, Integer> getPyrolysisFuelList()
	{
		return pyrolysisFuelList;
	}



	public HashMap<ItemStack, ItemStack[]> getPyrolysisRecipeList()
	{
		return pyrolysisRecipeList;
	}



	public ArrayList<ItemStack[]> getToolAndWeaponMaterials() {
		return toolAndWeaponMaterials;
	}



	public ArrayList<ItemStack> getToolAndWeaponResult() {
		return toolAndWeaponResult;
	}



	public ArrayList<Boolean> getToolAndWeaponSharpless() {
		return toolAndWeaponSharpless;
	}



	/**
	 * setting compound handler.
	 * @param handlerItemName
	 * @param compoundHandler
	 */
	public void settingCompoundHandler(String handlerItemName, ICompoundHandler compoundHandler){
		compoundHandlers.add(compoundHandler);
		compoundHandlerItemNames.add(handlerItemName);
	}

}
