/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  fsnd_dtq.c
 * @brief %jp{ǡ塼ؤζ}%en{Forced Send to Data Queue}
 *
 * Copyright (C) 1998-2006 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/dtqobj.h"



#if _KERNEL_SPT_FSND_DTQ


/** %jp{ǡ塼ؤζ}%en{Forced Send to Data Queue}
 * @param  dtqid    %jp{оݤΥǡ塼IDֹ}%en{ID number of the data queue to which the data element is sent}
 * @param  data     %jp{ǡ塼ǡ}%en{Data element tobe sent}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(dtqid뤤ϻѤǤʤ)}%en{Invalid ID number(dtqid is invalid or unusable)}
 * @retval E_NOEXS  %jp{֥̤(оݥǡ塼̤Ͽ)}%en{Non-existant object(specified data queue is not registerd)}
 */
ER fsnd_dtq(ID dtqid, VP_INT data)
{
	_KERNEL_T_DTQCB          *dtqcb;
	const _KERNEL_T_DTQCB_RO *dtqcb_ro;
	_KERNEL_T_TSKHDL         tskhdl;
	_KERNEL_T_TCB            *tcb;
	_KERNEL_DTQ_T_DTQCNT     dtqcnt;
	
	/* %jp{ID Υå} */
#if _KERNEL_SPT_FSND_DTQ_E_ID
	if ( !_KERNEL_DTQ_CHECK_DTQID(dtqid) )
	{
		return E_ID;	/* %jp{IDֹ}%en{Invalid ID number} */
	}
#endif

	_KERNEL_ENTER_SVC();	/* %jp{ӥ}%en{enter service-call} */
	
	/* %jp{֥¸ߥå} */
#if _KERNEL_SPT_FSND_DTQ_E_NOEXS
	if ( !_KERNEL_DTQ_CHECK_EXS(dtqid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{ӥ뽪λ} */
		return E_NOEXS;			/* %jp{֥̤} */
	}
#endif
	
	/* %jp{ǡ塼ȥ֥å} */
	dtqcb    = _KERNEL_DTQ_ID2DTQCB(dtqid);
	dtqcb_ro = _KERNEL_DTQ_GET_DTQCB_RO(dtqid, dtqcb);	/* %jp{RO} */

	/* %jp{ǡ塼ΰ̼} */
	dtqcnt = _KERNEL_DTQ_GET_DTQCNT(dtqcb_ro);

#if _KERNEL_SPT_FSND_DTQ_E_ILUSE
	if ( dtqcnt == 0 )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{ӥ뽪λ} */
		return E_ILUSE;			/* %jp{ӥ} */
	}
#endif
	
	/* %jp{ԤƬ饿Ф} */
	tskhdl = _KERNEL_DTQ_RMH_RQUE(dtqcb);
	if ( tskhdl != _KERNEL_TSKHDL_NULL )
	{
		VP_INT *p_data;

		/* %jp{ԤԤ} */
		tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);			/* %jp{顼} */
		p_data = (VP_INT *)_KERNEL_TSK_GET_DATA(tcb);
		*p_data = data;
		_KERNEL_DSP_WUP_TSK(tskhdl);				/* %jp{Ԥ} */
		_KERNEL_DTQ_RMV_RTOQ(tskhdl);
		
		/* %jp{ǥѥåμ¹} */
		_KERNEL_DSP_TSK();
	}
	else
	{
		_KERNEL_DTQ_T_DTQCNT sdtqcnt;
		_KERNEL_DTQ_T_DTQCNT head;
		VP_INT               *dtq;
		
		/* %jp{ǡ塼} */
		sdtqcnt = _KERNEL_DTQ_GET_SDTQCNT(dtqcb);
		head    = _KERNEL_DTQ_GET_HEAD(dtqcb);
		dtq     = _KERNEL_DTQ_GET_DTQ(dtqcb_ro);

		if ( sdtqcnt < dtqcnt )		/* %jp{塼˶Ϥ뤫} */
		{
			/* %jp{ǡ塼ɲ} */
			if ( head < dtqcnt - sdtqcnt - 1 )
			{
				dtq[head + sdtqcnt] = data;
			}
			else
			{
				dtq[head + sdtqcnt - dtqcnt] = data;
			}
			sdtqcnt++;
			_KERNEL_DTQ_SET_SDTQCNT(dtqcb, sdtqcnt);
		}
		else
		{
			/* %jp{ǡ塼˶ɲ} */
			dtq[head] = data;

			/* %jp{Ƭ֤򤺤餹} */
			if ( head + 1 < dtqcnt )
			{
				head++;
			}
			else
			{
				head = 0;
			}
			_KERNEL_DTQ_SET_HEAD(dtqcb, head);
		}		
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{ӥ뤫Ф}%en{leave service-call} */
	
	return E_OK;	/* %jp{ｪλ}%en{Normal completion} */
}


#else	/* _KERNEL_SPT_FSND_DTQ */


#if _KERNEL_SPT_FSND_DTQ_E_NOSPT

/** %jp{ǡ塼ؤζ}%en{Forced Send to Data Queue}
 * @param  dtqid    %jp{оݤΥǡ塼IDֹ}%en{ID number of the data queue to which the data element is sent}
 * @param  data     %jp{ǡ塼ǡ}%en{Data element tobe sent}
 * @retval E_NOSPT  %jp{̤ݡȵǽ}%en{Unsupported function}
 */
ER fsnd_dtq(ID dtqid, VP_INT data)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_FSND_DTQ */



/* end of file */
