/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  trcv_dtq.c
 * @brief %jp{ǡ塼μ(ॢȤ)}%en{Receive from Data Queue(with Timeout)}
 *
 * Copyright (C) 1998-2006 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/dtqobj.h"



#if _KERNEL_SPT_TRCV_DTQ


/** %jp{ǡ塼񸻤γ(ॢȤ)}%en{Acquire Data queue Resource(with Timeout)}
 * @param  dtqid    %jp{񸻳оݤΥǡ塼IDֹ}%en{ID number of the data queue from which resource is acquired}
 * @param  tmout    %jp{ॢȻ}%en{Specified timeout}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(dtqid뤤ϻѤǤʤ)}%en{Invalid ID number(dtqid is invalid or unusable)}
 * @retval E_CTX    %jp{ƥȥ顼}%en{Context error}
 * @retval E_NOEXS  %jp{֥̤(оݥǡ塼̤Ͽ)}%en{Non-existant object(specified data queue is not registerd)}
 * @retval E_RLWAI  %jp{Ԥ֤ζ(Ԥ֤δ֤rel_wai)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_TMOUT  %jp{ॢ}%en{Timeout}
 * @retval E_DLT    %jp{Ԥ֥Ȥκ(Ԥ֤δ֤оݥǡ塼)}%en{Waiting object deleted(data queue is deleted waiting)}
 */
ER trcv_dtq(ID dtqid, VP_INT *p_data, TMO tmout)
{
	return _kernel_rcv_dtq(dtqid, p_data, tmout);
}


#else	/* _KERNEL_SPT_TRCV_DTQ */


#if _KERNEL_SPT_TRCV_DTQ_E_NOSPT

/** %jp{ǡ塼񸻤γ(ॢȤ)}%en{Acquire Data queue Resource(with Timeout)}
 * @param  dtqid    %jp{񸻳оݤΥǡ塼IDֹ}%en{ID number of the data queue from which resource is acquired}
 * @retval E_NOSPT  %jp{̤ݡȵǽ}%en{Unsupported function}
 */
ER trcv_dtq(ID dtqid, VP_INT *p_data, TMO tmout)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_TRCV_DTQ */



#if _KERNEL_SPT_KRCV_DTQ


/** %jp{ǡ塼񸻤γ()}%en{Acquire Data queue Resource}
 * @param  dtqid    %jp{񸻳оݤΥǡ塼IDֹ}%en{ID number of the data queue from which resource is acquired}
 * @param  tmout    %jp{ॢȻ}%en{Specified timeout}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(dtqid뤤ϻѤǤʤ)}%en{Invalid ID number(dtqid is invalid or unusable)}
 * @retval E_CTX    %jp{ƥȥ顼}%en{Context error}
 * @retval E_NOEXS  %jp{֥̤(оݥǡ塼̤Ͽ)}%en{Non-existant object(specified data queue is not registerd)}
 * @retval E_RLWAI  %jp{Ԥ֤ζ(Ԥ֤δ֤rel_wai)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_TMOUT  %jp{ॢ}%en{Timeout}
 * @retval E_DLT    %jp{Ԥ֥Ȥκ(Ԥ֤δ֤оݥǡ塼)}%en{Waiting object deleted(data queue is deleted waiting)}
 */
ER _kernel_rcv_dtq(ID dtqid, VP_INT *p_data, TMO tmout)
{
	_KERNEL_T_DTQCB				*dtqcb;
	const _KERNEL_T_DTQCB_RO	*dtqcb_ro;
	_KERNEL_T_TSKHDL			tskhdl;
	_KERNEL_T_TCB				*tcb;
	_KERNEL_DTQ_T_DTQCNT		sdtqcnt;
	ER							ercd;
	
	/* %jp{ƥȥå} */
#if _KERNEL_SPT_KRCV_DTQ_E_CTX
	if ( tmout != TMO_POL && _KERNEL_SYS_SNS_DPN() )
	{
		return E_CTX;			/* %jp{ƥȥ顼}%en{Context error} */
	}
#endif

	/* %jp{ID Υå} */
#if _KERNEL_SPT_KRCV_DTQ_E_ID
	if ( !_KERNEL_DTQ_CHECK_DTQID(dtqid) )
	{
		return E_ID;	/* %jp{IDֹ}%en{Invalid ID number} */
	}
#endif
		
	_KERNEL_ENTER_SVC();		/* %jp{ӥ}%en{enter service-call} */
	
	/* %jp{֥¸ߥå} */
#if _KERNEL_SPT_KRCV_DTQ_E_NOEXS
	if ( !_KERNEL_DTQ_CHECK_EXS(dtqid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{ӥ뤫Ф}%en{leave service-call} */
		return E_NOEXS;			/* %jp{֥̤}%en{Non-existant object} */
	}
#endif
	
	/* %jp{ǡ塼ȥ֥å} */
	dtqcb = _KERNEL_DTQ_ID2DTQCB(dtqid);
	
	/* %jp{ԤƬ饿Ф} */
	tskhdl = _KERNEL_DTQ_RMH_SQUE(dtqcb);
	if ( tskhdl != _KERNEL_TSKHDL_NULL )
	{
		/* %jp{ԤԤ} */
		tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB} */
		
		/* %jp{򵯤} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);	/* %jp{顼} */
		_KERNEL_DSP_WUP_TSK(tskhdl);		/* %jp{Ԥ} */
		_KERNEL_DTQ_RMV_STOQ(tskhdl);		/* %jp{ॢȥ塼Ϥ} */
	}
	
	
#if _KERNEL_SPT_DTQ_DTQCNT_NONZERO		/* %jp{ǡ塼} */
	sdtqcnt = _KERNEL_DTQ_GET_SDTQCNT(dtqcb);
	if ( sdtqcnt > 0 )		/* %jp{塼˥ǡϤ뤫} */
	{
		_KERNEL_DTQ_T_DTQCNT		dtqcnt;
		VP_INT						*dtq;
		_KERNEL_DTQ_T_DTQCNT		head;
		
		/* %jp{RO} */
		dtqcb_ro = _KERNEL_DTQ_GET_DTQCB_RO(dtqid, dtqcb);
		
		/* %jp{ǡ塼} */
		dtq = _KERNEL_DTQ_GET_DTQ(dtqcb_ro);
		
		/* %jp{塼ǡФ} */
		head   = _KERNEL_DTQ_GET_HEAD(dtqcb);
		dtqcnt = _KERNEL_DTQ_GET_DTQCNT(dtqcb_ro);
		*p_data = dtq[head];
		
		/* %jp{ԤХǡɲ} */
		if ( tskhdl != _KERNEL_TSKHDL_NULL )
		{
			/* %jp{ǡǼ} */
			tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB} */
			dtq[head] = (VP_INT)_KERNEL_TSK_GET_DATA(tcb);
		}
		else
		{
			/* %jp{Ԥ̵Х塼ΥǡĿ򸺤} */
			sdtqcnt--;
			_KERNEL_DTQ_SET_SDTQCNT(dtqcb, sdtqcnt);
		}
		
		/* %jp{Ƭ֤򤺤餹} */
		if ( head + 1 < dtqcnt )
		{
			head++;
		}
		else
		{
			head = 0;
		}
		_KERNEL_DTQ_SET_HEAD(dtqcb, head);

		if ( tskhdl != _KERNEL_TSKHDL_NULL )
		{
			/* %jp{ǥѥåμ¹} */
			_KERNEL_DSP_TSK();
		}
		
		ercd = E_OK;	/* %jp{ｪλ}%en{Normal completion} */
	}
	else
#endif
	{
#if _KERNEL_SPT_DTQ_DTQCNT_ZERO
		if ( tskhdl != _KERNEL_TSKHDL_NULL )
		{
			/* %jp{ǡ} */
			tcb     = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB} */
			*p_data = (VP_INT)_KERNEL_TSK_GET_DATA(tcb);	/* %jp{ǡǼ} */

			/* %jp{ǥѥåμ¹} */
			_KERNEL_DSP_TSK();

			ercd = E_OK;	/* %jp{ｪλ}%en{Normal completion} */
		}
		else
#endif
		{
			if ( tmout == TMO_POL )
			{
				ercd = E_TMOUT;  /* %jp{ॢ}%en{Timeout} */
			}
			else
			{
				/* %jp{RO} */
				dtqcb_ro = _KERNEL_DTQ_GET_DTQCB_RO(dtqid, dtqcb);
				
				/* %jp{¹楿} */
				tskhdl = _KERNEL_SYS_GET_RUNTSK();			/* %jp{ϥɥ} */
				tcb    = _KERNEL_TSK_TSKHDL2TCB(tskhdl);	/* %jp{TCB} */
				
				/* %jp{ǡǼɥ쥹} */
				_KERNEL_TSK_SET_DATA(tcb, (VP_INT)p_data);
				
				/* %jp{Ԥ֤ˤ} */
				_KERNEL_TSK_SET_TSKWAIT(tcb, _KERNEL_TTW_RDTQ);		/* %jp{Ԥװ} */
				_KERNEL_TSK_SET_WOBJID(tcb, dtqid);					/* %jp{Ԥ֥ID} */
				_KERNEL_DSP_WAI_TSK(tskhdl);						/* %jp{Ԥ֤} */
				_KERNEL_DTQ_ADD_RQUE(dtqcb, dtqcb_ro, tskhdl);		/* %jp{Ԥɲ} */

#if _KERNEL_SPT_TRCV_DTQ
				if ( tmout != TMO_FEVR )
				{
					_KERNEL_DTQ_ADD_RTOQ(tskhdl, tmout);				/* %jp{ॢȥ塼ɲ} */
				}
#endif
				
				/* %jp{ǥѥåμ¹} */
				_KERNEL_DSP_TSK();
				
				/* %jp{顼ɤμ} */
				ercd = _KERNEL_TSK_GET_ERCD(tcb);
			}
		}		
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{ӥ뤫Ф}%en{leave service-call} */
	
	return ercd;
}


#endif	/* _KERNEL_SPT_KRCV_DTQ */


/* end of file */
