/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  tsnd_dtq.c
 * @brief %jp{ǡ塼ؤ(ॢȤ)}%en{Send to Data Queue(with Timeout)}
 *
 * Copyright (C) 1998-2009 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/dtqobj.h"



#if _KERNEL_SPT_TSND_DTQ


/** %jp{ǡ塼ؤ(ॢȤ)}%en{Send to Data Queue(with Timeout)}
 * @param  dtqid    %jp{оݤΥǡ塼IDֹ}%en{ID number of the data queue to which the data element is sent}
 * @param  data     %jp{ǡ塼ǡ}%en{Data element to be sent}
 * @param  tmout    %jp{ॢȻ}%en{Specified timeout}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(dtqid뤤ϻѤǤʤ)}%en{Invalid ID number(dtqid is invalid or unusable)}
 * @retval E_CTX    %jp{ƥȥ顼}%en{Context error}
 * @retval E_NOEXS  %jp{֥̤(оݥǡ塼̤Ͽ)}%en{Non-existant object(specified data queue is not registerd)}
 * @retval E_PAR    %jp{ѥ᡼顼(tmout)}%en{Parameter error(tmout is invalid)}
 * @retval E_RLWAI  %jp{Ԥ֤ζ(Ԥ֤δ֤rel_wai)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_TMOUT  %jp{ॢ}%en{Timeout}
 * @retval E_DLT    %jp{Ԥ֥Ȥκ(Ԥ֤δ֤оݥޥե)}%en{Waiting object deleted(semaphore is deleted waiting)}
 */
ER tsnd_dtq(ID dtqid, VP_INT data, TMO tmout)
{
	return _kernel_snd_dtq(dtqid, data, tmout);
}


#else	/* _KERNEL_SPT_TSND_DTQ */


#if _KERNEL_SPT_TSND_DTQ_E_NOSPT

/** %jp{ǡ塼ؤ(ॢȤ)}%en{Send to Data Queue(with Timeout)}
 * @param  dtqid    %jp{оݤΥǡ塼IDֹ}%en{ID number of the data queue to which the data element is sent}
 * @param  data     %jp{ǡ塼ǡ}%en{Data element to be sent}
 * @param  tmout    %jp{ॢȻ}%en{Specified timeout}
 * @retval E_NOSPT  %jp{̤ݡȵǽ}%en{Unsupported function}
 */
ER tsnd_dtq(ID dtqid, VP_INT data, TMO tmout)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_TSND_DTQ */



#if _KERNEL_SPT_KSND_DTQ

ER _kernel_snd_dtq(ID dtqid, VP_INT data, TMO tmout)
{
	_KERNEL_T_DTQCB  *dtqcb;
	_KERNEL_T_TSKHDL tskhdl;
	_KERNEL_T_TCB    *tcb;
	ER               ercd;
	
	/* %jp{ƥȥå} */
#if _KERNEL_SPT_TSND_DTQ_E_CTX
	if ( _KERNEL_SYS_SNS_DPN() )
	{
		return E_CTX;			/* %jp{ƥȥ顼}%en{Context error} */
	}
#endif
	
	/* %jp{ID Υå} */
#if _KERNEL_SPT_TSND_DTQ_E_ID
	if ( !_KERNEL_DTQ_CHECK_DTQID(dtqid) )
	{
		return E_ID;	/* %jp{IDֹ}%en{Invalid ID number} */
	}
#endif

	/* %jp{ѥ᡼Υå} */
#if _KERNEL_SPT_TSND_DTQ_E_PAR
	if ( tmout < 0 && tmout != TMO_FEVR && tmout != TMO_POL)
	{
		return E_PAR;	/* %jp{ѥ᡼顼(tmout)}%en{Parameter error(tmout is invalid)} */
	}
#endif
	
	_KERNEL_ENTER_SVC();	/* %jp{ӥ}%en{enter service-call} */
	
	/* %jp{֥¸ߥå} */
#if _KERNEL_SPT_TSND_DTQ_E_NOEXS
	if ( !_KERNEL_DTQ_CHECK_EXS(dtqid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{ӥ뽪λ} */
		return E_NOEXS;			/* %jp{֥̤} */
	}
#endif
	
	/* %jp{ǡ塼ȥ֥å} */
	dtqcb = _KERNEL_DTQ_ID2DTQCB(dtqid);
	
	/* %jp{ԤƬ饿Ф} */
	tskhdl = _KERNEL_DTQ_RMH_RQUE(dtqcb);
	if ( tskhdl != _KERNEL_TSKHDL_NULL )
	{
		VP_INT *p_data;

		/* %jp{ԤԤ} */
		tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);			/* %jp{顼} */
		p_data = (VP_INT *)_KERNEL_TSK_GET_DATA(tcb);
		*p_data = data;
		_KERNEL_DSP_WUP_TSK(tskhdl);				/* %jp{Ԥ} */
		_KERNEL_DTQ_RMV_RTOQ(tskhdl);
		
		/* %jp{ǥѥåμ¹} */
		_KERNEL_DSP_TSK();
		
		ercd = E_OK;	/* %jp{ｪλ}%en{Normal completion} */
	}
	else
	{
		const _KERNEL_T_DTQCB_RO *dtqcb_ro;
		_KERNEL_DTQ_T_DTQCNT     sdtqcnt;
		_KERNEL_DTQ_T_DTQCNT     dtqcnt;
		
		/* %jp{RO} */
		dtqcb_ro = _KERNEL_DTQ_GET_DTQCB_RO(dtqid, dtqcb);

		/* %jp{ǡ塼󥿼} */
		sdtqcnt = _KERNEL_DTQ_GET_SDTQCNT(dtqcb);
		dtqcnt  = _KERNEL_DTQ_GET_DTQCNT(dtqcb_ro);

		if ( sdtqcnt < dtqcnt )		/* %jp{塼˶Ϥ뤫} */
		{
			_KERNEL_DTQ_T_DTQCNT head;
			VP_INT               *dtq;

			/* %jp{ǡ塼} */
			head = _KERNEL_DTQ_GET_HEAD(dtqcb);
			dtq  = _KERNEL_DTQ_GET_DTQ(dtqcb_ro);
			
			/* %jp{ǡ塼ɲ} */
			if ( head < dtqcnt - sdtqcnt - 1 )
			{
				dtq[head + sdtqcnt] = data;
			}
			else
			{
				dtq[head + sdtqcnt - dtqcnt] = data;
			}
			
			/* %jp{ǡĿû} */
			sdtqcnt++;
			_KERNEL_DTQ_SET_SDTQCNT(dtqcb, sdtqcnt);

			ercd = E_OK;	/* %jp{ｪλ}%en{Normal completion} */
		}
		else
		{
#if _KERNEL_SPT_TSND_DTQ || _KERNEL_SPT_PSND_DTQ
			if ( tmout == TMO_POL )
			{
				ercd = E_TMOUT;  /* %jp{ॢ}%en{Timeout} */
			}
			else
#endif
			{
				/* %jp{Ԥ֤ˤ} */
				tskhdl = _KERNEL_SYS_GET_RUNTSK();
				tcb    = _KERNEL_TSK_TSKHDL2TCB(tskhdl);			/* %jp{TCB} */
				_KERNEL_TSK_SET_TSKWAIT(tcb, _KERNEL_TTW_SDTQ);
				_KERNEL_TSK_SET_WOBJID(tcb, dtqid);
				_KERNEL_TSK_SET_DATA(tcb, data);
				_KERNEL_DSP_WAI_TSK(tskhdl);
				_KERNEL_DTQ_ADD_SQUE(dtqcb, dtqcb_ro, tskhdl);		/* %jp{Ԥɲ} */

#if _KERNEL_SPT_TSND_DTQ
			if ( tmout != TMO_FEVR )
			{
				_KERNEL_DTQ_ADD_STOQ(tskhdl, tmout);				/* %jp{ॢȥ塼ɲ} */
			}
#endif
				/* %jp{ǥѥåμ¹} */
				_KERNEL_DSP_TSK();
				
				/* %jp{顼ɤμ} */
				ercd = _KERNEL_TSK_GET_ERCD(tcb);
			}
		}		
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{ӥ뤫Ф}%en{leave service-call} */
	
	return ercd;
}

#endif	/* _KERNEL_SPT_KSND_DTQ */



/* end of file */
