/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  sig_sem.c
 * @brief %jp{ޥե񸻤ֵ}%en{Release Semaphore Resource}
 *
 * Copyright (C) 1998-2006 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/semobj.h"



#if _KERNEL_SPT_SIG_SEM


/** %jp{ޥե񸻤ֵ}%en{Release Semaphore Resource}
 * @param  semid    %jp{ޥեֵоݤΥޥեIDֹ}%en{ID number of the semaphore to which resource is released}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(semid뤤ϻѤǤʤ)}%en{Invalid ID number(semid is invalid or unusable)}
 * @retval E_NOEXS  %jp{֥̤(оݥޥե̤Ͽ)}%en{Non-existant object(specified semaphore is not registerd)}
 * @retval E_QOVR   %jp{塼󥰥Сե(񸻿ֵۤ)}%en{Queue overflow(release will exceed maximum resource count)}
 */
ER sig_sem(ID semid)
{
	_KERNEL_T_SEMCB_PTR  semcb;
	_KERNEL_T_TSKHDL     tskhdl;
	_KERNEL_T_TCB        *tcb;
	_KERNEL_SEM_T_SEMCNT semcnt;
	
	/* %jp{ID Υå} */
#if _KERNEL_SPT_SIG_SEM_E_ID
	if ( !_KERNEL_SEM_CHECK_SEMID(semid) )
	{
		return E_ID;	/* %jp{IDֹ}%en{Invalid ID number} */
	}
#endif
	
	_KERNEL_ENTER_SVC();	/* %jp{ӥ}%en{enter service-call} */
	
	/* %jp{֥¸ߥå} */
#if _KERNEL_SPT_SIG_SEM_E_NOEXS
	if ( !_KERNEL_SEM_CHECK_EXS(semid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{ӥ뽪λ} */
		return E_NOEXS;			/* %jp{֥̤} */
	}
#endif
	
	/* %jp{ޥեȥ֥å} */
	semcb = _KERNEL_SEM_ID2SEMCB(semid);
	
	/* %jp{ԤƬ饿Ф} */
	tskhdl = _KERNEL_SEM_RMH_QUE(semcb);
	if ( tskhdl != _KERNEL_TSKHDL_NULL )
	{
		/* %jp{ԤԤ} */
		tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);			/* %jp{顼} */
		_KERNEL_DSP_WUP_TSK(tskhdl);				/* %jp{Ԥ} */
		_KERNEL_SEM_RMV_TOQ(tskhdl);
		
		/* %jp{ǥѥåμ¹} */
		_KERNEL_DSP_TSK();
	}
	else
	{
		/* %jp{ޥե󥿼} */
		semcnt = _KERNEL_SEM_GET_SEMCNT(semcb);

		/* %jp{塼󥰥Сեå} */
#if _KERNEL_SPT_SIG_SEM_E_QOVR
		{
			_KERNEL_T_SEMCB_RO_PTR semcb_ro;

			semcb_ro = _KERNEL_SEM_GET_SEMCB_RO(semid, semcb);
			if ( semcnt >= _KERNEL_SEM_GET_MAXSEM(semcb_ro) )
			{
				_KERNEL_LEAVE_SVC();	/* %jp{ӥ뤫Ф}%en{leave service-call} */
				return E_QOVR;			/* %jp{塼󥰥Сե}%en{Queue overflow} */
			}
		}
#endif
		
		/* %jp{ޥեֵ} */
		_KERNEL_SEM_SET_SEMCNT(semcb, semcnt + 1);
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{ӥ뤫Ф}%en{leave service-call} */
	
	return E_OK;	/* %jp{ｪλ}%en{Normal completion} */
}


#else	/* _KERNEL_SPT_SIG_SEM */


#if _KERNEL_SPT_SIG_SEM_E_NOSPT

/** %jp{ޥե񸻤ֵ}%en{Release Semaphore Resource}
 * @param  semid    %jp{ޥեֵоݤΥޥեIDֹ}%en{ID number of the semaphore to which resource is released}
 * @retval E_NOSPT  %jp{̤ݡȵǽ}%en{Unsupported function}
 */
ER sig_sem(ID semid)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_SIG_SEM */



/* end of file */
