/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  twai_sem.c
 * @brief %jp{ޥե񸻤γ(ॢȤ)}%en{Acquire Semaphore Resource(with Timeout)}
 *
 * Copyright (C) 1998-2006 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"
#include "object/semobj.h"



#if _KERNEL_SPT_TWAI_SEM


/** %jp{ޥե񸻤γ(ॢȤ)}%en{Acquire Semaphore Resource(with Timeout)}
 * @param  semid    %jp{񸻳оݤΥޥեIDֹ}%en{ID number of the semaphore from which resource is acquired}
 * @param  tmout    %jp{ॢȻ}%en{Specified timeout}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(semid뤤ϻѤǤʤ)}%en{Invalid ID number(semid is invalid or unusable)}
 * @retval E_CTX    %jp{ƥȥ顼}%en{Context error}
 * @retval E_NOEXS  %jp{֥̤(оݥޥե̤Ͽ)}%en{Non-existant object(specified semaphore is not registerd)}
 * @retval E_PAR    %jp{ѥ᡼顼(tmout)}%en{Parameter error(tmout is invalid)}
 * @retval E_RLWAI  %jp{Ԥ֤ζ(Ԥ֤δ֤rel_wai)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_TMOUT  %jp{ॢ}%en{Timeout}
 * @retval E_DLT    %jp{Ԥ֥Ȥκ(Ԥ֤δ֤оݥޥե)}%en{Waiting object deleted(semaphore is deleted waiting)}
 */
ER twai_sem(ID semid, TMO tmout)
{
	return _kernel_wai_sem(semid, tmout);
}


#else	/* _KERNEL_SPT_TWAI_SEM */


#if _KERNEL_SPT_TWAI_SEM_E_NOSPT

/** %jp{ޥե񸻤γ(ॢȤ)}%en{Acquire Semaphore Resource(with Timeout)}
 * @param  semid    %jp{񸻳оݤΥޥեIDֹ}%en{ID number of the semaphore from which resource is acquired}
 * @retval E_NOSPT  %jp{̤ݡȵǽ}%en{Unsupported function}
 */
ER twai_sem(ID semid, TMO tmout)
{
	return E_NOSPT;
}

#endif


#endif	/* _KERNEL_SPT_TWAI_SEM */



#if _KERNEL_SPT_KWAI_SEM


/** %jp{ޥե񸻤γ()}%en{Acquire Semaphore Resource}
 * @param  semid    %jp{񸻳оݤΥޥեIDֹ}%en{ID number of the semaphore from which resource is acquired}
 * @param  tmout    %jp{ॢȻ}%en{Specified timeout}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(semid뤤ϻѤǤʤ)}%en{Invalid ID number(semid is invalid or unusable)}
 * @retval E_CTX    %jp{ƥȥ顼}%en{Context error}
 * @retval E_NOEXS  %jp{֥̤(оݥޥե̤Ͽ)}%en{Non-existant object(specified semaphore is not registerd)}
 * @retval E_PAR    %jp{ѥ᡼顼(tmout)}%en{Parameter error(tmout is invalid)}
 * @retval E_RLWAI  %jp{Ԥ֤ζ(Ԥ֤δ֤rel_wai)}%en{Forced release from waiting(accept rel_wai while waiting)}
 * @retval E_TMOUT  %jp{ॢ}%en{Timeout}
 * @retval E_DLT    %jp{Ԥ֥Ȥκ(Ԥ֤δ֤оݥޥե)}%en{Waiting object deleted(semaphore is deleted waiting)}
 */
ER _kernel_wai_sem(ID semid, TMO tmout)
{
	_KERNEL_T_SEMCB_PTR  semcb;
	_KERNEL_T_TSKHDL     tskhdl;
	_KERNEL_T_TCB        *tcb;
	_KERNEL_SEM_T_SEMCNT semcnt;
	ER                   ercd;

	/* %jp{ƥȥå} */
#if _KERNEL_SPT_KWAI_SEM_E_CTX
	if ( tmout != TMO_POL && _KERNEL_SYS_SNS_DPN() )
	{
		return E_CTX;			/* %jp{ƥȥ顼}%en{Context error} */
	}
#endif

	/* %jp{ID Υå} */
#if _KERNEL_SPT_KWAI_SEM_E_ID
	if ( !_KERNEL_SEM_CHECK_SEMID(semid) )
	{
		return E_ID;	/* %jp{IDֹ}%en{Invalid ID number} */
	}
#endif

	/* %jp{ѥ᡼Υå} */
#if _KERNEL_SPT_TWAI_SEM_E_PAR
	if ( tmout < 0 && tmout != TMO_FEVR && tmout != TMO_POL)
	{
		return E_PAR;	/* %jp{ѥ᡼顼(tmout)}%en{Parameter error(tmout is invalid)} */
	}
#endif
	
	_KERNEL_ENTER_SVC();		/* %jp{ӥ}%en{enter service-call} */
	
	/* %jp{֥¸ߥå} */
#if _KERNEL_SPT_KWAI_SEM_E_NOEXS
	if ( !_KERNEL_SEM_CHECK_EXS(semid) )
	{
		_KERNEL_LEAVE_SVC();	/* %jp{ӥ뤫Ф}%en{leave service-call} */
		return E_NOEXS;			/* %jp{֥̤}%en{Non-existant object} */
	}
#endif

	/* %jp{ޥեȥ֥å} */
	semcb = _KERNEL_SEM_ID2SEMCB(semid);
	
	/* %jp{ޥե󥿼} */
	semcnt = _KERNEL_SEM_GET_SEMCNT(semcb);
	
	if ( semcnt > 0 )
	{
		/* %jp{ޥե񸻤Ǥ} */
		_KERNEL_SEM_SET_SEMCNT(semcb, semcnt - 1); 		/* %jp{ޥե񸻤γ} */
		ercd = E_OK;
	}
	else
	{
#if _KERNEL_SPT_TWAI_SEM || _KERNEL_SPT_POL_SEM
		if ( tmout == TMO_POL )
		{
			ercd = E_TMOUT;  /* %jp{ॢ}%en{Timeout} */
		}
		else
#endif
		{
			/* %jp{Ԥ֤ˤ} */
			tskhdl = _KERNEL_SYS_GET_RUNTSK();
			tcb    = _KERNEL_TSK_TSKHDL2TCB(tskhdl);		/* %jp{TCB} */
			_KERNEL_TSK_SET_TSKWAIT(tcb, _KERNEL_TTW_SEM);
			_KERNEL_TSK_SET_WOBJID(tcb, semid);
			_KERNEL_DSP_WAI_TSK(tskhdl);
			_KERNEL_SEM_ADD_QUE(semcb, _KERNEL_SEM_GET_SEMCB_RO(semid, semcb), tskhdl);			/* %jp{Ԥɲ} */
			
#if _KERNEL_SPT_TWAI_SEM
			if ( tmout != TMO_FEVR )
			{
				_KERNEL_SEM_ADD_TOQ(tskhdl, tmout);				/* %jp{ॢȥ塼ɲ} */
			}
#endif

			/* %jp{ǥѥåμ¹} */
			_KERNEL_DSP_TSK();

			/* %jp{顼ɤμ} */
			ercd = _KERNEL_TSK_GET_ERCD(tcb);
		}
	}
	
	_KERNEL_LEAVE_SVC();	/* %jp{֥̤}%en{Non-existant object} */
	
	return ercd;
}


#endif	/* _KERNEL_SPT_KWAI_SEM */



/* end of file */
