/** 
 *  Hyper Operating System V4 Advance
 *
 * @file  iwup_tsk.c
 * @brief %jp{ε}%en{Wakeup Task}
 *
 * Copyright (C) 1998-2009 by Project HOS
 * http://sourceforge.jp/projects/hos/
 */



#include "core/core.h"


#if _KERNEL_SPT_IWUP_TSK

#if _KERNEL_SPT_DPC


static void _kernel_dpc_iwup_tsk(ID tskid, VP_INT param);


/** %jp{ε}%en{Wakeup Task}
 * @param  tskid    %jp{оݤΥIDֹ}%en{ID number of the task to be woken up}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(tskid뤤ϻѤǤʤ)}%en{Invalid ID number(tskid is invalid or unusable)}
 * @retval E_NOMEM  %jp{­}%en{Insufficient memory}
 */
ER wup_tsk(
		ID tskid)
{
#if _KERNEL_SPT_IWUP_TSK_E_ID
	if ( tskid == TSK_SELF )		/* %jp{Ѵ} */
	{
		return E_ID;			/* %jp{IDֹ}%en{Invalid ID number} */
	}
	
	if ( !_KERNEL_TSK_CHECK_TSKID(tskid) )
	{
		return E_ID;			/* %jp{IDֹ}%en{Invalid ID number} */
	}

#endif

	/* %jp{ٱ¹׵} */
	return _KERNEL_SYS_REQ_DPC(_kernel_dpc_iwup_tsk, tskid, 0);
}


void _kernel_dpc_iwup_tsk(ID tskid, VP_INT param)
{
	_KERNEL_T_TSKHDL tskhdl;
	_KERNEL_T_TCB    *tcb;
	
		
	/* %jp{֥¸ߥå} */
#if _KERNEL_SPT_IWUP_TSK_E_NOEXS
		if ( !_KERNEL_TSK_CHECK_EXS(tskid) )
		{
			return;					/* %jp{֥̤}%en{Non-existant object} */
		}
#endif
	
	/* %jp{IDֹѴ} */
	tskhdl = _KERNEL_TSK_ID2TSKHDL(tskid);
	
	/* %jp{TCB} */
	tcb = _KERNEL_TSK_TSKHDL2TCB(tskhdl);
	
	if ( ((_KERNEL_TSK_GET_TSKSTAT(tcb) & _KERNEL_TTS_WAI)) && _KERNEL_TSK_GET_TSKWAIT(tcb) == _KERNEL_TTW_SLP )
	{
		/* %jp{Ԥ} */
		_KERNEL_TSK_SET_ERCD(tcb, E_OK);		/* %jp{顼} */
		_KERNEL_DSP_WUP_TSK(tskhdl);			/* %jp{Ԥ} */
	}
	else
	{
#if _KERNEL_TCB_WUPCNT
		{
			_KERNEL_TSK_T_WUPCNT wupcnt;
			
			wupcnt = _KERNEL_TSK_GET_WUPCNT(tcb);

#if _KERNEL_SPT_IWUP_TSK_E_QOVR
			if ( wupcnt >= _KERNEL_TMAX_WUPCNT )
			{
				_KERNEL_LEAVE_SVC();	/* %jp{ӥ뽪λ} */
				return E_QOVR;
			}
#endif
			
			_KERNEL_TSK_SET_WUPCNT(tcb, wupcnt + 1);
		}
#endif
	}

	_KERNEL_LEAVE_SVC();	/* %jp{ӥ뽪λ} */

	return E_OK;
}

#else

/** %jp{ε}%en{Wakeup Task}
 * @param  tskid    %jp{оݤΥIDֹ}%en{ID number of the task to be woken up}
 * @retval E_OK     %jp{ｪλ}%en{Normal completion}
 * @retval E_ID     %jp{IDֹ(tskid뤤ϻѤǤʤ)}%en{Invalid ID number(tskid is invalid or unusable)}
 * @retval E_NOEXS  %jp{֥̤(оݥ̤Ͽ)}%en{Non-existant object(specified task is not registered)}
 * @retval E_OBJ    %jp{֥Ⱦ֥顼(оݥٻ߾)}%en{Object state error(specified task is in DORMANT state)}
 * @retval E_QOVR   %jp{塼󥰥Сե(׵ᥭ塼󥰤ΥСե)}%en{Queue overflow(overflow of wakeup request count)}
 */
ER iwup_tsk(
		ID tskid)
{
	return wup_tsk(tskid);
}

#endif


#else	/* _KERNEL_SPT_IWUP_TSK */


/** %jp{ε}%en{Wakeup Task}
 * @param  tskid    %jp{оݤΥIDֹ}%en{ID number of the task to be woken up}
 * @retval E_NOSPT  %jp{̤ݡȵǽ}%en{Unsupported function}
 */
ER iwup_tsk(
		ID tskid)
{
	return E_NOSPT;
}


#endif	/* _KERNEL_SPT_IWUP_TSK */


/* end of file */
