/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.exch2k.store;

import java.io.*;
import java.util.*;
import java.text.*;

import javax.mail.*;
import javax.mail.internet.*;
import javax.naming.*;
import javax.naming.directory.*;
import org.apache.commons.httpclient.*;
import org.apache.commons.httpclient.util.*;
import org.apache.webdav.lib.*;
import org.apache.webdav.lib.methods.*;

import jp.co.connectone.common.*;
import jp.co.connectone.eai.exch2k.dav.*;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.IServiceInfo;
import jp.co.connectone.service.IServiceInfoRawData;
import jp.co.connectone.store.client.*;
import jp.co.connectone.store.*;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.*;

public abstract class EAIExch2kBase
{
	private static int HTTP_FILE_NOT_FOUND = 404;

	protected abstract DavSearchCondition convertCondition(SearchCondition src) throws HandleException;

	protected static final String DEFAULT_EXCHANGE_BASE_PATH_ELEMENT = "/exchange/";

	private String exchageBasePathElement = DEFAULT_EXCHANGE_BASE_PATH_ELEMENT;

	protected static final String DEFAULT_PROTOCOL_PREFIX = "http://";

	protected static final String SSL_PROTOCOL_PREFIX = "https://";

	protected String protocolPrefix = DEFAULT_PROTOCOL_PREFIX;

	public static final int DEPTH_0 = 0;

	public static final int DEPTH_1 = 1;

	public static final int DEPTH_INFINITY = 2;

	protected abstract String getElmentName();
	private static final int URL_TYPE_NULL=0;//initial type. need auto-check for url type;
	private static final int URL_TYPE_MAIL_ACCOUNT_NAME_PART=1;//use left side of "@" on mail address;
	private static final int URL_TYPE_MAIL_ADDRESS=2;//use whole mail address;
	private static int urlType=URL_TYPE_NULL;

	protected int setUrlType() throws NoSuchRights
	{
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");

		int type = URL_TYPE_NULL;
		if (urlType!=type) return urlType;
		
		ExWebdavResource davClient;

		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		String url=null;
		try {
			url=protocolPrefix + this.serverAddress + exchageBasePathElement + mailAddress + "/";
			davClient = new ExWebdavResource(url, cred);
			type = URL_TYPE_MAIL_ADDRESS;
		}
		catch (Exception e) {
			String codeStr = "";
			if (e instanceof HttpException) {
				HttpException he = (HttpException)e;
				int code = he.getReasonCode();
				codeStr = "" + code;
			}
			Log.debug("access with url "+ url + " failed. message is("+e.getMessage()+":"+codeStr+")");
			try {
				url=protocolPrefix + this.serverAddress + exchageBasePathElement + mailAddress.substring(0, mailAddress.indexOf('@')) + "/";
				davClient = new ExWebdavResource(url, cred);
				davClient.getExistence();// dummy call to elace compiler worning...Orz
				type = URL_TYPE_MAIL_ACCOUNT_NAME_PART;
			}
			catch (Exception e2) {
				codeStr = "";
				if (e2 instanceof HttpException) {
					HttpException he = (HttpException)e2;
					int code = he.getReasonCode();
					codeStr = "" + code;
					if (code==401) {
						throw new NoSuchRights("access with url "+ url + " failed. message is("+e2.getMessage()+":401)");
					}
				}
				Log.debug("access with url "+ url + " failed. message is("+e2.getMessage()+":"+codeStr+")");
			}

		}
		urlType = type;
		return type;
	}
	
	protected String generateBaseURL() throws HandleException
	{
		if (urlType==URL_TYPE_NULL) {
			setUrlType();
		}
		switch (urlType) {
		case URL_TYPE_MAIL_ACCOUNT_NAME_PART:
			return protocolPrefix + this.serverAddress + exchageBasePathElement + mailAddress.substring(0, mailAddress.indexOf('@')) + "/";
		case URL_TYPE_MAIL_ADDRESS:
			return protocolPrefix + this.serverAddress + exchageBasePathElement + mailAddress + "/";
		}
		throw new HandleException("Illegal urlType detected.");
	}

	protected String generateBaseURL(String strStore)
	{
		return protocolPrefix + this.serverAddress + exchageBasePathElement + strStore + "/";
	}

	protected String getDomainName(String ADAddress,String ADAddress2) throws ServerDown,HandleException
	{
		Log.debug("getDomainName:ADAddress="+ADAddress+",ADAddress2="+ADAddress2);
		String domainName = null;
		NamingEnumeration<SearchResult> results;
		ADTool tool = new ADTool();
		try {
			results = tool.getRootDSE(ADAddress,ADAddress2);
		}
		catch (NoSuchRights nsr) {
			throw nsr;
		}
		catch (AuthenticationException ae) {
			Log.error("getDomainName:Authentication failed(" + ae.getMessage() + ")", ae);
			throw new NoSuchRights("getDomainName:Authentication failed(" + ae.getMessage() + ")");
		}
		catch(Exception e) {
			Log.error("getDomainName:error getting rootDSN("+e.getMessage()+")",e);
			throw new ServerDown("getDomainName:error getting rootDSN("+e.getMessage()+")");
		}
		SearchResult result;
		try {
			result = results.next();
		}
		catch(NamingException e){
			Log.error("getDomainName:error getting rootDSN("+e.getMessage()+")",e);
			throw new HandleException("getDomainName:error getting rootDSN("+e.getMessage()+")");
		}
		Attribute attr = result.getAttributes().get("defaultNamingContext");
		try {
			domainName = (String)attr.get(0);
		}
		catch(NamingException e){
			Log.error("getDomainName:error getting defaultNamingContext, assume server is not AD("+e.getMessage()+")",e);
			throw new HandleException("getDomainName:error getting defaultNamingContext, assume server is not AD("+e.getMessage()+")");
		}
		domainName = domainName.replaceAll(",DC=",".").substring(3);
		Log.debug("getDomainName:domainName="+domainName);
		return domainName;
	}

	protected String getFirstExch2kServer(String userID, String paassword, String ADAddress, String ADAddress2, String domain) throws ServerDown, HandleException
	{
		Log.debug("getFirstExch2kServer:ADAddress=" + ADAddress);
		String exchAddress = null;
		NamingEnumeration<SearchResult> results;
		ADTool tool = new ADTool();
		try {
			results = tool.getUserDetail(userID, password, ADAddress, ADAddress2, domain);
		}
		catch (NoSuchRights nsr) {
			throw nsr;
		}
		catch (AuthenticationException ae) {
			Log.error("getFirstExch2kServer:Authentication failed(" + ae.getMessage() + ")", ae);
			throw new NoSuchRights("getFirstExch2kServer:Authentication failed(" + ae.getMessage() + ")");
		}
		catch (Exception e) {
			Log.error("getFirstExch2kServer:error getting user record(" + e.getMessage() + ")", e);
			throw new ServerDown("getFirstExch2kServer:error getting user record(" + e.getMessage() + ")");
		}
		SearchResult result;
		try {
			if (results.hasMore()==false) {
				throw new ServerDown("getFirstExch2kServer:LDAP returns no result of userDetail. assume auth error");
			}
			result = results.next();
		}
		catch (AuthenticationException ae) {
			Log.error("getFirstExch2kServer:Authentication failed(" + ae.getMessage() + ")", ae);
			throw new NoSuchRights("getFirstExch2kServer:Authentication failed(" + ae.getMessage() + ")");
		}
		catch (NamingException e) {
			Log.error("getFirstExch2kServer:error getting user record(" + e.getMessage() + ")", e);
			throw new HandleException("getFirstExch2kServer:error getting user record(" + e.getMessage() + ")");
		}
		Attribute attr = result.getAttributes().get("mail");
		try {
			mailAddress = (String) attr.get(0);
		}
		catch (Exception e) {
			Log.error("getFirstExch2kServer:error getting mail", e);
			throw new HandleException("getFirstExch2kServer:error getting mail, assume server is not AD(" + e.getMessage() + ")");
		}
		attr = result.getAttributes().get("msExchHomeServerName");
		try {
			exchAddress = (String) attr.get(0);
			Log.debug("getFirstExch2kServer:msExchHomeServerName=" + exchAddress);
		}
		catch (Exception e) {
			Log.error("getFirstExch2kServer:error getting msExchHomeServerName", e);
			throw new HandleException("getFirstExch2kServer:error getting msExchExchangeServer, assume server is not AD(" + e.getMessage() + ")");
		}
		exchAddress = exchAddress.substring(exchAddress.lastIndexOf("cn=") + 3);
		Log.debug("getFirstExch2kServer:exchAddress=" + exchAddress);
		return exchAddress;
	}

	protected String exGetFirstExch2kServer(String idToSee, String userID, String paassword, String ADAddress, String ADAddress2, String domain) throws ServerDown, HandleException
	{
		Log.debug("exGetFirstExch2kServer:ADAddress=" + ADAddress);
		String exchAddress = null;
		NamingEnumeration<SearchResult> results;
		ADTool tool = new ADTool();
		try {
			results = tool.getOtherUserDetail(idToSee, userID, password, ADAddress,ADAddress2, domain);
		}
		catch (NoSuchRights nsr) {
			throw nsr;
		}
		catch (AuthenticationException ae) {
			Log.error("exGetFirstExch2kServer:Authentication failed(" + ae.getMessage() + ")", ae);
			throw new NoSuchRights("exGetFirstExch2kServer:Authentication failed(" + ae.getMessage() + ")");
		}
		catch (Exception e) {
			Log.error("exGetFirstExch2kServer:error getting user record(" + e.getMessage() + ")", e);
			throw new ServerDown("exGetFirstExch2kServer:error getting user record(" + e.getMessage() + ")");
		}
		SearchResult result;
		try {
			if (results.hasMore()==false) {
				throw new ServerDown("getFirstExch2kServer:LDAP returns no result of userDetail. assume auth error");
			}
			result = results.next();
		}
		catch (NamingException e) {
			Log.error("exGetFirstExch2kServer:error getting user record(" + e.getMessage() + ")", e);
			throw new HandleException("exGetFirstExch2kServer:error getting user record(" + e.getMessage() + ")");
		}
		Attribute attr = result.getAttributes().get("mail");
		try {
			mailAddress = (String) attr.get(0);
		}
		catch (NamingException e) {
			Log.error("exGetFirstExch2kServer:error getting mail.", e);
			throw new HandleException("exGetFirstExch2kServer:error getting mail, assume server is not AD(" + e.getMessage() + ")");
		}
		attr = result.getAttributes().get("msExchHomeServerName");
		try {
			exchAddress = (String) attr.get(0);
		}
		catch (NamingException e) {
			Log.error("exGetFirstExch2kServer:error getting mail", e);
			throw new HandleException("exGetFirstExch2kServer:error getting msExchExchangeServer, assume server is not AD(" + e.getMessage() + ")");
		}
		exchAddress = exchAddress.substring(exchAddress.lastIndexOf("cn=") + 3);
		Log.debug("exGetFirstExch2kServer:exchAddress=" + exchAddress);
		return exchAddress;
	}

	protected void setServerAddress(Exch2kServiceInfo si) throws ServerDown, HandleException
	{
		if (si.protocolPrefix!=null) {
			if (si.protocolPrefix.length()>0) {
				protocolPrefix = si.protocolPrefix;
			}
		}
		String serverAddress = si.getServerAddress();
		String serverAddress2 = si.getStandbyADAddress();
		this.adAddress = serverAddress;
		this.adAddress2 = serverAddress2;
		setDomain(getDomainName(serverAddress,serverAddress2));
		String firstMDB = getFirstExch2kServer(userID, password, serverAddress,serverAddress2, domain);
		mbkServerAddress = si.getCasAddress();
		this.serverAddress = mbkServerAddress;
		if ((mbkServerAddress == null) ||("".equalsIgnoreCase(mbkServerAddress))) {
			mbkServerAddress = firstMDB;
			DNSTool dns = new DNSTool(serverAddress,serverAddress2);
			this.serverAddress = dns.getIPAddress(mbkServerAddress + "." + domain);
		}
	}

	protected void exSetServerAddress(String idToSee, Exch2kServiceInfo si) throws ServerDown, HandleException
	{
		if (si.protocolPrefix!=null) {
			if (si.protocolPrefix.length()>0) {
				protocolPrefix = si.protocolPrefix;
			}
		}
		String serverAddress = si.getServerAddress();
		String serverAddress2 = si.getStandbyADAddress();
		this.adAddress = serverAddress;
		this.adAddress2 = serverAddress2;
		setDomain(getDomainName(serverAddress,serverAddress2));
		String firstMDB = exGetFirstExch2kServer(idToSee, userID, password, serverAddress,serverAddress2, domain);
		mbkServerAddress = si.getCasAddress();
		this.serverAddress = mbkServerAddress;
		if ((mbkServerAddress == null) ||("".equalsIgnoreCase(mbkServerAddress))) {
			mbkServerAddress = firstMDB;
			DNSTool dns = new DNSTool(serverAddress,serverAddress2);
			this.serverAddress = dns.getIPAddress(mbkServerAddress + "." + domain);
		}
	}

	protected IObjectIndex saveData(String escapedUrl, Hashtable<String,String> itemList) throws ServerDown, HandleException
	{
		if (itemList == null) return null;
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		String baseURL = escapedUrl.substring(0, escapedUrl.lastIndexOf("/") + 1);
		try {
			baseURL = URIUtil.decode(baseURL, "UTF-8");

		}
		catch (URIException e) {
		}
		;
		Log.debug("baseURL=" + baseURL);
		try {
			davClient = new ExWebdavResource(baseURL, cred);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		try {
			davClient.propPatchMethod(generateBaseURL(), escapedUrl, itemList);
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}

		// int status = davClient.getStatusCode();
		String oid = davClient.getHttpURL().getEscapedURI();

		return new Exch2kObjectIndex(oid);
	}

	protected List<HashMap<String,String>> searchUserFromActiveDirectory(SearchConditionCollection conds) throws NullPointerException, ServerDown, HandleException
	{
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (adAddress == null) throw new NullPointerException("set adAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		Iterator<ISearchConditionElement> it = conds.iterator();
		ISearchCondition cond = null;
		String whereStr = "";
		while (it.hasNext()) {
			ISearchConditionElement elem = it.next();
			cond = (ISearchCondition)elem;
			Log.debug("condition:" + cond.getType() + ":" + cond.getValue());
		}
		if (cond.getType() == AddressBookSearchCondition.CONDITION.TYPE_DISPLAYNAME) {
			whereStr = " (displayName=" + cond.getValue() + "*) ";
		}
		else if (cond.getType() == AddressBookSearchCondition.CONDITION.TYPE_FIRSTNAME) {
			whereStr = " (sn=*" + cond.getValue() + "*) ";
		}
		else if (cond.getType() == AddressBookSearchCondition.CONDITION.TYPE_LASTNAME) {
			whereStr = " (givenName=*" + cond.getValue() + "*) ";
		}
		else if (cond.getType() == AddressBookSearchCondition.CONDITION.TYPE_ACCOUNTNAME) {
			whereStr = " (sAMAccountName=" + cond.getValue() + ") ";
		}
		else {
			throw new HandleException("Illegal search type");
		}
		Log.debug("whereStr:" + whereStr);
		ADTool ad = new ADTool();
		NamingEnumeration<SearchResult> results = null;
		ArrayList<HashMap<String,String>> lists = new ArrayList<HashMap<String,String>>();
		try {
			results = ad.getUserDatas(userID, password, adAddress, adAddress2, domain, whereStr);
			while (results.hasMoreElements()) {
				Log.debug("LOOP FIRST LINE");
				SearchResult aResult = (SearchResult) results.nextElement();
				Log.debug("name: " + aResult.getName());
				Attributes attrs = aResult.getAttributes();
				NamingEnumeration<? extends Attribute> ae = attrs.getAll();
				HashMap<String,String> list = new HashMap<String,String>();
				Log.debug("loop IN");
				while (ae.hasMoreElements()) {
					Attribute attr = (Attribute) ae.nextElement();
					String strAttrID = attr.getID();
					NamingEnumeration<?> e = attr.getAll();
					String value = "";
					while (e.hasMoreElements()) {
						Object o = e.nextElement();
						Log.trace("class of object("+ strAttrID + "):"+o.getClass());
						if (o instanceof String) {
							value = (String) o;
						}
					}
					Log.trace( strAttrID + ": " + value);
					list.put(strAttrID, value);
				}
				Log.debug("loop OUT");
				lists.add(list);
				list = null;
				Log.debug("LOOP LAST LINE");
			}
			Log.debug("no of accounts:" + lists.size());
			results.close();
		}
		catch (AuthenticationException ae) {
			Log.error("searchUserFromActiveDirectory:Authentication failed(" + ae.getMessage() + ")", ae);
			throw new NoSuchRights("searchUserFromActiveDirectory:Authentication failed(" + ae.getMessage() + ")");
		}
		catch (Exception e) {
			Log.error("", e);
			throw new ServerDown(e.getMessage());
		}

		return lists;
	}

	protected boolean sendSavedMsg(String oid) throws HandleException,ServerDown, DataNotFound
	{
		if (oid == null) return false;
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient = null;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		Log.debug("oid=" + oid);
		try {
			davClient = new ExWebdavResource(this.generateBaseURL(), cred);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}

		String escapedUrl = escapeURL(this.generateBaseURL() + "##DavMailSubmissionURI##/");
		String url = null;
		try {
			url = URIUtil.decode(oid, "UTF-8");
			escapedUrl = URIUtil.decode(escapedUrl, "UTF-8");
		}
		catch (Exception e) {
		}
		;

		try {
			davClient.moveMethod(url, escapedUrl);
		}
		catch (HttpException he) {
			int code = 0;
			if (davClient != null) {
				code = davClient.getStatusCode();
				if (code == HTTP_FILE_NOT_FOUND) {
					throw new DataNotFound(oid);
				}
			}
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + code + ") " + he.getMessage());
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		int status = davClient.getStatusCode();
		Log.debug("code=" + status);

		return true;
	}

	protected String generateStoreURL(String src) throws HandleException
	{
		String rc = src;
		if (src == null) return null;
		if (!rc.startsWith(protocolPrefix)) {
			rc = generateBaseURL() + rc;
		}
		if (!rc.endsWith("/")) {
			rc = rc + "/";
		}
		return rc;
	}

	protected IObjectIndex saveData(String storeName, MimeMessage msg) throws ServerDown, HandleException
	{
		if (msg == null) return null;
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		String storeUrl = generateStoreURL(storeName);
		Log.debug("storeURL=" + storeUrl);

		try {
			davClient = new ExWebdavResource(storeUrl, cred);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		String BCC = "";
		try {
			Address recipients[] = msg.getRecipients(Message.RecipientType.BCC);
			if (recipients != null) {
				for (int i = 0; i < recipients.length; i++) {
					String tmp = recipients[i].toString();
					tmp = tmp.substring(tmp.indexOf('<') + 1, tmp.indexOf('>')) + ",";
					BCC = BCC + tmp;
				}
				msg.setRecipients(Message.RecipientType.BCC, (Address[]) null);
			}
			if (BCC.length() > 1) {
				BCC = BCC.substring(0, BCC.length() - 1);
			}
		}
		catch (MessagingException me) {
			Log.debug("error on getting BCC addresses (" + me.getMessage() + ")");
		}
		Log.debug("BCC=" + BCC);

		Hashtable<String,String> BCCProp = null;
		try {
			if (BCC.length() != 0) {
				BCCProp = new Hashtable<String,String>();
				BCCProp.put("urn:schemas:mailheader:bcc", BCC);
			}
		}
		catch (NullPointerException e) {
		}
		;
		String msgStr = getStringFromMessage(msg);
		String escapedUrl = escapeURL(storeUrl + +new Date().getTime() + ".EML");

		try {
			davClient.putMethod(escapedUrl, msgStr);
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		// int status = davClient.getStatusCode();
		String oid = davClient.getHttpURL().getEscapedURI();
		// String url = oid;
		try {
			if (BCCProp != null) {
				davClient.propPatchMethod(generateBaseURL(), escapedUrl, BCCProp);
			}
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		return new Exch2kObjectIndex(oid);
	}

	@SuppressWarnings("unchecked")
	protected Vector<HashMap<String,String>> search(String storeURL, Vector<String> itemList, SearchConditionCollection conds) throws NullPointerException, ServerDown, HandleException
	{
		if (itemList == null) return null;
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		String storeUrl = generateStoreURL(storeURL);
		Log.debug("storeURL=" + storeUrl);
		try {
			davClient = new ExWebdavResource(storeUrl, cred);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		Enumeration<Enumeration<BaseProperty>> em;
		try {
			Vector<DavSearchCondition> conditions = new Vector<DavSearchCondition>();
			Iterator<ISearchConditionElement> it = conds.iterator();
			while (it.hasNext()) {
				conditions.add(convertCondition((SearchCondition) it.next()));
			}

			em = davClient.searchPropMethod(DepthSupport.DEPTH_1, itemList, conditions);
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}

		Vector<HashMap<String,String>> lists = new Vector<HashMap<String,String>>();
		while (em.hasMoreElements()) {
			Enumeration<BaseProperty> em2 = em.nextElement();
			HashMap<String,String> list = new HashMap<String,String>();
			while (em2.hasMoreElements()) {
				BaseProperty o = (BaseProperty) em2.nextElement();
				list.put(o.getName(), o.getPropertyAsString());
				Log.trace(o.getName() + "::" + o.getPropertyAsString());
			}
			lists.add(list);
		}

		try {
			davClient.close();
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}

		return lists;
	}

	@SuppressWarnings("unchecked")
	protected Vector<HashMap<String,Object>> exGetListByDate(int depth, IFolderIndex folder, Vector<String> itemList, Date date, String dateColumnName) throws NullPointerException, ServerDown, HandleException
	{
		if (itemList == null) return null;
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		
		ArrayList<String> splitPath = dividePathInSharp(generateStoreURL((String) folder.getIndex()));
		String storeUrl = splitPath.get(0);
		String addPath = splitPath.get(1);
		
		Log.debug("storeURL=" + storeUrl + " addPath=" + addPath);
		Log.debug("date=" + date);
		try {
			davClient = new ExWebdavResource(storeUrl, cred);
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}

		Log.debug("depth=" + depth);
		Enumeration<Enumeration<BaseProperty>> em;
		int d = DepthSupport.DEPTH_INFINITY;
		switch (depth) {
		case DEPTH_0:
			d = DepthSupport.DEPTH_0;
			break;
		case DEPTH_1:
			d = DepthSupport.DEPTH_1;
			break;
		}
		try {
			if (date == null) {
				em = davClient.searchPropMethod(addPath, d, itemList);
			}
			else {
				Vector<DavSearchCondition> conds = new Vector<DavSearchCondition>();
				Date tmpDate = date;
				SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");
				SimpleDateFormat df2 = new SimpleDateFormat("yyyy/MM/dd hh:mm:ss");
				String tmpStr = df.format(date) + " 00:00:00";
				try {
					tmpDate = df2.parse(tmpStr);
				}
				catch (java.text.ParseException e) {
				}
				;
				conds.add(new DavSearchCondition(dateColumnName, DavSearchType.TYPE_DATETIME_SZ, tmpDate, DavSearchOperator.OP_BIG_EQUAL));
				tmpStr = df.format(date) + " 23:59:59";
				try {
					tmpDate = df2.parse(tmpStr);
				}
				catch (java.text.ParseException e) {
				}
				;
				conds.add(new DavSearchCondition(dateColumnName, DavSearchType.TYPE_DATETIME_SZ, tmpDate, DavSearchOperator.OP_NOT_BIG));
				em = davClient.searchPropMethod(addPath, d, itemList, conds);
			}
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:searchPropMethod:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:searchPropMethod:IOException(" + ie.getMessage() + ")");
		}

		Vector<HashMap<String,Object>> lists = new Vector<HashMap<String,Object>>();
		while (em.hasMoreElements()) {
			Enumeration<BaseProperty> em2 = em.nextElement();
			HashMap<String,Object> list = new HashMap<String,Object>();
			while (em2.hasMoreElements()) {
				try {
					BaseProperty o = em2.nextElement();
					String propName = o.getName();
					String value = o.getPropertyAsString();
					list.put(propName, value);
					Log.trace(propName + "::" + value);
				}
				catch (Exception e) {
					Log.error("error on getting property element.continue..", e);
				}
			}
			if (itemList.contains("urn:schemas:httpmail:attachmentfilename")) {
				String hasattachment = (String) list.get("hasattachment");
				Log.debug("hasattachment:" + hasattachment);
				if ("1".equals(hasattachment)) {
					HashMap<String,byte[]> m = null;
					try {
						m = getAllAttachments(getRealEscapedURL((String) list.get("href")));
					}
					catch (Exception e) {
						Log.error("unexpected exception on getAllAttachments. ignore attachment information.", e);
						try {
							m = new HashMap<String,byte[]>();
							m.put("file get error", "unexpected exception on getAllAttachments. ignore attachment information.".getBytes());
						}
						catch (Exception ex) {
						}
						;
					}
					list.put("attachmentfile", m);
				}
			}
			lists.add(list);
		}

		try {
			davClient.close();
		}
		catch (HttpException he) {
			Log.error("error on closing davClient", he);
		}
		catch (IOException ie) {
			Log.error("error on closing davClient", ie);
		}

		return lists;

	}

	@SuppressWarnings("unchecked")
	private HashMap<String,byte[]> getAllAttachments(String encodedPath) throws NullPointerException, ServerDown, NoSuchRights
	{
		ExWebdavResource davClient;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		String storeUrl = encodedPath;
		String addPath = "";
		
		try {
			ArrayList<String> splitPath = dividePathInSharp(URIUtil.decode(encodedPath, "UTF-8"));
			storeUrl = splitPath.get(0);
			addPath = splitPath.get(1);
		}
		catch (Exception e) {
		}
		;
		Log.debug("getAllAttachments::storeURL=" + storeUrl + " addPath=" + addPath);
		try {
			davClient = new ExWebdavResource(storeUrl, cred);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}

		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}

		HashMap<String,Enumeration<BaseProperty>> attachments = null;
		try {
			attachments = davClient.xmsEnumAttsMethod(addPath);
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}

		if (attachments == null) return null;
		Set<String> keys = attachments.keySet();
		Iterator<String> it = keys.iterator();
		HashMap<String,byte[]> list = new HashMap<String,byte[]>();
		while (it.hasNext()) {
			String url = it.next();
			Log.debug("url:" + url);
			Enumeration<BaseProperty> em2 = attachments.get(url);
			String fileName = null;
			while (em2.hasMoreElements()) {
				BaseProperty o = em2.nextElement();
				String propName = o.getName();
				String value = o.getPropertyAsString();
				Log.trace(propName + "::" + value);
				if (propName.indexOf("attachmentfilename") >= 0) {
					fileName = value;
					break;
				}
				if (propName.indexOf("cn") >= 0) { // changed to "cn" because
					// of "EML" attach doesn't
					// have attachmentfilename
					if (fileName == null) {
						fileName = value + ".eml";
					}
				}
			}
			// list.put(url,fileName);
			// list.put(getFileBody(url),fileName);
			// try {
			// url = URIUtil.encodePath(storeUrl+"/"+fileName,"UTF-8");
			// }
			// catch(URIException e) {};
			url = getRealEscapedURL(url);
			Log.debug("url (after solve exchange encode bug):" + url);
			try {
				list.put(fileName, getFileBody(url));
			}
			catch (Exception e) {
				Log.error("error on getting attachment body of file " + fileName + ", fill dummy into body", e);
				list.put(fileName, "error on getting attachment body".getBytes());
			}
		}

		try {
			davClient.close();
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:close:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:close:IOException(" + ie.getMessage() + ")");
		}
		return list;
	}

	protected byte[] getFileBody(String fileName) throws ServerDown, NoSuchRights, NullPointerException
	{
		if (fileName == null) return null;
		String encodedURL = fileName;
		Log.debug("get the message stream of file " + encodedURL);
		InputStream is = getFileData(encodedURL);
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		try {
			byte[] buf = new byte[8192];
			while (true) {
				int i = is.read(buf);
				if (i <= 0) break;
				os.write(buf, 0, i);
			}
		}
		catch (IOException ie) {
			Log.error("", ie);
			throw new ServerDown(ie.getMessage());
		}
		byte[] rc = os.toByteArray();
		return rc;
	}

	@SuppressWarnings("unchecked")
	private Vector<String> getFirstChildlen(String path) throws ServerDown,NoSuchRights
	{

		ExWebdavResource davClient;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		ArrayList<String> splitPath = dividePathInSharp(path);
		String storeUrl = (String)splitPath.get(0);
		String addPath = (String)splitPath.get(1);
		
		Log.debug("storeURL=" + storeUrl + " addPath=" + addPath);
		try {
			davClient = new ExWebdavResource(storeUrl, cred);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}

		Enumeration<Enumeration<BaseProperty>> em;
		Vector<String> itemList = new Vector<String>();
		itemList.add("DAV:id");
		itemList.add("DAV:href");
		itemList.add("DAV:getcontentlength");
		try {
			em = davClient.searchFolderMethod(addPath, DepthSupport.DEPTH_1, itemList);
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		Vector<String> lists = new Vector<String>();
		while (em.hasMoreElements()) {
			Enumeration<BaseProperty> em2 = em.nextElement();
			while (em2.hasMoreElements()) {
				BaseProperty o = em2.nextElement();
				try {
					Log.trace(o.getName() + "::" + URIUtil.decode(o.getPropertyAsString(), "UTF-8"));
					if (o.getName().equals("href")) {
						lists.add(URIUtil.decode(o.getPropertyAsString(), "UTF-8"));
					}
				}
				catch (URIException e) {
				}
				;
			}
		}

		return lists;
	}

	protected Vector<Object> getAllSubFolders(String parentFolder) throws ServerDown,HandleException
	{
		String storeUrl;
		if (parentFolder.startsWith("http")) {
			storeUrl = parentFolder;
		}
		else {
			storeUrl = generateStoreURL(parentFolder);
		}
		Vector<Object> v = new Vector<Object>();
		Log.debug("getAllSubFolders:enter " + storeUrl);
		Vector<String> thisFolder = null;
		try {
			thisFolder = getFirstChildlen(storeUrl);
		}
		catch (NoSuchRights ne) {
			throw ne;
		}
		catch (Exception e) {
			Log.debug("getAllSubFolders:could not get subfolder of " + storeUrl);
			return v;
		}
		Enumeration<String> e = thisFolder.elements();
		while (e.hasMoreElements()) {
			String path = e.nextElement();
			if (path != null) {
				Log.debug("adding path to vector :" + path);
				Vector<Object> child = getAllSubFolders(path);
				v.add(child);
			}
		}
		v.add(0, storeUrl);

		return v;
	}

	@SuppressWarnings("unchecked")
	protected Vector<HashMap<String,Object>> getListByDate(IFolderIndex storeURL, Vector<String> itemList, Date date, String dateColumnName) throws NullPointerException, ServerDown, HandleException
	{
		if (itemList == null) return null;
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		String storeUrl = generateStoreURL((String) storeURL.getIndex());
		Log.debug("storeURL=" + storeUrl);
		try {
			davClient = new ExWebdavResource(storeUrl, cred);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}

		Enumeration<Enumeration<BaseProperty>> em;
		try {
			if (date == null) {
				em = davClient.searchPropMethod(DepthSupport.DEPTH_1, itemList);
			}
			else {
				Vector<DavSearchCondition> conds = new Vector<DavSearchCondition>();
				Date tmpDate = date;
				SimpleDateFormat df = new SimpleDateFormat("yyyy/MM/dd");
				SimpleDateFormat df2 = new SimpleDateFormat("yyyy/MM/dd hh:mm:ss");
				String tmpStr = df.format(date) + " 00:00:00";
				try {
					tmpDate = df2.parse(tmpStr);
				}
				catch (java.text.ParseException e) {
				}
				;
				conds.add(new DavSearchCondition(dateColumnName, DavSearchType.TYPE_DATETIME_SZ, tmpDate, DavSearchOperator.OP_BIG_EQUAL));
				tmpStr = df.format(date) + " 23:59:59";
				try {
					tmpDate = df2.parse(tmpStr);
				}
				catch (java.text.ParseException e) {
				}
				;
				conds.add(new DavSearchCondition(dateColumnName, DavSearchType.TYPE_DATETIME_SZ, tmpDate, DavSearchOperator.OP_NOT_BIG));
				em = davClient.searchPropMethod(DepthSupport.DEPTH_1, itemList, conds);
			}
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}

		Vector<HashMap<String,Object>> lists = new Vector<HashMap<String,Object>>();
		while (em.hasMoreElements()) {
			Enumeration<BaseProperty> em2 = em.nextElement();
			HashMap<String,Object> list = new HashMap<String,Object>();
			while (em2.hasMoreElements()) {
				BaseProperty o = (BaseProperty) em2.nextElement();
				list.put(o.getName(), o.getPropertyAsString());
				Log.trace(o.getName() + "::" + o.getPropertyAsString());
			}
			lists.add(list);
		}

		try {
			davClient.close();
		}
		catch (HttpException he) {
			Log.error("WebdavResource:HttpException(" + he.getMessage() + ")", he);
			throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + ")");
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}

		return lists;
	}

	protected Vector<HashMap<String,Object>> getList(IFolderIndex storeURL, Vector<String> itemList) throws NullPointerException, ServerDown, HandleException
	{
		return getListByDate(storeURL, itemList, null, null);
	}

	/*---- getter/setter --------*/
	private String adAddress;
	private String adAddress2;

	private String serverAddress;

	private String mbkServerAddress;

	private String path;

	private String userID;

	private String password;

	private String domain;

	private String mailAddress;

	public String getAdAddress()
	{
		return adAddress;
	}

	public void setAdAddress(String adAddress)
	{
		this.adAddress = adAddress;
	}

	/**
	 * @return domain ߂܂B
	 */
	public String getDomain()
	{
		return domain;
	}

	/**
	 * @param domain
	 *            domain ݒB
	 */
	protected void setDomain(String domain)
	{
		this.domain = domain;
	}

	/**
	 * @return password ߂܂B
	 */
	public String getPassword()
	{
		return password;
	}

	/**
	 * @param password
	 *            password ݒB
	 */
	protected void setPassword(String password)
	{
		this.password = password;
	}

	/**
	 * @return path ߂܂B
	 */
	public String getPath()
	{
		return path;
	}

	/**
	 * @param path
	 *            path ݒB
	 */
	protected void setPath(String path)
	{
		this.path = path;
	}

	/**
	 * @return serverAddress ߂܂B
	 */
	public String getServerAddress()
	{
		return serverAddress;
	}

	/**
	 * @return userID ߂܂B
	 */
	public String getUserID()
	{
		return userID;
	}

	/**
	 * @param userID
	 *            userID ݒB
	 */
	protected void setUserID(String userID)
	{
		this.userID = userID;
	}

	protected void setAccountData(PasswordBasedAccountDataImpl acc) throws HandleException
	{
		Exch2kServiceInfo si = (Exch2kServiceInfo)acc.getServiceInfo();
		setUserID(acc.getUserID());
		setPassword(acc.getPassword());
		setServerAddress(si);
	}

	protected void exSetAccountData(PasswordBasedAccountDataImpl acc, String idToSee) throws HandleException
	{
		Exch2kServiceInfo si = (Exch2kServiceInfo)acc.getServiceInfo();
		setUserID(acc.getUserID());
		setPassword(acc.getPassword());
		exSetServerAddress(idToSee, si);
	}

	/**
	 * @param strDate
	 * @return if (type.equals(DavSearchType.TYPE_DATETIME_SZ)) return "CAST(\"" +
	 *         sdf.format((Date)value) + "\" AS \"dateTime\")";
	 * 
	 */

	protected Date davDateStr2JavaDate(String strDate)
	{
		Date date = null;
		try {
			if (strDate != null) {
				DavDateFormat sdf = new DavDateFormat();
				date = sdf.parse(strDate);
			}
		}
		catch (java.text.ParseException e) {
			Log.debug("ErrF" + e.getMessage());
		}
		return date;
	}

	protected String javaDate2DavDateStr(Date date)
	{
		if (date == null) return null;
		DavDateFormat sdf = new DavDateFormat();
		return sdf.format(date);
	}

	protected String javaDate2MSDavDateStr(Date date)
	{
		if (date == null) return null;
		DavDateFormat sdf = new DavDateFormat(DavDateFormat.FORMAT_DAV_T_SEC);
		return sdf.format(date);
	}

	protected String escapeURL(String originalUrl)
	{
		try {
			return URIUtil.encodePath(originalUrl);
		}
		catch (URIException e) {
			return originalUrl;
		}
	}

	@SuppressWarnings("unchecked")
	protected String getStringFromMessage(MimeMessage mmsg)
	{
		if (mmsg == null) return null;
		String msg = "";
		try {
			Enumeration<String> lines = mmsg.getAllHeaderLines();
			while (lines.hasMoreElements()) {
				msg = msg + lines.nextElement() + "\r\n";
			}
			msg = msg + "\r\n";
			Log.debug(msg);

			InputStream is = mmsg.getInputStream();
            BufferedReader b = new BufferedReader(new InputStreamReader(is),1024);

            String tmp;
            StringBuilder builder = new StringBuilder();
            builder.append(msg);
            while ((tmp = b.readLine()) != null) {
                    builder.append(tmp);
                    builder.append("\r\n");
            }
            msg = builder.toString();
		}
		catch (Exception e) {
			Log.error("Exception on creating MimeMessage", e);
		}
		return msg;
	}

	protected String populateMessageFile(Hashtable<String,String> itemList)
	{
		// String msg = "content-class: urn:content-classes:message\r\n";
		String msg = "";
		String subject = itemList.get("urn:schemas:httpmail:subject");
		String body = itemList.get("urn:schemas:httpmail:textdescription");
		String from = itemList.get("urn:schemas:httpmail:sendername");
		if (from == null) {
			from = itemList.get("urn:schemas:httpmail:from");
		}

		try {
			MimeMessage mmsg = new MimeMessage((Session) null);
			MimeMultipart part = new MimeMultipart();
			MimeBodyPart pbody = new MimeBodyPart();

			pbody.setText(body, "UTF-8");
			part.addBodyPart(pbody);

			mmsg.setSubject(subject, "UTF-8");

			if (from != null) {
				MailAddress add = new MailAddress(from);
				String address = add.getAddressPart();
				String name = add.getNamePart();
				if (address.equals(name)) {
					mmsg.setFrom(new InternetAddress(address, name, "UTF-8"));
				}
				else {
					mmsg.setFrom(new InternetAddress(address));
				}
			}

			mmsg.setContent(part);
			mmsg.saveChanges();

			msg = getStringFromMessage(mmsg);

		}
		catch (Exception e) {
			Log.error("Exception on creating MimeMessage", e);
		}

		return msg;
	}

	protected int delete(IObjectIndex oid) throws NullPointerException, ServerDown, NoSuchRights
	{
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		String encodedurl = (String) oid.getIndex();
		String url = encodedurl;
		try {
			url = URIUtil.decode(encodedurl, "UTF-8");
			Log.debug("delete:encodedurl=" + encodedurl + ":delodedurl=" + url);
		}
		catch (Exception e) {
		}
		;
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);

		try {
			davClient = new ExWebdavResource(generateBaseURL(), cred);
			davClient.deleteMethod(url);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		int code = davClient.getStatusCode();

		return code;
	}

	protected MimeMessage getMessageInstance()
	{
		Properties env = new Properties();
		env.put("mail.smtp.host", getServerAddress());
		Session session = Session.getDefaultInstance(env);
		MimeMessage msg = new MimeMessage(session);

		return msg;
	}

	@SuppressWarnings("unchecked")
	protected HashMap<String,Object> getFile(String encodedurl) throws NullPointerException, ServerDown, HandleException
	{
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		Enumeration<ResponseEntity> em;
		Enumeration<BaseProperty> em2=null;
		String url = encodedurl;
		try {
			url = URIUtil.decode(encodedurl, "UTF-8");
		}
		catch (Exception e) {
		}
		;
		Log.debug("decoded url=" + url);
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		HashMap<String,Object> msg = new HashMap<String,Object>();
		try {
			davClient = new ExWebdavResource(generateBaseURL(), cred);
			em = davClient.propfindMethod(url, DepthSupport.DEPTH_0);
			if (em != null) {
				if (em.hasMoreElements()) {
					ResponseEntity r = em.nextElement();
					em2 = r.getProperties();
				}
			}
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		int code = davClient.getStatusCode();
		if (em2 == null) {
			throw new HandleException("Server doesn't return proper mail data:code=" + code);
		}
		while (em2.hasMoreElements()) {
			BaseProperty o = em2.nextElement();
			String propName = o.getName();
			propName = propName.substring(propName.indexOf(':') + 1);
			String value = o.getPropertyAsString();
			Object old = msg.put(propName, value);
			if (old != null) {
				msg.put(propName, old);
				msg.put(o.getName(), value);
			}
			Log.trace(propName + "::" + value);
		}
		String hasattachment = (String) msg.get("hasattachment");
		Log.debug("hasattachment:" + hasattachment);
		if ("1".equals(hasattachment)) {
			HashMap<String,byte[]> m = getAllAttachments(getRealEscapedURL((String) msg.get("href")));

			msg.put("attachmentfile", m);
		}

		return msg;
	}

	@SuppressWarnings("unchecked")
	protected HashMap<String,Object> getProps(String encodedurl, Vector<String> itemList) throws NullPointerException, ServerDown, HandleException
	{
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		Enumeration<ResponseEntity> em;
		Enumeration<BaseProperty> em2=null;
		String url = encodedurl;
		try {
			url = URIUtil.decode(encodedurl, "UTF-8");
		}
		catch (Exception e) {
		}
		;
		Log.debug("decoded url=" + url);
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		HashMap<String,Object> msg = new HashMap<String,Object>();
		try {
			davClient = new ExWebdavResource(generateBaseURL(), cred);
			em = davClient.propfindMethod(url, DepthSupport.DEPTH_0, itemList);
			if (em != null) {
				if (em.hasMoreElements()) {
					ResponseEntity r = em.nextElement();
					em2 = r.getProperties();
				}
			}
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}
		int code = davClient.getStatusCode();
		if (em2 == null) {
			throw new HandleException("Server doesn't return proper mail data:code=" + code);
		}
		while (em2.hasMoreElements()) {
			BaseProperty o = em2.nextElement();
			String propName = o.getName();
			propName = propName.substring(propName.indexOf(':') + 1);
			String value = o.getPropertyAsString();
			Object old = msg.put(propName, value);
			if (old != null) {
				msg.put(propName, old);
				msg.put(o.getName(), value);
			}
			Log.trace(propName + "::" + value);
		}
		String hasattachment = (String) msg.get("hasattachment");
		Log.debug("hasattachment:" + hasattachment);
		if ("1".equals(hasattachment)) {
			HashMap<String,byte[]> m = getAllAttachments(getRealEscapedURL((String) msg.get("href")));

			msg.put("attachmentfile", m);
		}

		return msg;
	}

	protected String getRealEscapedURL(String srcUrl)
	{
		try {
			return URIUtil.encodePath(URIUtil.decode(srcUrl, "UTF-8"), "UTF-8");
		}
		catch (Exception e) {
		}
		return srcUrl;
	}

	protected InputStream getFileData(String encodedurl) throws NullPointerException, ServerDown, NoSuchRights
	{
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (serverAddress == null) throw new NullPointerException("set serverAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		ExWebdavResource davClient;
		Log.debug("url=" + encodedurl);
		String url = encodedurl;
		try {
			url = URIUtil.decode(encodedurl, "UTF-8");
			url = url.substring(0, url.lastIndexOf('/'));
			ArrayList<String> splitPath = dividePathInSharp(generateStoreURL(url));
			url = splitPath.get(0);
		}
		catch (Exception e) {
		}
		;
		Log.debug("decoded url=" + url);
		NTCredentials cred = new NTCredentials(userID, password, serverAddress, domain);
		InputStream is = null;
		try {
			davClient = new ExWebdavResource(url, cred);
			is = davClient.getMethodData(encodedurl);
		}
		catch (HttpException he) {
			if (he.getReasonCode() == 401) {
				throw new NoSuchRights("WebdavResource:Authentication failed(" + he.getMessage() + ")");
			} else {
				Log.error("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")", he);
				throw new ServerDown("WebdavResource:HttpException(" + he.getMessage() + " : " + he.getReasonCode() + ")");
			}
		}
		catch (IOException ie) {
			Log.error("WebdavResource:IOException(" + ie.getMessage() + ")", ie);
			throw new ServerDown("WebdavResource:IOException(" + ie.getMessage() + ")");
		}
		catch (IllegalArgumentException iae) {
			Log.error("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ")", iae);
			throw new ServerDown("WebdavResource:IllegalArgumentException(" + iae.getMessage() + ") check serverAddress/domain");
		}

		return is;
	}

	public String getMailAddress()
	{
		return mailAddress;
	}

	protected Address populateAddress(MailAddress src)
	{
		InternetAddress add = null;
		if (src == null) return add;

		try {
			String address = src.getAddressPart();
			if (address==null) address = src.getRawAddress();
			String name = src.getNamePart();
			String raw = src.getRawAddress();
			if (address.equalsIgnoreCase(raw)) {
				if (address.indexOf('@') < 1) {// hC
					address = address + "@" + this.getDomain();
				}
				add = new InternetAddress(address, name, "ISO-2022-JP");
			}
			else {
				if (raw.indexOf('@') < 1) {// hC
					raw = raw + "@" + this.getDomain();
				}
				add = new InternetAddress(raw);
			}
		}
		catch (UnsupportedEncodingException ue) {
			Log.error("", ue);
		}
		catch (AddressException ae) {
			Log.error("", ae);
		}

		return add;
	}

	public IRecordObject getFolderIndexFromString(String folderName) throws IncorrectData
	{
		if (folderName == null) folderName = getElmentName();
		BasicRecordObject rc = new BasicRecordObject();
		rc.setOid(new Exch2kFolderIndex(folderName));
		HashMap<String,Object> h = new HashMap<String,Object>();
		h.put("IFolderIndex", rc.getOid());
		try {
			rc.setFieldSet(h);
		}
		catch (IllegalAccessException e) {
			Log.error("getFolderIndexFromString:error on setting IFolderIndex", e);
			throw new IncorrectData("getFolderIndexFromString:error on setting IFolderIndex");
		}
		return rc;
	}

	protected FolderMetadata populateFolder(IFolderIndex idx)
	{
		FolderMetadata folder = new FolderMetadata();
		folder.setOid(idx);
		HashMap<String,Object> h = new HashMap<String,Object>();
		h.put("oid", idx);
		folder.setFieldSet(h);
		folder.setFolderName(idx.getIndex().toString());
		return folder;
	}

	@SuppressWarnings("unchecked")
	protected ArrayList<FolderMetadata> populateFolderList(int layer, String root, Vector<Object> src) throws HandleException
	{
		if (src == null) return null;
		if (src.size() == 0) return null;
		Enumeration<?> em = src.elements();
		ArrayList<FolderMetadata> dst = new ArrayList<FolderMetadata>();

		String path = null;
		String sub = null;
		String layerStr = "";
		for (int i = 0; i < layer; i++) {
			layerStr = layerStr + "-";
		}
		while (em.hasMoreElements()) {
			Object o = em.nextElement();
			try {
				Class<?> c = o.getClass();
				if (c.equals(Class.forName("java.lang.String"))) {
					Log.debug("raw data of foldername:" + (String) o);
					FolderMetadata folder = new FolderMetadata();
					path = (String) o;
					sub = path.substring(path.indexOf(root) + root.length());
					Log.debug("substring:" + sub);
					IFolderIndex oid = new Exch2kFolderIndex(path);
					folder.setOid(oid);
					Log.debug("oid:" + (String) oid.getIndex());
					HashMap<String,Object> h = new HashMap<String,Object>();
					h.put("oid", oid);
					folder.setFieldSet(h);
					Log.debug("layerStr:" + layerStr);
					folder.setFolderName(layerStr + sub.replaceAll("/", ""));
					Log.debug("folderName:" + folder.getFolderName());
					dst.add(folder);
				}
				else if (c.equals(Class.forName("java.util.Vector"))) {
					Vector<Object> v = (Vector<Object>) o;
					dst.addAll(populateFolderList(layer + 1, path, v));
				}
			}
			catch (ClassNotFoundException e) {
				throw new HandleException(e.getMessage());
			}
		}

		return dst;
	}

	protected IRecordObject makeIDObject(Object o) throws HandleException
	{
		IObjectIndex idx = new BasicObjectIndex(o);
		BasicRecordObject ro = new BasicRecordObject();
		HashMap<String,Object> h = new HashMap<String,Object>();
		h.put("UNID", idx);
		try {
			ro.setFieldSet(h);
			ro.setOid(idx);
		}
		catch (IllegalAccessException iae) {
			Log.error("error setting HashMap with IObjectIndex to BasicRecordObject.", iae);
			throw new HandleException("error setting HashMap with IObjectIndex to BasicRecordObject.");
		}
		return ro;
	}

	public IServiceInfo getServiceInfo(IStoreID storeID,String storeName, IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		Exch2kServiceInfo info = new Exch2kServiceInfo(storeID, storeName);
		String serverAddress = (String) h.get("server");
		if (serverAddress != null) info.setServerAddress(serverAddress);
		String tmp = (String)h.get("casAddress");
		if (tmp!=null) info.setCasAddress(tmp);
		tmp = (String)h.get("protocolPrefix");
		if (tmp!=null) info.setProtocolPrefix(tmp);
		tmp = (String)h.get("standbyADAddress");
		if (tmp!=null) info.setStandbyADAddress(tmp);
		return info;
	}
	
	/**
	 * URL is divided before and behind #. 
	 * @param url URL that becomes origin
	 * @return The 0th array is URL before #. The 1st array is the following URL from it. 
	 */
	private ArrayList<String> dividePathInSharp(String url){
		
		ArrayList<String> al = new ArrayList<String>(2);
		
		String storeUrl = url;
		String addPath = "";
		String[] splitPath = storeUrl.split("/");
		for(int i = 0; i < splitPath.length; i++){
			if(splitPath[i].indexOf("#") > -1){
				storeUrl = "";
				for(int j = 0; j < i; j++){
					storeUrl = storeUrl + splitPath[j] + "/";
				}
				for(int j = i; j < splitPath.length; j++){
					addPath = addPath + escapeURL(splitPath[j]) + "/";
				}
				break;
			}
		}
		
		al.add(0, storeUrl);
		al.add(1, addPath);
		return al;
	}

	/**
	 * @return adAddress2
	 */
	public String getAdAddress2()
	{
		return adAddress2;
	}

	/**
	 * @param adAddress2 adAddress2ݒ肵܂B
	 */
	public void setAdAddress2(String adAddress2)
	{
		this.adAddress2 = adAddress2;
	}
}
