/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.notes.store;

import java.util.*;

import jp.co.connectone.eai.notes.user.NotesAccountData;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.IServiceInfo;
import jp.co.connectone.service.IServiceInfoRawData;
import jp.co.connectone.store.*;
import jp.co.connectone.store.client.AddressBookSearchCondition;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;
import lotus.domino.*;

/**
 * @author amanot
 *
 */
public class NotesAddressBookStoreImpl extends EAINotesBase implements IAddressBookStore
{
	public static final IStoreID storeID = new SimpleStoreID(NotesAddressBookStoreImpl.class.getName());
	public static final String storeName = NotesConstants.ADDRESSBOOKSTORE_NAME;
	public String getName() throws Exception
	{
		return storeName;
	}

	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		HashMap<String,Object> h = serviceData.getFieldSet();
		NotesServiceInfo info = new NotesServiceInfo(storeID,storeName);
		String serverAddress = (String)h.get("server");
		String serverAddress2 = (String)h.get("server2");
		if (serverAddress!=null) info.setServerAddress(serverAddress);
		if (serverAddress2!=null) info.setServerAddress2(serverAddress2);
		return info;
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new NotesServiceInfo(storeID,storeName);
	}

	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}


	/* ( Javadoc)
	 * @see jp.co.connectone.store.pim.IAddressBookStore#createNewAddressBookItem(jp.co.connectone.user.IAccountData, java.lang.String, jp.co.connectone.store.pim.IAddressDTO)
	 */
	public IObjectIndex createNewAddressBookItem(IAccountData acc, ISearchDestination dest, IAddressDTO address) throws IncorrectStore, ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("not yet implemented");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.pim.IAddressBookStore#deleteAddressBookItem(jp.co.connectone.user.IAccountData, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public void deleteAddressBookItem(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		// TODO ꂽ\bhEX^u
		throw new UnsupportedOperationException("not yet implemented");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.pim.IAddressBookStore#getAddressBookItem(jp.co.connectone.user.IAccountData, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public IAddressDTO getAddressBookItem(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		IAddressDTO l_objIAddressDTO = null;
		Document l_docAddBook = null;

		super.initSession((NotesAccountData)acc, dest.getDatabase());
		try {
			//Obtain the document corresponig to a Unique id
			l_docAddBook = getDocumentByUNID(oid);

			//Populate the DTO with the Address Book Information
			l_objIAddressDTO = populateIAddressDTO(l_docAddBook);
		}
		catch (Exception e) {
			throw new DataNotFound();
		}
		finally {
			try {
				if (l_docAddBook != null) {
					l_docAddBook.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}
		return l_objIAddressDTO;
	}

	private IAddressDTO populateIAddressDTO(Document objDocAddBook) throws ServerDown, DataNotFound, IncorrectData
	{

		IAddressDTO l_objIAddressDTO = null;
		try {
			String altFullName = objDocAddBook.getItemValueString(NotesConstants.ALT_FULL_NAME);
			if(altFullName == null || "".equals(altFullName)) {
				l_objIAddressDTO = new NotesAddressDTO();
			}
			else {
				NotesDJXAddressDTO djxDTO = new NotesDJXAddressDTO();
				djxDTO.setDjxDisplayName(trimDJXDisplayName(altFullName));
				l_objIAddressDTO = djxDTO;
			}
			//Getting the various infromaton form the address book
			l_objIAddressDTO.setOid(new NotesObjectIndex(objDocAddBook.getUniversalID()));
			l_objIAddressDTO.setTid(objDocAddBook.getUniversalID());
			l_objIAddressDTO.setFirstName(objDocAddBook.getItemValueString(NotesConstants.FIRST_NAME));
			l_objIAddressDTO.setFamilyName(objDocAddBook.getItemValueString(NotesConstants.LAST_NAME));
			l_objIAddressDTO.setDisplayName(getAccountStringFromFullname(objDocAddBook.getItemValueString(NotesConstants.FULL_NAME)));
			l_objIAddressDTO.setEmail(getAccountStringFromFullname(objDocAddBook.getItemValueString(NotesConstants.FULL_NAME))+"@"+objDocAddBook.getItemValueString(NotesConstants.MAIL_DOMAIN));
			l_objIAddressDTO.setCompanyPhone(objDocAddBook.getItemValueString(NotesConstants.COMPANY_PHONE));
			l_objIAddressDTO.setCompanyName(objDocAddBook.getItemValueString(NotesConstants.COMPANY_NAME));
			l_objIAddressDTO.setCompanyAddress(objDocAddBook.getItemValueString(NotesConstants.OFFICE_STATE)+objDocAddBook.getItemValueString(NotesConstants.OFFICE_CITY)+objDocAddBook.getItemValueString(NotesConstants.OFFICE_STREET_ADDRESS)+objDocAddBook.getItemValueString(NotesConstants.OFFICE_NUMBER));
			l_objIAddressDTO.setMobilePhone(objDocAddBook.getItemValueString(NotesConstants.MOBILE_PHONE));
			l_objIAddressDTO.setHomePhone(objDocAddBook.getItemValueString(NotesConstants.HOME_PHONE));
		}
		catch (NullPointerException l_nullExObj) {
			throw new DataNotFound();
		}
		catch (NotesException l_notesExcepObj) {
			throw new ServerDown(l_notesExcepObj.text);
		}
		return l_objIAddressDTO;
		} //end populateIAddressDTO

	private String trimDJXDisplayName(String dixDisplayName) {
		dixDisplayName = dixDisplayName.replaceFirst("CN=", "");
		dixDisplayName = dixDisplayName.replaceFirst("OU=", "");
		dixDisplayName = dixDisplayName.replaceFirst("O=", "");
		return dixDisplayName;
	}
	/* ( Javadoc)
	 * @see jp.co.connectone.store.pim.IAddressBookStore#getAddressBookItems(jp.co.connectone.user.IAccountData, java.lang.String)
	 */
	public IAddressDTO[] getAddressBookItems(IAccountData acc, ISearchDestination dest) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		View l_objViewMailUsers = null;
		Document l_objDocAddBook = null;
		IAddressDTO[] l_objIAddressDTO = null;

		super.initSession((NotesAccountData)acc, dest.getDatabase());
		try {
			//Getting the documnet that contains the mail users list
			l_objViewMailUsers = getView(dest.getFolder());
			//Calculating the total users list

			l_objIAddressDTO = new IAddressDTO[0];
			//Accessing the first document
			l_objDocAddBook = l_objViewMailUsers.getFirstDocument();
			ArrayList<IAddressDTO> list = new ArrayList<IAddressDTO>();
			while (l_objDocAddBook!=null) {
				list.add(populateIAddressDTO(l_objDocAddBook));
				l_objDocAddBook = l_objViewMailUsers.getNextDocument(l_objDocAddBook);
			}
			l_objIAddressDTO = (IAddressDTO[])list.toArray(l_objIAddressDTO);
		}
		catch (NullPointerException l_nullExObj) {
			if (l_objViewMailUsers == null) {
				throw new StoreNotFound(dest.getDatabase().getIndex().toString());
			}
			else {
				throw new DataNotFound();
			}
		}
		catch (NotesException l_notesExObj) {
			Log.debug("NotesException:"+l_notesExObj);
			throw new ServerDown(acc.getServiceInfo().getServerAddress());
		}
		finally {
			try {
				if ((l_objDocAddBook != null) && (l_objDocAddBook != null)) {
					l_objDocAddBook.recycle();
					l_objViewMailUsers.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}

		return l_objIAddressDTO;
	}
	public IAddressDTO[] getAddressBookItemsByFTSearch(IAccountData acc, ISearchDestination dest,String searchByEnglish,String searchByJapanese) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		View l_objViewMailUsers = null;
		IAddressDTO[] l_objIAddressDTO = null;
		int maxDocs = 21;

		super.initSession((NotesAccountData)acc, dest.getDatabase());
		try {
			//Getting the documnet that contains the mail users list
			l_objViewMailUsers = getView(dest.getFolder());

			Set<IAddressDTO> addrSet = FTSearch(l_objViewMailUsers,searchByEnglish,maxDocs);
			if(addrSet.size() < maxDocs) {
				Set<IAddressDTO> addrSetJ = FTSearch(l_objViewMailUsers,searchByJapanese,maxDocs);
				addrSet.addAll(addrSetJ);
			}

			l_objIAddressDTO = new IAddressDTO[0];
			l_objIAddressDTO = (IAddressDTO[])addrSet.toArray(l_objIAddressDTO);
		}
		catch (NullPointerException l_nullExObj) {
			if (l_objViewMailUsers == null) {
				throw new StoreNotFound(dest.getDatabase().getIndex().toString());
			}
			else {
				throw new DataNotFound();
			}
		}
		catch (NotesException l_notesExObj) {
			Log.debug("NotesException:"+l_notesExObj);
			throw new ServerDown(acc.getServiceInfo().getServerAddress());
		}
		finally {
			try {
				if (l_objViewMailUsers != null) {
					l_objViewMailUsers.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}

		return l_objIAddressDTO;
	}
	private Set<IAddressDTO> FTSearch(View view,String query,int maxDocs) throws NotesException, HandleException{
		if(query == null) {
			return new TreeSet<IAddressDTO>();
		}
		query = query.trim();
		if("".equals(query)) {
			return new TreeSet<IAddressDTO>();
		}
		
		Document l_objDocAddBook = null;
		Set<IAddressDTO> addrSet = new TreeSet<IAddressDTO>(new IAddressDTOComparator(IAddressDTOComparator.DISPLAYNAME,false));
		try {
			view.FTSearch(query, maxDocs);
			l_objDocAddBook = view.getFirstDocument();
			while(l_objDocAddBook != null) {
				IAddressDTO l_objNotesAddressDTO = populateIAddressDTO(l_objDocAddBook);
				addrSet.add(l_objNotesAddressDTO);
				l_objDocAddBook = view.getNextDocument(l_objDocAddBook);
			}
		} 
		catch (NotesException l_notesExObj) {
			throw l_notesExObj;
		}
		finally {
			if (l_objDocAddBook != null) {
				l_objDocAddBook.recycle();
			}
		}
		return addrSet;
	}
	/* ( Javadoc)
	 * @see jp.co.connectone.store.pim.IAddressBookStore#searchAddressBook(jp.co.connectone.user.IAccountData, java.lang.String, jp.co.connectone.store.SearchConditionCollection)
	 */
	public IAddressDTO[] searchAddressBook(IAccountData acc, ISearchFormula form) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		Iterator<ISearchConditionElement> it = form.getSearchConditions().iterator();
		SearchCondition cond = null;
		ISearchType condType = null;
		String name = "";
		String searchByEnglish = "";
		String searchByJapanese = "";
		String prefix = "";
		String suffix = "";
		while (it.hasNext()) {
			cond = (SearchCondition)it.next();
			condType = cond.getType();
			if (condType==AddressBookSearchCondition.CONDITION.TYPE_FIRSTNAME) {
				name = (String)cond.getValue();
				searchByEnglish = "\"" + name + "*\"";
				searchByJapanese = "\"" + name + "\"";
				prefix = name;
			}
			if (condType==AddressBookSearchCondition.CONDITION.TYPE_DISPLAYNAME) {
				name = (String)cond.getValue();
				searchByEnglish = "\"*" + name + "*\"";
				searchByJapanese = "\"" + name + "\"";
			}
			if (condType==AddressBookSearchCondition.CONDITION.TYPE_LASTNAME) {
				name = (String)cond.getValue();
			}
		}
		
		IAddressDTO[] addrs = getAddressBookItemsByFTSearch(acc,form.getDest(),searchByEnglish,searchByJapanese);
		List<IAddressDTO> ar = narrowAddressBookItems(addrs,prefix,suffix);
		addrs = new IAddressDTO[0];
		return (IAddressDTO[])ar.toArray(addrs);
	}
	private List<IAddressDTO> narrowAddressBookItems(IAddressDTO[] addresses,String prefix, String suffix) {
		List<IAddressDTO> addressList = new ArrayList<IAddressDTO>();
		for (int i=0;i<addresses.length;i++) {
			NotesAddressDTO addr = null;
			if(addresses[i] instanceof NotesAddressDTO) {
				addr = ((NotesAddressDTO)addresses[i]);
			}
			else {
				continue;
			}
			if (addr.commonNameStartsWith(prefix) && addr.commonNameEndsWith(suffix)) {
				addressList.add(addresses[i]);
			}
			else {
				continue;
			}
		}
		return addressList;
	}
	/* ( Javadoc)
	 * @see jp.co.connectone.store.pim.IAddressBookStore#updateAddressBookItem(jp.co.connectone.user.IAccountData, java.lang.String, jp.co.connectone.store.pim.IAddressDTO)
	 */
	public IObjectIndex updateAddressBookItem(IAccountData acc, ISearchDestination dest, IAddressDTO address) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		Document l_docObj = null;
		IObjectIndex rc = null;

		super.initSession((NotesAccountData)acc, dest.getDatabase());
		try {
			l_docObj = getDocumentByUNID(address.getOid());
			//Incorporate Changes to the Document
			l_docObj.replaceItemValue(NotesConstants.FIRST_NAME, address.getFirstName());
			l_docObj.replaceItemValue(NotesConstants.LAST_NAME, address.getFamilyName());
			//Save the Document
			l_docObj.save();
			rc = new NotesObjectIndex(l_docObj.getUniversalID()); 
		}
		catch (NotesException l_notesExcepObj) {
			if (l_docObj != null && address != null) {
				throw new UpdateFailed();
			}
			else {
				throw new ServerDown(acc.getServiceInfo().getServerAddress());
			}
		}
		catch (NullPointerException l_objNullPointerException) {
			if (l_docObj != null) {
				throw new IncorrectData("Invalid Details to Update Address Book!");
			}
			else {
				throw new DataNotFound();
			}
		}
		finally {
			try {
				if (l_docObj != null) {
					l_docObj.recycle();
				}
			}
			catch (NotesException e) {
				// error level WARN
			}
			super.recycleObjects();
		}
		return rc;
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#delete(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		deleteAddressBookItem(acc,dest,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getAllDatas(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getAddressBookItems(acc,dest);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#getFolderList(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String)
	 */
	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		throw new UnsupportedOperationException("get folder list on NotesAddressBook not supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#read(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IObjectIndex)
	 */
	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		return getAddressBookItem(acc,dest,oid);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#search(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.SearchConditionCollection)
	 */
	public IRecordObject[] search(IAccountData acc, ISearchFormula form) throws Exception
	{
		return searchAddressBook(acc,form);
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#searchByDate(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, java.util.Date)
	 */
	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date searchDate) throws Exception
	{
		throw new UnsupportedOperationException("searchByDate on NotesAddressBook not supported.");
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.store.IStore#write(jp.co.connectone.user.IAccountData, java.lang.String, java.lang.String, jp.co.connectone.store.IRecordObject)
	 */
	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject address) throws Exception
	{
		IObjectIndex rc = null;
		if (address.isNew()) {
			rc = createNewAddressBookItem(acc,dest,(IAddressDTO)address);
		}
		else {
			rc = updateAddressBookItem(acc,dest,(IAddressDTO)address);
		}
		return rc;
	}

	/* ( Javadoc)
	 * @see jp.co.connectone.eai.notes.store.EAINotesBase#getElmentName()
	 */
	protected String getElmentName()
	{
		// TODO ꂽ\bhEX^u
		return null;
	}

	public ISearchDestination getPresetDestination(IAccountData acc, int type) throws IncorrectData, HandleException
	{
		IFolderIndex folder = null;
		IDatabaseIndex db = null;
		switch(type) {
		case IAddressBookStore.DEST_TYPE_DEFAULT_SERVER_ADDRESSBOOK:
			db = new NotesDatabaseIndex(NotesConstants.FILE_NAME_GROBAL_ADDRESSBOOK);
			folder = new NotesFolderIndex(NotesConstants.VIEW_NAME_GROBAL_ADDRESSBOOK);
			break;
		case IAddressBookStore.DEST_TYPE_DEFAULT_ADDRESSBOOK_FOLDER:
			initSession((NotesAccountData)acc);
			db = new NotesDatabaseIndex(getFilePath());
			folder = new NotesFolderIndex(NotesConstants.VIEW_NAME_PERSONAL_ADDRESSBOOK);
			break;
		}
		return (ISearchDestination)new BasicSearchDestination(db,folder);
	}

}
