/**
 * Copyright (C) 2006-2011 Takanori Amano, Amax Inc., and Connectone Co.,Ltd.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package jp.co.connectone.eai.exchews.store;

import java.util.*;

import javax.naming.AuthenticationException;
import javax.naming.NamingEnumeration;
import javax.naming.directory.*;

import jp.co.connectone.common.ADTool;
import jp.co.connectone.common.MailTool;
import jp.co.connectone.eai.exchews.ews.*;
import jp.co.connectone.exception.*;
import jp.co.connectone.log.Log;
import jp.co.connectone.service.IServiceInfo;
import jp.co.connectone.service.IServiceInfoRawData;
import jp.co.connectone.store.*;
import jp.co.connectone.store.client.AddressBookSearchCondition;
import jp.co.connectone.store.pim.*;
import jp.co.connectone.user.IAccountData;
import jp.co.connectone.user.PasswordBasedAccountDataImpl;

import com.microsoft.schemas.exchange.services._2006.types.*;

public class ExchewsAddressBookStoreImpl extends EAIExchewsBase implements IAddressBookStore
{
	public static final IStoreID storeID = new SimpleStoreID(ExchewsAddressBookStoreImpl.class.getName());
	public static final String storeName = "Exchange EWS addressbook store";
	protected static final String itemType = "IPM.Contact";
	protected String adAddress;
	protected String adAddress2;

	public ExchewsAddressBookStoreImpl()
	{
		super();
		includeMimeContent=false;
	}

	public IServiceInfo getServiceInfo() throws Exception
	{
		return new ExchewsServiceInfo(storeID, storeName);
	}

	@Override
	protected void setAccountData(IAccountData acc) throws HandleException
	{
		ExchewsServiceInfo si = (ExchewsServiceInfo) acc.getServiceInfo();
		adAddress = si.getAdAddress();
		adAddress2 = si.getAdAddress2();

		super.setAccountData(acc);
	}

	@Override
	protected UnindexedFieldURIType getDateField()
	{
		return UnindexedFieldURIType.ITEM_DATE_TIME_CREATED;
	}

	@Override
	protected TargetFolderIdType getTargetFolder()
	{
		TargetFolderIdType target = new TargetFolderIdType();
		DistinguishedFolderIdType id = new DistinguishedFolderIdType();
		id.setId(addressbookElement);
		target.setDistinguishedFolderId(id);
		return target;
	}

	@Override
	public IObjectIndex createNewAddressBookItem(IAccountData acc, ISearchDestination dest, IAddressDTO addressBookDTO) throws IncorrectStore, ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		throw new UnsupportedOperationException("not yet implemented");
	}

	@Override
	public void deleteAddressBookItem(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws ServerDown, NoSuchRights, DataNotFound, IncorrectData, HandleException
	{
		throw new UnsupportedOperationException("not yet implemented");
	}

	@Override
	public IAddressDTO getAddressBookItem(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		IAddressDTO l_objIAddressDTO = null;

Log.debug("getAddressBookItem::oid=" + oid);

		this.setAccountData(acc);
		if (dest.getDatabase() != null) {
			SearchConditionCollection conds = new SearchConditionCollection();
			AddressBookSearchCondition cond = new AddressBookSearchCondition(AddressBookSearchCondition.CONDITION.TYPE_ACCOUNTNAME,oid.getIndex().toString());
			conds.add(cond);
			List<HashMap<String, String>> col = searchUserFromActiveDirectory(conds);
			if (col.size()==0) {
				throw new DataNotFound("account that has sAMAAccountName="+oid.getIndex().toString()+" not found.");
			}
			l_objIAddressDTO = populateAddressDTOG(col.get(0));
		}
		else {
			ContactItemType contact = (ContactItemType) getItem(dest,(ItemIdType) oid.getIndex());
			ExchewsAddressDTO rc = null;
			try {
				rc = ExchewsAddressDTO.convertFromContactItemType(contact);
			}
			catch (HandleException he) {
				throw he;
			}
			catch (Exception e) {
				Log.error("address convert error", e);
				throw new HandleException(e);
			}
			l_objIAddressDTO = rc;
		}
		return l_objIAddressDTO;
	}

	@Override
	public IAddressDTO[] getAddressBookItems(IAccountData acc, ISearchDestination dest) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		throw new UnsupportedOperationException("not yet implemented");
	}

	@Override
	public IAddressDTO[] searchAddressBook(IAccountData acc, ISearchFormula conds) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		if (conds.getDest().getDatabase() != null) {
			Log.debug("search from active directory");
			return searchActiveDirectory(acc, conds.getSearchConditions());
		}
		else {
			return searchPersonalAddressbook(acc, conds);
		}
	}

	@Override
	public IObjectIndex updateAddressBookItem(IAccountData acc, ISearchDestination dest, IAddressDTO addressBookDTO) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		throw new UnsupportedOperationException("not yet implemented");
	}

	@Override
	public void delete(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		throw new UnsupportedOperationException("not yet implemented");
	}

	@Override
	public IRecordObject[] getAllDatas(IAccountData acc, ISearchDestination dest) throws Exception
	{
		return getAddressBookItems(acc, dest);
	}

	@Override
	public FolderMetadata[] getFolderList(IAccountData acc, ISearchDestination dest) throws Exception
	{
		super.setAccountData(acc);

		ExchewsFolderIndex idx = (ExchewsFolderIndex) dest.getFolder();

		BaseFolderIdType src = (BaseFolderIdType) idx.getIndex();

		List<BaseFolderType> folders = super.getAllSubFolders(src,0);

		ArrayList<ExchewsFolderMetadata> rc = new ArrayList<ExchewsFolderMetadata>();

		ExchewsFolderMetadata a[] = new ExchewsFolderMetadata[0];
		for (BaseFolderType folder : folders) {
			rc.add(ExchewsFolderMetadata.convertFromEWSFolderType((FolderType) folder));
		}

		return rc.toArray(a);
	}

	@Override
	public String getName() throws Exception
	{
		return storeName;
	}

	@Override
	public ISearchDestination getPresetDestination(IAccountData acc, int type) throws IncorrectData, HandleException
	{
		super.setAccountData((PasswordBasedAccountDataImpl)acc);
		IFolderIndex folder = null;
		IDatabaseIndex db = null;
		switch (type) {
		case IAddressBookStore.DEST_TYPE_DEFAULT_ADDRESSBOOK_FOLDER:
			DistinguishedFolderIdType src = new DistinguishedFolderIdType();
			src.setId(addressbookElement);
			folder = (IFolderIndex) populateFolder(new ExchewsFolderIndex(src)).getOid();
			break;
		case IAddressBookStore.DEST_TYPE_DEFAULT_SERVER_ADDRESSBOOK:
			db= (IDatabaseIndex)populateADDatabase().getOid();
			break;
		default:
			throw new IncorrectData("illegal folderType");
		}
		
		return (ISearchDestination)new BasicSearchDestination(db,folder);
	}
	
	protected DatabaseMetadata populateADDatabase()
	{
		DatabaseMetadata db = new DatabaseMetadata();
		db.setServerName(this.adAddress);
		IDatabaseIndex oid = new ExchewsDatabaseIndex("connector_exchews:ActiveDirectory."+this.adAddress);
		db.setOid(oid);
		HashMap<String,Object> h = db.getFieldSet();
		h.put("oid",oid);
		
		return db;
	}

	@Override
	public IServiceInfo getServiceInfo(IServiceInfoRawData serviceData) throws Exception
	{
		return super.getServiceInfo(storeID, storeName, serviceData);
	}

	@Override
	public IStoreID getStoreID() throws Exception
	{
		return storeID;
	}

	@Override
	public IRecordObject read(IAccountData acc, ISearchDestination dest, IObjectIndex oid) throws Exception
	{
		return getAddressBookItem(acc, dest, oid);
	}

	@Override
	public IRecordObject[] search(IAccountData acc, ISearchFormula conds) throws Exception
	{
		return searchAddressBook(acc, conds);
	}

	@Override
	public IRecordObject[] searchByDate(IAccountData acc, ISearchDestination dest, Date date) throws Exception
	{
		throw new UnsupportedOperationException("not yet implemented");
	}

	@Override
	public IObjectIndex write(IAccountData acc, ISearchDestination dest, IRecordObject data) throws Exception
	{
		throw new UnsupportedOperationException("not yet implemented");
	}

	private IAddressDTO populateAddressDTOG(HashMap<String, String> itemList)
	{
		IAddressDTO l_objIAddressDTO = new BasicAddressDTO();
		String tid = itemList.get("userPrincipalName");
		if (tid == null) tid = "";
		l_objIAddressDTO.setTid(tid.replaceAll("@", "."));
		l_objIAddressDTO.setOid(new ExchewsObjectIndex(itemList.get("sAMAccountName")));
		l_objIAddressDTO.setFirstName(itemList.get("givenName"));
		l_objIAddressDTO.setFamilyName(itemList.get("sn"));
		l_objIAddressDTO.setEmail(itemList.get("mail"));
		l_objIAddressDTO.setCompanyPhone(itemList.get("telephoneNumber"));
		l_objIAddressDTO.setHomePhone(itemList.get("homePhone"));
		l_objIAddressDTO.setMobilePhone(itemList.get("mobile"));
		l_objIAddressDTO.setCompanyName(itemList.get("company"));
		l_objIAddressDTO.setDisplayName(itemList.get("displayName"));
		Log.debug("displayName=" + itemList.get("displayName"));

		String st = itemList.get("st");
		if (st == null) st = "";
		String l = itemList.get("l");
		if (l == null) l = "";
		String streetAddress = itemList.get("streetAddress");
		if (streetAddress == null) streetAddress = "";
		l_objIAddressDTO.setCompanyAddress(st + l + streetAddress);

		return l_objIAddressDTO;
	}

	private IAddressDTO[] searchActiveDirectory(IAccountData acc, SearchConditionCollection conds) throws IncorrectStore, ServerDown, NoSuchRights, StoreNotFound, DataNotFound, IncorrectData, HandleException
	{
		setAccountData(acc);

		List<HashMap<String, String>> col = searchUserFromActiveDirectory(conds);
		int len = col.size();
		Log.debug("results(searchUserFromActiveDirectory):" + len);
		IAddressDTO[] l_IAddressDTOArray = new IAddressDTO[0];
		ArrayList<IAddressDTO> ar = new ArrayList<IAddressDTO>();
		for (int i = 0; i < len; i++) {
			IAddressDTO dto = null;
			try {
				dto = populateAddressDTOG(col.get(i));
			}
			catch (Exception e) {
				Log.error("IAddressDTO populate error:Igunore address data no " + i, e);
				continue;
			}
			if(dto.getOid().getIndex() != null) {
				ar.add(dto);
			}
		}
		l_IAddressDTOArray = ar.toArray(l_IAddressDTOArray);

		return l_IAddressDTOArray;
	}

	protected List<HashMap<String, String>> searchUserFromActiveDirectory(SearchConditionCollection conds) throws NullPointerException, ServerDown, HandleException
	{
		if (userID == null) throw new NullPointerException("set userID first");
		if (password == null) throw new NullPointerException("set password first");
		if (adAddress == null) throw new NullPointerException("set adAddress first");
		if (domain == null) throw new NullPointerException("set domain first");
		Iterator<ISearchConditionElement> it = conds.iterator();
		ISearchCondition cond = null;
		String whereStr = "";
		while (it.hasNext()) {
			ISearchConditionElement elem = it.next();
			cond = (ISearchCondition) elem;
			Log.debug("condition:" + cond.getType() + ":" + cond.getValue());
		}
		if (cond.getType() == AddressBookSearchCondition.CONDITION.TYPE_DISPLAYNAME) {
			whereStr = " (displayName=" + cond.getValue() + "*) ";
		}
		else if (cond.getType() == AddressBookSearchCondition.CONDITION.TYPE_FIRSTNAME) {
			whereStr = " (sn=*" + cond.getValue() + "*) ";
		}
		else if (cond.getType() == AddressBookSearchCondition.CONDITION.TYPE_LASTNAME) {
			whereStr = " (givenName=*" + cond.getValue() + "*) ";
		}
		else if (cond.getType() == AddressBookSearchCondition.CONDITION.TYPE_ACCOUNTNAME) {
			whereStr = " (sAMAccountName=" + cond.getValue() + ") ";
		}
		else {
			throw new HandleException("Illegal search type");
		}
		Log.debug("whereStr:" + whereStr);
		ADTool ad = new ADTool();
		NamingEnumeration<SearchResult> results = null;
		ArrayList<HashMap<String, String>> lists = new ArrayList<HashMap<String, String>>();
		try {
			results = ad.getUserDatas(userID, password, adAddress, adAddress2, domain, whereStr);
			while (results.hasMoreElements()) {
				Log.debug("LOOP FIRST LINE");
				SearchResult aResult = (SearchResult) results.nextElement();
				Log.debug("name: " + aResult.getName());
				Attributes attrs = aResult.getAttributes();
				NamingEnumeration<? extends Attribute> ae = attrs.getAll();
				HashMap<String, String> list = new HashMap<String, String>();
				Log.debug("loop IN");
				while (ae.hasMoreElements()) {
					Attribute attr = (Attribute) ae.nextElement();
					String strAttrID = attr.getID();
					NamingEnumeration<?> e = attr.getAll();
					String value = "";
					while (e.hasMoreElements()) {
						Object o = e.nextElement();
						Log.trace("class of object(" + strAttrID + "):" + o.getClass());
						if (o instanceof String) {
							value = (String) o;
						}
					}
					Log.trace(strAttrID + ": " + value);
					list.put(strAttrID, value);
				}
				Log.debug("loop OUT");
				lists.add(list);
				list = null;
				Log.debug("LOOP LAST LINE");
			}
			Log.debug("no of accounts:" + lists.size());
			results.close();
		}
		catch (AuthenticationException ae) {
			Log.error("searchUserFromActiveDirectory:Authentication failed(" + ae.getMessage() + ")", ae);
			throw new NoSuchRights("searchUserFromActiveDirectory:Authentication failed(" + ae.getMessage() + ")");
		}
		catch (Exception e) {
			Log.error("", e);
			throw new ServerDown(e.getMessage());
		}

		return lists;
	}

	protected SearchConditionCollection convertCondition(SearchConditionCollection searchConditions) throws IncorrectData
	{
		if (searchConditions==null) return null;
		if (searchConditions.size()!=1) {
			throw new IncorrectData("convertCondition currently accepts single condition only.");
		}
		ISearchConditionElement cond = searchConditions.iterator().next();
		if (!(cond instanceof AddressBookSearchCondition)) {
			throw new IncorrectData("convertCondition accepts AddressBookSearchCondition only.");
		}
		EWS_CONTAINS_SearchConditionCollection rc = new EWS_CONTAINS_SearchConditionCollection();

		AddressBookSearchCondition adc = (AddressBookSearchCondition)cond;
		ISearchType type = adc.getType();
		String value = (String) adc.getValue();
		UnindexedFieldURIType field=null;
		if (AddressBookSearchCondition.CONDITION.TYPE_DISPLAYNAME==type) {
			field = UnindexedFieldURIType.CONTACTS_DISPLAY_NAME;
		}
		else if (AddressBookSearchCondition.CONDITION.TYPE_FIRSTNAME==type) {
			field = UnindexedFieldURIType.CONTACTS_SURNAME;
		}
		else if (AddressBookSearchCondition.CONDITION.TYPE_LASTNAME==type) {
			field = UnindexedFieldURIType.CONTACTS_GIVEN_NAME;
		}
		else if (AddressBookSearchCondition.CONDITION.TYPE_ACCOUNTNAME==type) {
			throw new IncorrectData("CONTACT doesn't contain AccountName.");
		}
		EWSSearchCondition ews = new EWSSearchCondition(field, EWSSearchType.TYPE_STRING, value, EWSSearchOperator.OP_CONTAINS);
		rc.add(ews);

		return rc;
	}

	protected IAddressDTO[] searchPersonalAddressbook(IAccountData acc, ISearchFormula conds) throws HandleException
	{
		super.setAccountData(acc);
		IAddressDTO[] dtos = new IAddressDTO[1];
		List<? extends ItemType> messages = new ArrayList<MessageType>();
		try {
			messages = super.search(conds.getDest(), (List<String>)conds.getFieldsToGet(), convertCondition(conds.getSearchConditions()));
		}
		catch (HandleException he) {
			throw he;
		}
		catch (Exception e) {
			Log.error("", e);
			throw new HandleException(e);
		}
		ArrayList<ExchewsAddressDTO> rc = new ArrayList<ExchewsAddressDTO>();

		MailTool tool = new MailTool();
		tool.setAllowNullOrBlankMailAddress(false);
		for (ItemType message : messages) {
			try {
				ExchewsAddressDTO dto = ExchewsAddressDTO.convertFromContactItemType((ContactItemType)message);
				
				MailAddress address = new MailAddress(dto.getEmail());
				if (tool.checkAddressFormat(address)==false)continue;// ignore values other than mail address.
				rc.add(dto);
				if (message.isHasAttachments().booleanValue()) {
//					dto.convertAndSetEWSAttachments(getAttachments(message));
				}
			}
			catch (Exception e) {
				Log.debug("error on converting MessageType to ExchewsMailDTO. skip one data..", e);
			}
		}

		return rc.toArray(dtos);
	}
}
