#ifndef RLOG_HH_
#define RLOG_HH_

#ifdef _OPENMP
	#include <omp.h>
#endif
#include <vector>
#include <string>

using namespace std;


typedef int zListnerID;

/**********************************************************************************/
/**
 * 簡易ログクラス
 */
/**
 * エラーレベル
 */
enum zLevel
{
    zDebug,                 ///< デバッグ
    zInfo,                  ///< インフォ
    zWarn,                  ///< ワーニング
    zError,                 ///< エラー
    zFatal,                 ///< フェイタル
};


class CRLog
{
public:
    /** Logリスナークラス
     * 各プログラムからメッセージを送付された受信するクラス。
     * このクラスを継承し、ファイルへの出力等を記載してください。
     */
    class Listner
    {
    public:
       	/// デストラクタ
       	virtual ~Listner() {};
       	/// メッセージ受信
       	virtual void log(const zLevel& cLevel, ///< [in]:エラーレベル
                         const string& cpStr ///< [in]:メッセージ文字列
                       ) = 0;

       	/// メッセージ受信（呼び出し元のスレッド番号により処理を変える。）
       	virtual void log(const zLevel& cLevel, ///< [in]:エラーレベル
       					 const int& threadID, ///< [in]:スレッド番号
                         const string& cpStr ///< [in]:メッセージ文字列
                       ) = 0;

       	/** クラス識別子
         * removeしたい場合はこのIDを指定して、削除するクラスを指定する。
         * removeによる削除が不要の場合は、OverWrite不要。
         */
       	virtual zListnerID putId(void) const = 0;
    };

    /**
     * ファイル名、行番号 展開用ヘルパークラス
     */
    class set
    {
    private:
        zLevel m_Level;
        int threadID;
    public:
        set(const zLevel& cLevel) : m_Level(cLevel), threadID(-1) {};
        set(const zLevel& cLevel, const int& tID) : m_Level(cLevel), threadID(tID) {};
        inline void log(const string& cpStr);
    };

    static bool append(Listner* cpPtr);
    static bool remove(const zListnerID& cId);
    static void clear(void);

    ~CRLog(void);
    static CRLog* getInstance(void);
    void log(const zLevel& cLevel, const string& cpStr);
    void log(const zLevel& cLevel, const int& threadID, const string& cpStr);

private:
    vector<Listner*> m_ListnerAry;

    CRLog(void);
    CRLog(const CRLog&);
    bool append_(Listner* cpPtr);
    bool remove_(const zListnerID& cId);
    void clear_(void);
};

/**
 * メッセージの送信
  * printf 形式の文字列とパラメータを受け取り、メッセージチェーンに流す
 */
inline void CRLog::set::log(const string& cpStr)         ///< [in]:printf形式のパラメータ
{
	if( threadID < 0 )
	{
		CRLog::getInstance()->log(m_Level, cpStr);
	}
	else
	{
		CRLog::getInstance()->log(m_Level, threadID, cpStr);
	}
};

#define ZLOG_INFO CRLog::set(zInfo).log
#define ZLOG_WARN CRLog::set(zWarn).log
#define ZLOG_ERROR CRLog::set(zError).log
#define ZLOG_FATAL CRLog::set(zFatal).log

#ifdef _OPENMP
#define ZLOG_INFO_THREAD CRLog::set(zInfo, omp_get_thread_num()).log
#define ZLOG_WARN_THREAD CRLog::set(zWarn, omp_get_thread_num()).log
#define ZLOG_ERROR_THREAD CRLog::set(zError, omp_get_thread_num()).log
#define ZLOG_FATAL_THREAD CRLog::set(zFatal, omp_get_thread_num()).log
#endif

#endif
