package osm.jp;
import hayashi.yuu.tools.logger.LoggerFactory;

import java.io.*;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.logging.Logger;

import jp.co.areaweb.tools.database.*;
import jp.co.areaweb.tools.csv.*;

public class ConvAddr {

	String filter = "";
	String urlStr = "";

	public static final boolean DB_INIT = false;

	// 近くの類似ノードを探す範囲（基準ノードを中心としたNEER×２ｍ四方の領域
	static final int NEER = 2000;
	static boolean nocheck = true;
	static Logger logger = LoggerFactory.getInstance();
	
	public static String version = "";

	/**
	 * メイン
	 *
	 *	java -cp .:ConvAddr.jar:hayashi_0225.jar:hsqldb_2.2.9.jar osm.jp.ConvAddr <option> <import file>
	 *		OPTION: -nocheck	OSMデータ上に既存ノードが存在するかどうかをチェックしない
	 *		OPTION: -check	OSMデータ上に既存ノードが存在するかどうかをチェックする
	 *	
	 *		IMPORT FILE : インポートファイルが指定されている場合には、インポートファイルのみインポートする
	 *						インポートファイル指定が省略されている場合には、カレントディレクトリ
	 * @throws Exception */
	public static void main(String[] args) throws Exception
	{
		int index = 0;
		if (args.length > index) {
			if (args[index].equals("-check")) {
				ConvAddr.nocheck = false;
				index++;
			}
			if (args[index].equals("-nocheck")) {
				ConvAddr.nocheck = true;
				index++;
			}
		}

		/**
		 * アプリケーション [ConvAddr]
		 * > java -jar ConvAddr.jar <オプション>　
		 * 	オプション： -exp	実行する直前にデータベースを初期化する（省略可能）
		 */
		File dbdir = new File("database");
		if (!dbdir.isDirectory()) {
			dbdir.mkdir();
		}

		Connection con = DatabaseTool.openDb("database");
		DbAddr.create(con);
		
		long count = DbAddr.clear(con);
	    logger.info("'ISJ_CHO'から "+ count +" 件のデータを削除しました。");

		try {
			// CSVファイルを読み取ってDB.table.ISJ_CHO に格納する
			int fcounter = 0;
			if (args.length > index) {
				File iFile = new File(args[index]);
				fcounter++;
				inputFile(con, iFile);
			}
			else {
				File dir = new File(".");
				File[] files = dir.listFiles();
				for (File iFile : files) {
					if (checkFile(iFile)) {
						logger.info("ファイル '"+ iFile.getName() +"' を読み込みます。");
						fcounter++;
						inputFile(con, iFile);
					}
				}
			}
			logger.info("["+ fcounter +"]つのファイルをインポートしました。");
			
			// ローカルデータベース内の情報を出力する
			if (fcounter > 0) {
				outputDb(con, "ISJ_CHO");
			}
		}
		finally {
			DatabaseTool.closeDb(con);
		}
	}

	static String[] shiftArgs(String[] args) {
		String[] values = new String[args.length - 1];
		for (int i=1; i < args.length; i++) {
			values[i - 1] = new String(args[i]);
		}
		return values;
	}

	/**
	 *
	 * ソースファイルを読み取ってローカルベータベースへ記録する
	 * 
	 * 	- ソースファイル名と同じ名前のディレクトリを作成
	 *  - OSMファイル用のディスクリプタを生成。XMLのコンテンツは
	 *  	--> showNodes()
	 *  
	 * 	0: "都道府県コード",		prefecturecode	VARCHAR(10)
	 * 	1: "都道府県名",			prefecture			VARCHAR(32)
	 * 	2: "市区町村コード",		municipalitycode	VARCHAR(20)
	 * 	3: "市区町村名",			municipality 		VARCHAR(128)
	 * 	4: "大字町丁目コード",	namecode	VARCHAR(20)
	 * 	5: "大字町丁目名",		name VARCHAR(128)
	 * 	6: "緯度",				lat DOUBLE
	 * 	7: "経度",				lon DOUBLE
	 * 	8: "原典資料コード",
	 * 	9: "大字・字・丁目区分コード"		typecode	INT
	 * 		1 = 大字 = quarter
	 * 		2 = 字 = neighbourhood
	 * 		3 = 丁目 = neighbourhood
	 * 		4 = 不明 = locality
	 * 
	 * @param con
	 * @param iFile				source file
	 * @throws Exception 
	 */
	public static void inputFile (Connection con, File iFile) throws Exception {
		int iCounter = 0;

		iCounter = 0;
		CsvFile csv = new CsvFile(iFile);
		csv.load();
		
		for (int iRec = 0 ; iRec < csv.size() ; iRec++){
			CsvRecord csvRec = csv.get(iRec);
			if (iRec == 0) {
				continue;		// 見出し行は読み飛ばす
			}
			
			// CSVレコードを'table.ISJ_CHO'に書き込む
			iCounter += DbAddr.insert(con, csvRec);
		}
		
		logger.info("地名数["+ iCounter +"]");
	}


	/**
	 * ローカルデータベース内の情報を出力する
	 * @param con
	 * @param iCode
	 * @throws IOException
	 * @throws SQLException
	 */
	public static void outputDb(Connection con, String iCode) throws IOException, SQLException {
		SimpleDateFormat timeStampFmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
		String timeStampStr = timeStampFmt.format(new Date(Calendar.getInstance().getTimeInMillis()));
		File dir = new File(iCode);
		dir.mkdir();

		File osmFile;
		File gpxFile;
		
		BufferedWriter ow = null;
		BufferedWriter gw = null;
		BufferedWriter hw = null;
		BufferedWriter ww = null;

		// HTML header
		File htmlFile = new File(iCode  +".html");
		hw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(htmlFile), "UTF-8"));
		hw.write("<!DOCTYPE html>");
		hw.newLine();
		hw.write("<html><head><meta charset=\"utf-8\" /></head>");
		hw.newLine();
		hw.write("<body><table border='1'>");
		hw.newLine();
		hw.write("<tr>");
		hw.write("<td>都道府県コード</td>");
		hw.write("<td>都道府県名</td>");
		hw.write("<td>市区町村コード</td>");
		hw.write("<td>市区町村名</td>");
		hw.write("<td>osm</td>");
		hw.write("<td>gpx</td>");
		hw.write("<td>地名数</td>");
		hw.write("<td>既存OSMデータ</td>");
		hw.write("<td>パラメータ</td>");
		hw.write("</tr>");
		hw.newLine();

		// Wiki header
		File wikiFile = new File(iCode +".wiki.txt");
		ww = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(wikiFile), "UTF-8"));
		ww.write("= マッピング状況 =");
		ww.newLine();
		ww.newLine();
		
		String maeCode = "";
		String maePre = "";
		boolean firstData = true;
		boolean firstPre = true;
		
		int dataCount = 0;
		double maxLat = 0.0;
		double minLat = 180.0;
		double maxLon = 0.0;
		double minLon = 180.0;
		
		PreparedStatement ps7 = con.prepareStatement("SELECT prefecturecode,prefecture,municipalitycode,municipality,namecode,name,lat,lon,typecode FROM ISJ_CHO ORDER BY municipalitycode");
		ResultSet rset7 = ps7.executeQuery();
		while (rset7.next()) {
			String prefecturecode = rset7.getString(1);
			String prefecture = rset7.getString(2);
			String municipalitycode = rset7.getString(3);
			String municipality = rset7.getString(4);
			String namecode = rset7.getString(5);
			String name = rset7.getString(6);
			double lat = rset7.getDouble(7);
			double lon = rset7.getDouble(8);
			int typecode = rset7.getInt(9);
			
			//--------------------------------------------
			//	Wiki見出し２: 都道府県コード
			//------------
			if (!maePre.equals(prefecturecode)) {
				if (firstPre == false) {
					ww.write("|}");
					ww.newLine();
					ww.newLine();
				}
				firstPre = false;
				
				ww.write("=== "+ prefecture +" ===");
				ww.newLine();
				ww.newLine();
				ww.write("{| class=\"wikitable sortable\" style=\"table-layout: fixed; width: 100%\"");
				ww.newLine();
				ww.write("!style=\"width: 100px\"| 市区町村");
				ww.newLine();
				ww.write("!class=\"unsortable\" style=\"width: 152px\"| 編集状況");
				ww.newLine();
				ww.write("!class=\"unsortable\" style=\"width: 80px\"| 地名数");
				ww.newLine();
				ww.write("!class=\"unsortable\"| 備考");
				ww.newLine();

				maePre = new String(prefecturecode);
			}

			//--------------------------------------------
			//	市区町村名 毎
			//------------
			if (!maeCode.equals(municipalitycode)) {
				if (firstData) {
					firstData = false;
				}
				else {
					// OSM file footer
					ow.write("</osm>");
					ow.newLine();
					ow.close();

					// GPX file footer
					gw.write("</gpx>");
					gw.newLine();
					gw.close();

					// HTML list
					hw.write("<form action=\"http://overpass-api.de/api/interpreter\" method=\"post\" accept-charset=\"UTF-8\">");
					hw.write("<td>"+ dataCount +"</td>");
					hw.write("<td><input type=\"submit\" value=\"Download\"/></td>");
					hw.write("<td><textarea name=\"data\" rows=\"1\" cols=\"40\"><bbox-query s=\""+ (minLat - 0.01D) +"\" n=\""+ (maxLat+0.01D) +"\" w=\""+ (minLon - 0.01D) +"\" e=\""+ (maxLon + 0.01D) +"\"/><query type=\"node\"><item/><has-kv k=\"place\"/></query><print/></textarea></td>");
					hw.write("</form>");
					hw.write("</tr>");
					hw.newLine();
					
					ww.write("|| "+ dataCount +" ");			// 地名数
					ww.write("|| ");								// 備考
					ww.newLine();
				}
				
				osmFile = new File(dir, municipalitycode + ".osm");
				gpxFile = new File(dir, municipalitycode + ".gpx");

				// index file header
				hw.write("<tr>");
				hw.write("<td>"+ prefecturecode +"</td>");
				hw.write("<td>"+ prefecture +"</td>");
				hw.write("<td>"+ municipalitycode +"</td>");
				hw.write("<td>"+ municipality +"</td>");
				hw.write("<td><a href='"+ dir.getName() +"/"+ osmFile.getName() +"'>"+ osmFile.getName() +"</a></td>");
				hw.write("<td><a href='"+ dir.getName() +"/"+ gpxFile.getName() +"'>"+ gpxFile.getName() +"</a></td>");
				
				// Wiki
				ww.write("|-");
				ww.newLine();
				ww.write("| "+ municipality +" ");			// 市区町村名
				ww.write("|| {{State Route|h=0}} ");		// 編集状況
				
				// OSM file header
				ow = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(osmFile), "UTF-8"));
				ow.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
				ow.newLine();
				ow.write("<osm version=\"0.5\" generator=\"ReadKIBAN\">");
				ow.newLine();

				// GPX file header
				gw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(gpxFile), "UTF-8"));
				gw.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
				gw.newLine();
				gw.write("<gpx xmlns=\"http://www.topografix.com/GPX/1/1\" version=\"1.1\" creator=\"osmtracker-android\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd \">");
				gw.newLine();

				dataCount = 0;
				maxLat = 0.0;
				minLat = 180.0;
				maxLon = 0.0;
				minLon = 180.0;

				maeCode = new String(municipalitycode);
			}

			{
				dataCount++;
				
				if (lat > maxLat) {
					maxLat = lat;
				}
				if (lon > maxLon) {
					maxLon = lon;
				}
				if (lat < minLat) {
					minLat = lat;
				}
				if (lon < minLon) {
					minLon = lon;
				}
			
				// OSM node
				int nodeid = dataCount * -1;
				String osm_node = nodeISJ(nodeid, namecode, name, lat, lon, timeStampStr, typecode);
				ow.write(osm_node);
				ow.newLine();

				// TEXT node
				File txtFile = new File(dir, namecode +".txt");
				BufferedWriter gw2 = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(txtFile), "UTF-8"));
				gw2.write(osm_node);
				gw2.newLine();
				gw2.close();

				// GPX waypoint
				gw.write("<wpt lat=\""+ lat +"\" lon=\""+ lon +"\">\n");
				gw.write(" <time>"+ timeStampStr +"Z</time>\n");
				gw.write(" <name><![CDATA["+ name +"]]></name>\n");
				gw.write(" <link href=\""+ txtFile.getName() +"\"><text>"+ namecode +"</text></link>\n");
				gw.write("</wpt>\n");
				gw.newLine();
			}
		}
		rset7.close();
		
		{
			// OSM file footer
			ow.write("</osm>");
			ow.newLine();
			ow.close();

			// GPX file footer
			gw.write("</gpx>");
			gw.newLine();
			gw.close();
			
			ww.write("|| "+ dataCount +" ");			// 地名数
			ww.write("|| ");								// 備考
			ww.newLine();
			
			// HTML list
			hw.write("<form action=\"http://overpass-api.de/api/interpreter\" method=\"post\" accept-charset=\"UTF-8\">");
			hw.write("<td>"+ dataCount +"</td>");
			hw.write("<td><input type=\"submit\" value=\"Download\"/></td>");
			hw.write("<td><textarea name=\"data\" rows=\"1\" cols=\"40\"><bbox-query s=\""+ (minLat - 0.01D) +"\" n=\""+ (maxLat+0.01D) +"\" w=\""+ (minLon - 0.01D) +"\" e=\""+ (maxLon + 0.01D) +"\"/><query type=\"node\"><item/><has-kv k=\"place\"/></query><print/></textarea></td>");
			hw.write("</form>");
			hw.write("</tr>");
			hw.newLine();
		}

		// Wiki footer
		ww.close();

		// index file footer
		hw.write("</table></body></html>");
		hw.newLine();
		hw.close();
	}

	/**
	 * 
	 * @param code
	 * @param name
	 * @param lat
	 * @param lon
	 * @param timeStampStr
	 * @param typecode
	 * 	9: "大字・字・丁目区分コード"
	 * 		1 = 大字 = quarter
	 * 		2 = 字 = neighbourhood
	 * 		3 = 丁目 = neighbourhood
	 * 		4 = 不明 = locality
	 * @return
	 */
	public static String nodeISJ(int code, String ref, String name, Double lat, Double lon, String timeStampStr, int typecode) {
		String osm_node = ("<node id=\""+ code +"\" timestamp=\""+ timeStampStr +"Z\" lat=\""+ lat +"\" lon=\""+ lon +"\">\n");
		osm_node += "<tag k=\"name\" v=\""+ name +"\"/>\n";
		osm_node += "<tag k=\"ref\" v=\""+ ref +"\"/>\n";
		switch (typecode) {
		case 1:
			osm_node += "<tag k=\"place\" v=\"quarter\"/>\n";
			break;
		case 2:
			osm_node += "<tag k=\"place\" v=\"neighbourhood\"/>\n";
			break;
		case 3:
			osm_node += "<tag k=\"place\" v=\"neighbourhood\"/>\n";
			break;
		case 4:
			osm_node += "<tag k=\"place\" v=\"locality\"/>\n";
		}
		osm_node += "<tag k=\"fixme\" v=\"既存のノードが存在しないかチェックしてください\"/>\n";
		osm_node += "<tag k=\"source\" v=\"ISJ 2012\"/>\n";
		//osm_node += "<tag k=\"source_ref\" v=\"http://nlftp.mlit.go.jp/isj/\"/>\n";
		//osm_node += "<tag k=\"note\" v=\"National-Land Numerical Information (place) 2012, MLIT Japan\"/>\n";
		//osm_node += "<tag k=\"note:ja\" v=\"国土数値情報（大字・町丁目レベル位置参照情報）2012年 国土交通省\"/>\n";
		osm_node += "</node>\n";
		return osm_node;
	}

	/**
	 * インポートファイルの形式チェック
	 * 	「位置参照情報(都道府県別)」ファイルかどうかを判別する。
	 * 		・ファイル拡張子が「.CSV」であること
	 * 		・ファイル名の長さが11文字であること
	 * 		・３文字目が「_」であること
	 * 		・先頭２文字が数字であること
	 * 		・４〜８文字が数字であること
	 * 	「位置参照情報(市町村別)」ファイルかどうかを判別する。
	 * 		・ファイル拡張子が「.CSV」であること
	 * 		・ファイル名の長さが14文字であること
	 * 		・6文字目が「_」であること
	 * 		・先頭5文字が数字であること
	 * 		・6〜10文字が数字であること
	 * @param f
	 * @return
	 */
	static boolean checkFile(File f) {
		String name = f.getName();
		if (!name.toUpperCase().endsWith(".CSV")) {
			return false;
		}
		
		if (name.length() == 11) {
			String str = name.substring(2,3);
			if (!str.equals("_")) {
				return false;
			}

			str = name.substring(0,2);
			try {
				Integer.parseInt(str);
			}
			catch (NumberFormatException e) {
				return false;
			}

			str = name.substring(3,6);
			try {
				Integer.parseInt(str);
			}
			catch (NumberFormatException e) {
				return false;
			}
			return true;
		}
		if (name.length() == 14) {
			String str = name.substring(5,6);
			if (!str.equals("_")) {
				return false;
			}

			str = name.substring(0,5);
			try {
				Integer.parseInt(str);
			}
			catch (NumberFormatException e) {
				return false;
			}

			str = name.substring(6,10);
			try {
				Integer.parseInt(str);
			}
			catch (NumberFormatException e) {
				return false;
			}
			return true;
		}
		return false;
	}
}