/**
 * \brief Template repository and loader module for CoolRain template engine.
 *
 * Copyright (c) HATTORI, Hiroki
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#if !defined(COOLRAIN__LOADER__H__)
#define COOLRAIN__LOADER__H__

#include <glib/gmacros.h>

#include <CoolRain/thread.h>
#include <CoolRain/template.h>

G_BEGIN_DECLS;


struct coolrain_stash;
struct coolrain_writer;
struct coolrain_tagset;
struct coolrain_loader;



struct coolrain_loader_ops {
	int	(*load)(struct coolrain_loader * __restrict loader, struct coolrain_template * __restrict tmpl, char const * __restrict name);
	void	(*destroy)(struct coolrain_loader *loader);
};


#define COOLRAIN_LOADER_FLAGS_AUTO_FREE_TAGSET	(0)
#define COOLRAIN_LOADER_FLAGS_BINNARY_LOADER	(1)


struct coolrain_loader {
	struct coolrain_loader_ops *ops;
	void	*driver_data;
	unsigned long flags;

	struct coolrain_tagset *tagset;
	coolrain_template_error_handler_t error_handler;
	void	*error_user_data;

	void	*user_data;
	int	(*preprocess)(struct coolrain_eval_context * __restrict cntx, void ** __restrict preproc_data);
	void	(*postprocess)(struct coolrain_eval_context * __restrict cntx, void * __restrict preproc_data);
};


int coolrain_loader_initialize(struct coolrain_loader * __restrict loader, struct coolrain_loader_ops * __restrict ops, struct coolrain_tagset * __restrict tagset);
void coolrain_loader_destroy(struct coolrain_loader *loader);


static inline void coolrain_loader_set_error_handler(
	struct coolrain_loader * __restrict loader,
	coolrain_template_error_handler_t handler,
	void * __restrict user_data)
{
	loader->error_handler = handler;
	loader->error_user_data = user_data;
}

int coolrain_loader_load_template(struct coolrain_loader * __restrict loader, struct coolrain_template * __restrict tmpl, char const * __restrict name);
int coolrain_loader_process_template(struct coolrain_loader * __restrict loader, char const * __restrict name, struct coolrain_stash * __restrict stash, struct coolrain_writer * __restrict writer);


int coolrain_file_loader_initialize(struct coolrain_loader * __restrict loader, struct coolrain_tagset * __restrict tagset, char const * __restrict base_dir);


G_END_DECLS;

#endif


