/**
 * Threading wrapper module of CoolRain
 *
 */
#if !defined(COOLRAIN__THREAD__H__)
#define COOLRAIN__THREAD__H__

#include <stdbool.h>
#include <glib/gmacros.h>

G_BEGIN_DECLS;

#if COOLRAIN_USE_GTHREAD

#include <glib/gthread.h>

typedef GStaticRecMutex	coolrain_mutex_t;

static inline void coolrain_mutex_init(coolrain_mutex_t *mutex)
{
	g_static_rec_mutex_init(mutex);
}

static inline void coolrain_mutex_destroy(coolrain_mutex_t *mutex)
{
	g_static_rec_mutex_free(mutex);
}

static inline void coolrain_mutex_lock(coolrain_mutex_t *mutex)
{
	g_static_rec_mutex_lock(mutex);
}

//static inline bool coolrain_mutex_timed_lock(
//	coolrain_mutex_t * restrict mutex,
//	struct timespec const * restrict abstime)

static inline bool coolrain_mutex_trylock(coolrain_mutex_t *mutex)
{
	return g_static_rec_mutex_trylock(mutex)? true : false;
}

static inline void coolrain_mutex_unlock(coolrain_mutex_t *mutex)
{
	g_static_rec_mutex_unlock(mutex);
}


#elif COOLRAIN_USE_PTHREAD


#include <pthread.h>

typedef pthread_mutex_t	coolrain_mutex_t;

static inline void coolrain_mutex_init(coolrain_mutex_t *mutex)
{
	pthread_mutexattr_t attr;
	pthread_mutexattr_init(&attr);
	pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE_NP);

	pthread_mutex_init(mutex, &attr);

	pthread_mutexattr_destroy(&attr);
}

static inline void coolrain_mutex_destroy(coolrain_mutex_t *mutex)
{
	pthread_mutex_destroy(mutex);
}

static inline void coolrain_mutex_lock(coolrain_mutex_t *mutex)
{
	pthread_mutex_lock(mutex);
}


#define COOLRAIN_MUTEX_HAS_TIMEDLOCK	1

static inline bool coolrain_mutex_timedlock(
	coolrain_mutex_t * restrict mutex,
	truct timespec const * restrict abstime)
{
	return (pthread_mutex_timedlock(mutex, abstime) == 0)? true : false;
}



static inline bool coolrain_mutex_trylock(coolrain_mutex_t *mutex)
{
	return (pthread_mutex_trylock(mutex) == 0)? true : false;
}

static inline void coolrain_mutex_unlock(coolrain_mutex_t *mutex)
{
	pthread_mutex_unlock(mutex);
}

#elif COOLRAIN_USE_APR_THREAD

#include <apr_thread_mutex.h>

typedef apr_thread_mutex_t	*coolrain_mutex_t;

static inline void coolrain_mutex_init(coolrain_mutex_t *mutex)
{
	apr_thread_mutex_create(mutex, APT_THREAD_MUTEX_NESTED, pool);
}

static inline void coolrain_mutex_destroy(coolrain_mutex_t *mutex)
{
	apr_thread_mutex_destroy(*mutex);
	*mutex = NULL;
}

static inline void coolrain_mutex_lock(coolrain_mutex_t *mutex)
{
	apr_thread_mutex_lock(*mutex);
}

//static inline bool coolrain_mutex_timedlock

static inline bool coolrain_mutex_trylock(coolrain_mutex_t *mutex)
{
	return (apr_thread_mutex_trylock(*mutex) == APR_STATUS_IS_BUSY)
		? false : true;
}


static inline void coolrain_mutex_unlock(coolrain_mutex_t *mutex)
{
	apr_thread_mutex_unlock(*mutex);
}

#else

typedef int	coolrain_mutex_t[0];
#define coolrain_mutex_init(x)		do {} while (0)
#define coolrain_mutex_destroy(x)	do {} while (0)
#define coolrain_mutex_lock(x)		do {} while (0)
//#define coolrain_mutex_timedlock(x, y)	(true)
#define coolrain_mutex_trylock(x)	(true)
#define coolrain_mutex_unlock(x)	do {} while (0)

#endif

G_END_DECLS;

#endif

