/**
 *
 *
 */
#if !defined(COOLRAIN__WRITER__H__)
#define COOLRAIN__WRITER__H__

#include <stddef.h>
#include <stdbool.h>

#include <stdbool.h>

#include <glib/gmacros.h>

#include <CoolRain/error.h>
#include <CoolRain/thread.h>

G_BEGIN_DECLS;

struct coolrain_writer;

typedef int (*coolrain_writer_buffer_full_handler_t)(struct coolrain_writer *writer);
typedef int (*coolrain_writer_destroy_handler_t)(struct coolrain_writer *writer);
typedef int (*coolrain_writer_write_handler_t)(
	struct coolrain_writer *writer,
	char const **pbuffer, size_t length);


#define COOLRAIN_WRITER_FLAGS_AUTO_FREE		(1 << 0)
#define COOLRAIN_WRITER_FLAGS_AUTO_CLOSE	(1 << 1)


/**
 *
 */
struct coolrain_writer {
	coolrain_mutex_t	mutex;

	char		*buffer;	/*!< Pointer to internal buffer */
	size_t		alloc_length;	/*!< Total allocation length of buffer */
	char		*cursor;	/*!< Write point */
	unsigned int	flags;

	union {
	    void		*pointee;
	    unsigned int	handle;
	    char		minibuffer[sizeof(unsigned int)];
	} write_handler_data, buffer_full_handler_data;

	coolrain_writer_buffer_full_handler_t	buffer_full_handler;
	coolrain_writer_write_handler_t		write_handler;
	coolrain_writer_destroy_handler_t	destroy_handler;
};


int coolrain_writer_initialize(struct coolrain_writer *writer, char *buffer, size_t length);
int coolrain_writer_destroy(struct coolrain_writer *writer);

int coolrain_writer_chain(struct coolrain_writer *writer, struct coolrain_writer *next_writer);


int coolrain_writer_write(struct coolrain_writer *writer, char const *buffer, size_t length);


static inline int coolrain_writer_flush_nonlock(struct coolrain_writer *writer)
{
	return (*writer->buffer_full_handler)(writer);
}


static inline int coolrain_writer_flush(struct coolrain_writer *writer)
{
	int r;

	coolrain_mutex_lock(&writer->mutex);
	r = writer->buffer_full_handler(writer);
	coolrain_mutex_unlock(&writer->mutex);

	return r;
}



/*
 * !{ @name Direct buffer operation (for filter developpers)
 * */
int coolrain_writer_drop(struct coolrain_writer *writer, size_t length);



/**
 * Discard all data from internal buffer
 * \param writer	Writer object.
 *
 * \par Threading:
 * This API is not thread safe.
 */
static inline void coolrain_writer_reset(struct coolrain_writer *writer)
{
	writer->cursor = writer->buffer;
}



/**
 * Wrote size in buffer
 * \param writer	Writer object
 * \return Length in bytes.
 * \par Threading:
 * This API is not thread safe.
 */
static inline size_t coolrain_writer_used_length(struct coolrain_writer *writer)
{
    return writer->cursor - writer->buffer;
}



/**
 * Remaining buffer length.
 * \param writer	Writer object
 * \return Length in bytes.
 *
 * \par Threading:
 * This API is not thread safe.
 */
static inline size_t coolrain_writer_unused_length(struct coolrain_writer *writer)
{
	return (writer->buffer + writer->alloc_length) - writer->cursor;
}


static inline char *coolrain_writer_reserve(struct coolrain_writer *writer, size_t length)
{
	char *p;

	if (coolrain_writer_unused_length(writer) < length) {
		coolrain_writer_flush(writer);
		if (coolrain_writer_unused_length(writer) < length)
			return NULL;
	}

	p = writer->cursor;
	writer->cursor += length;

	return p;
}
/* !} */





int coolrain_fd_writer_initialize(struct coolrain_writer *writer, int fd, bool autoclose);

#  if HAVE_FCGIAPP_H
#    include <fcgiapp.h>
int coolrain_FCGI_writer_initialize(struct coolrain_writer *writer, FCGX_Stream *stream);
#  endif





/*
 *
 */
typedef coolrain_writer_write_handler_t	coolrain_filter_handler_t;

int coolrain_html_filter(struct coolrain_writer *writer, char const **pbuffer, size_t length);
int coolrain_url_filter(struct coolrain_writer *writer, char const **pbufferr, size_t length);
int coolrain_javascript_filter(struct coolrain_writer *writer, char const **pbuffer, size_t length);

G_END_DECLS;

#endif

