/*$Id$*/
package nicobrowser;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.List;
import nicobrowser.Config.NicoFeed;
import org.apache.commons.io.FileUtils;
import org.junit.After;
import org.junit.BeforeClass;
import org.junit.Test;
import static org.junit.Assert.*;

public class ConfigTest {

    private static File APP_HOME;
    private static File CONFIG_FILE;
    private static File FEEDURL_FILE;
    private final File TEST_PROPERTY_FILE;
    private final File TEST_FEED_FILE;

    public ConfigTest() {
        TEST_PROPERTY_FILE = new File("test/testdata/nicobrowser.properties");
        TEST_FEED_FILE = new File("test/testdata/feedurl.txt");
    }

    @BeforeClass
    public static void setUpClass() throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
        try {
            Field f;
            f = Config.class.getDeclaredField("APP_HOME");
            f.setAccessible(true);
            APP_HOME = (File) f.get(null);

            f = Config.class.getDeclaredField("CONFIG_FILE");
            f.setAccessible(true);
            CONFIG_FILE = (File) f.get(null);

            f = Config.class.getDeclaredField("FEEDURL_FILE");
            f.setAccessible(true);
            FEEDURL_FILE = (File) f.get(null);

            if (APP_HOME.exists()) {
                String message = "ディレクトリを削除/移動してから再実行してください: " + APP_HOME;
                System.err.println(message);
                throw new Error(message);
            }
        } catch (Throwable th) {
            throw new Error(th);
        }
    }

    @After
    public void tearDown() throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
        // コンフィグディレクトリを削除
        FileUtils.deleteQuietly(APP_HOME);

        // コンフィグインスタンスを削除.
        Field f = Config.class.getDeclaredField("instance");
        f.setAccessible(true);
        f.set(null, null);
    }

    /**
     * Test of getInstance method, of class Config.
     */
    @Test
    public void testGetInstance() throws IOException {
        System.out.println("getInstance");

        try {
            Config.getInstance();
            fail("コンフィグファイル未作成の場合は例外発生");
        } catch (Exception ex) {
        }

        Config.createNewConfigFiles();
        assertNotNull(Config.getInstance());

    }

    /**
     * Test of createNewConfigFiles method, of class Config.
     */
    @Test
    public void testCreateNewConfigFiles() throws Exception {
        System.out.println("createNewConfigFiles");
        boolean result;

        result = Config.createNewConfigFiles();
        assertTrue("コンフィグが無いので新規作成される", result);
        assertTrue(CONFIG_FILE.isFile());
        assertTrue(FEEDURL_FILE.isFile());

        result = Config.createNewConfigFiles();
        assertFalse("作成済みなので作成されない", result);

        CONFIG_FILE.delete();
        assertFalse(CONFIG_FILE.exists());
        result = Config.createNewConfigFiles();
        assertTrue("コンフィグファイルだけ無くてもtrue", result);
        assertTrue(CONFIG_FILE.isFile());

        FEEDURL_FILE.delete();
        assertFalse(FEEDURL_FILE.exists());
        result = Config.createNewConfigFiles();
        assertTrue("フィードファイルだけ無くてもtrue", result);
        assertTrue(FEEDURL_FILE.isFile());
    }

    /**
     * 初期作成コンフィグの妥当性をテストする.
     */
    @Test
    public void testConfigInitialMake() throws IOException {
        System.out.println("testConfigInitialMake");
        Config.createNewConfigFiles();

        Config conf = Config.getInstance();
        assertEquals(new File(APP_HOME, "db/nicodb").getAbsolutePath(), conf.getDbFile());
        assertEquals(new File(APP_HOME, "flv").getAbsolutePath(), conf.getSrcSaveDir());
    }

    /**
     * コンフィグ初期生成を行うためのヘルパーメソッド.
     */
    private void initConfig() {
        try {
            Config.createNewConfigFiles();
            FileUtils.copyFile(TEST_PROPERTY_FILE, CONFIG_FILE);
            FileUtils.copyFile(TEST_FEED_FILE, FEEDURL_FILE);
        } catch (Exception ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Test of getNicoMail method, of class Config.
     */
    @Test
    public void testGetNicoMail() {
        System.out.println("getNicoMail");
        initConfig();

        String result = Config.getInstance().getNicoMail();
        assertEquals("my@mail.address", result);
    }

    /**
     * Test of getNicoPassword method, of class Config.
     */
    @Test
    public void testGetNicoPassword() {
        System.out.println("getNicoPassword");
        initConfig();

        String result = Config.getInstance().getNicoPassword();
        assertEquals("my_password", result);
    }

    /**
     * Test of getDbFile method, of class Config.
     */
    @Test
    public void testGetDbFile() {
        System.out.println("getDbFile");
        initConfig();

        String result = Config.getInstance().getDbFile();
        assertEquals("C:\\Documents and Settings\\test\\.nicobrowser\\db\\nicodb", result);
    }

    /**
     * Test of getSrcSaveDir method, of class Config.
     */
    @Test
    public void testGetSrcSaveDir() {
        System.out.println("getSrcSaveDir");
        initConfig();

        String result = Config.getInstance().getSrcSaveDir();
        assertEquals("d:\\test", result);
    }

    /**
     * Test of getEncoding method, of class Config.
     */
    @Test
    public void testGetEncoding() {
        System.out.println("getEncoding");
        initConfig();

        String result = Config.getInstance().getEncoding();
        assertEquals("UTF-8", result);
    }

    /**
     * Test of getMaxRetry method, of class Config.
     */
    @Test
    public void testGetMaxRetry() {
        System.out.println("getMaxRetry");
        initConfig();

        int result = Config.getInstance().getMaxRetry();
        assertEquals(3, result);
    }

    /**
     * Test of getDownLoadMyList method, of class Config.
     */
    @Test
    public void testGetDownLoadMyList() {
        System.out.println("getDownLoadMyList");
        initConfig();

        List<String> result = Config.getInstance().getDownLoadMyList();
        List expected = Arrays.asList(new String[]{"100", "200", "300"});
        assertEquals(expected, result);
    }

    @Test
    public void testGetFeeds() {
        System.out.println("testGetFeeds");
        initConfig();

        List<Config.NicoFeed> feeds = Config.getInstance().getNicoFeeds();
        assertEquals(5, feeds.size());

        NicoFeed feed = feeds.get(2);
        assertEquals("http://www.nicovideo.jp/tag/科学?sort=f&rss=2.0", feed.getUrl());
        assertEquals(3, feed.getNumber());

        for (Config.NicoFeed nf : feeds) {
            System.out.println(nf);
        }
    }
}
