/** */
package nicobrowser.update;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.Statement;
import liquibase.commandline.Main;
import liquibase.exception.CommandLineParsingException;
import nicobrowser.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DBUpdater {

    private static Logger log = LoggerFactory.getLogger(DBUpdater.class);
    private static final String UPDATE_FILE = "db_update_script.xml";
    private static final String SYNC_FILE = "db_for_sync_script.xml";

    /**
     * ver.0.4.xからver.0.5.xにアップグレードするためのDBアップデート.
     * LiquiBaseはIDとしてスクリプトファイル名を利用しているが, これは環境に依存する(フルパスで記録される)ため,
     * 環境を変更すると本来同一であるべきものが同一でないとみなされてしまう.
     * これの回避策としてファイル名を固定した.
     * 既存のカラム値をこの固定名にアップデートする.
     */
    public void sync_for_4() throws Exception {
        try {
            Class.forName("org.h2.Driver");
            Connection con = DriverManager.getConnection("jdbc:h2:" + Config.getInstance().getDbFile(), "sa", "");

            Statement statement = con.createStatement();
            String sql = "UPDATE DATABASECHANGELOG SET FILENAME='db_update_script.xml'";
            statement.executeUpdate(sql);

            statement.close();
            con.close();
            log.info("DBのアップデートが終了しました.");
        } catch (Exception ex) {
            log.error("DBのアップデートに失敗しました.", ex);
            throw ex;
        }
    }

    public void sync() throws Exception {
        sync_for_4();

        File updateFile = new File(Config.getAppHome(), UPDATE_FILE);
        InputStream resource = ClassLoader.getSystemResourceAsStream("resources/" + SYNC_FILE);

        createFile(resource, updateFile);

        String[] args = new String[]{
            "--driver=org.h2.Driver",
            "--changeLogFile=" + updateFile.toString(),
            "--url=jdbc:h2:" + Config.getInstance().getDbFile(),
            "--username=sa",
            "changeLogSync"};
        try {
            Main.main(args);

            log.info("DBのアップデートが終了しました.");
        } catch (CommandLineParsingException ex) {
            log.error("DBのアップデートに失敗しました.", ex);
            throw ex;
        }
    }

    public void update() throws Exception {
        File updateFile = new File(Config.getAppHome(), UPDATE_FILE);
        InputStream resource = ClassLoader.getSystemResourceAsStream("resources/" + UPDATE_FILE);

        createFile(resource, updateFile);

        String[] args = new String[]{
            "--driver=org.h2.Driver",
            "--changeLogFile=" + updateFile.toString(),
            "--url=jdbc:h2:" + Config.getInstance().getDbFile(),
            "--username=sa",
            "update"};
        try {
            Main.main(args);

            log.info("DBのアップデートが終了しました.");
        } catch (CommandLineParsingException ex) {
            log.error("DBのアップデートに失敗しました.", ex);
            throw ex;
        }
    }

    private void createFile(InputStream resource, File updateFile) throws IOException {
        BufferedInputStream is = null;
        BufferedOutputStream os = null;
        try {
            is = new BufferedInputStream(resource);
            os = new BufferedOutputStream(new FileOutputStream(updateFile));
            byte[] b = new byte[1024];
            int l;
            while ((l = is.read(b)) != -1) {
                os.write(b, 0, l);
            }
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException ex) {
            }
            if (os != null) {
                try {
                    os.close();
                } catch (IOException ex) {
                }
            }
        }
    }
}
