﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Net;
using Microsoft.Win32;
using Utility;
using CTester.Setting;
using CTester.NewProject;
using CTester.Emulator.Property;

namespace CTester.Command
{

    /// <summary>
    /// コマンドコレクション
    /// </summary>
    public class CommandCollection
    {
        private Dictionary<Type, CommandBase> _commands = new Dictionary<Type, CommandBase>();

        public CommandBase this[Type type]
        {
            get
            {
                if (!_commands.ContainsKey(type))
                {
                    return null;
                }
                return _commands[type];
            }
            set
            {
                if (!_commands.ContainsKey(type))
                {
                    var command = value as CommandBase;
                    if (command != null)
                    {
                        _commands.Add(type, value);
                    }
                }
                else
                {
                    _commands[type] = value;
                }
            }
        }

        public CommandBase Add(CommandBase command)
        {
            Type type = command.GetType();
            this[type] = command;
            return command;
        }
    }

    /// <summary>
    /// 新規プロジェクト作成コマンド
    /// </summary>
    public class CommandNewProject : CommandBase
    {
        Project _project;

        public CommandNewProject(Project project)
            : base()
        {
            _project = project;
        }

        public override bool CanExecute(object parameter)
        {
            return true;
        }

        public override void Execute(object parameter)
        {
            var dlg = new DialogNewProject();
            bool? dlgResult = dlg.ShowDialog();
            
            if (dlgResult.HasValue && dlgResult.Value)
            {
                var data = new Project.SerializeData();
                data.SourcePath = dlg.SourcePath;
                
                _project.NewProject(dlg.ProjectFolder, dlg.ProjectName, data);
            }
        }
    }

    /// <summary>
    /// プロジェクトオープンコマンド
    /// </summary>
    public class CommandOpenProject : CommandBase
    {
        Project _project;

        public CommandOpenProject(Project project)
            : base()
        {
            _project = project;
        }

        public override bool CanExecute(object parameter)
        {
            return true;
        }

        public override void Execute(object parameter)
        {
            var dlg = new OpenFileDialog();

            dlg.Filter = "プロジェクトファイル(*." + App.ProjectExtension + ")|*." + App.ProjectExtension;
            dlg.CheckFileExists = true;
            dlg.Multiselect = false;
            dlg.Title = "プロジェクトファイルを指定してください。";
            if (dlg.ShowDialog() == true)
            {
                _project.LoadProject(dlg.FileName);
            }
        }
    }

    /// <summary>
    /// ソースオープンコマンド
    /// </summary>
    public class CommandOpenSource : CommandBase
    {
        Project _project;

        public CommandOpenSource(Project project)
            : base()
        {
            _project = project;
        }

        public override bool CanExecute(object parameter)
        {
            return true;
        }

        public override void Execute(object parameter)
        {
            var dlg = new OpenFileDialog();

            dlg.Filter = "ソースDLLファイル(*." + App.SourceExtension + ")|*." + App.SourceExtension;
            dlg.CheckFileExists = true;
            dlg.Multiselect = false;
            dlg.Title = "ソースDLLファイルを指定してください。";
            if (dlg.ShowDialog() == true)
            {
                if (!_project.LoadSource(dlg.FileName))
                {
                    MessageBox.Show("DLLファイルが不正です。", "エラー", MessageBoxButton.OK);
                }
            }
        }
    }

    /// <summary>
    /// エミュレータ設定コマンド
    /// </summary>
    public class CommandEmuSetting : CommandBase
    {
        Project _project;

        public CommandEmuSetting(Project project)
            : base()
        {
            _project = project;
        }

        public override bool CanExecute(object parameter)
        {
            return true;
        }

        public override void Execute(object parameter)
        {
            var dlg = new DialogEmuSetting(_project.EmuSetting);

            if (dlg.ShowDialog() == true)
            {
                _project.EmuSetting = dlg.EmuSetting;
            }
        }
    }

    /// <summary>
    /// 終了コマンド
    /// </summary>
    public class CommandClose : CommandBase
    {
        Window _targetWindow;
        public CommandClose(Window targetWindow)
            : base()
        {
            _targetWindow = targetWindow;
        }

        public override bool CanExecute(object parameter)
        {

            return true;
        }

        public override void Execute(object parameter)
        {
            _targetWindow.Close();
        }
    }
}
