// -*-Mode: C++;-*-
//
//  Name label renderer class
//
// $Id: NameLabelRenderer.cpp,v 1.15 2011/05/02 14:51:29 rishitani Exp $

#include <common.h>
#include "NameLabelRenderer.hpp"

#include "MolCoord.hpp"
#include "MolChain.hpp"
#include "MolResidue.hpp"

#include <gfx/PixelBuffer.hpp>
#include <gfx/TextRenderManager.hpp>
#include <gfx/DisplayContext.hpp>
#include <qsys/SceneManager.hpp>

namespace molstr {

struct NameLabel
{
  int aid;
  LString strAid;

  gfx::PixelBuffer pixels;
  LString str;

  inline bool equals(const NameLabel &a) const {
    return aid==a.aid;
  }
};

//typedef std::list<NameLabel> NameLabelList;
struct NameLabelList : public std::list<NameLabel> {};

}

//////////////////////////////////////////////////////////////////////////

using namespace molstr;

NameLabelRenderer::NameLabelRenderer()
     : super_t()
{
  m_pdata = new NameLabelList;

  //m_nMax = 5;
  //m_xdispl = 0.0;
  //m_ydispl = 0.0;

  m_strFontStyle = "normal";
  m_strFontWgt = "normal";

  // will be called by RendererFactory
  //resetAllProps();
}

NameLabelRenderer::~NameLabelRenderer()
{
  delete m_pdata;
}

//////////////////////////////////////////////////////////////////////////

MolCoordPtr NameLabelRenderer::getClientMol() const
{
  qsys::ObjectPtr robj = qsys::SceneManager::getObjectS(getClientObjID());
  if (robj.isnull()) return MolCoordPtr();
  return MolCoordPtr(robj);
}

bool NameLabelRenderer::isCompatibleObj(qsys::ObjectPtr pobj) const
{
  MolCoord *ptest = dynamic_cast<MolCoord *>(pobj.get());
  return ptest!=NULL;
}

LString NameLabelRenderer::toString() const
{
  return LString::format("NameLabelRenderer %p", this);
}

//////////////////////////////////////////////////////////////////////////

void NameLabelRenderer::preRender(DisplayContext *pdc)
{

  Vector4D dv;
  qsys::View *pview = pdc->getTargetView();
  if (pview!=NULL)
    pview->convXYTrans(m_xdispl, m_ydispl, dv);

  pdc->enableDepthTest(false);

  pdc->pushMatrix();
  pdc->translate(dv);
//  pdc->color(m_color);
}

void NameLabelRenderer::postRender(DisplayContext *pdc)
{
  pdc->popMatrix();
  pdc->enableDepthTest(true);
}

void NameLabelRenderer::render(DisplayContext *pdl)
{
  // pdl->color(m_color);
  pdl->setLighting(false);

  MolCoordPtr rCliMol = getClientMol();
  if (rCliMol.isnull()) {
    MB_DPRINTLN("NameLabelRenderer::render> Client mol is null");
    return;
  }
  
  NameLabelList::iterator iter = m_pdata->begin();
  
  gfx::TextRenderManager *pTRM = gfx::TextRenderManager::getInstance();
  if (pTRM==NULL) return;

  LString strlab;
  bool bFontSet = false;

  for (; iter!=m_pdata->end(); iter++) {
    NameLabel &nlab = *iter;

    Vector4D pos;
    if (!makeLabelStr(nlab, strlab, pos)) {
      MB_DPRINTLN("NameLabel: mklab failed in Atom %d", nlab.aid);
    }

    int datasize = nlab.pixels.size();
    if ( datasize==0 || !strlab.equals(nlab.str) ) {
      if (!bFontSet) {
        pTRM->setupFont(m_dFontSize, m_strFontName,
                        m_strFontStyle, m_strFontWgt);
        bFontSet = true;
      }
      // new text or text label has been changed
      if (!pTRM->renderText(strlab, nlab.pixels)) {
        MB_DPRINTLN("NameLabel: render text label in Atom %d", nlab.aid);
      }

      nlab.str = strlab;
    }

    pdl->drawPixels(pos, nlab.pixels, *(m_color.get()) );
  }

//  pdl->setLighting(false);
}

Vector4D NameLabelRenderer::getCenter() const
{
  // TO DO: throw NoCenterException
  return Vector4D();
}

bool NameLabelRenderer::isHitTestSupported() const
{
  return false;
}

const char *NameLabelRenderer::getTypeName() const
{
  return "*namelabel";
}

//////////////////////////////////////////////////////////////////////////
// Label operations

bool NameLabelRenderer::makeLabelStr(NameLabel &nlab,
                                     LString &strlab,
                                     Vector4D &pos)
{
  MolCoordPtr pobj = getClientMol();
  MB_ASSERT(!pobj.isnull());
  
  if (nlab.aid<0) {
    nlab.aid = pobj->fromStrAID(nlab.strAid);
    if (nlab.aid<0)
      return false;
  }

  MolAtomPtr pAtom = pobj->getAtom(nlab.aid);
  if (pAtom.isnull())
    return false;

  pos = pAtom->getPos();
  LString sbuf = pAtom->getChainName() + " " +
    pAtom->getResName() +
      pAtom->getResIndex().toString() + " " +
        pAtom->getName();
  char confid = pAtom->getConfID();
  if (confid)
    sbuf += LString(":") + LString(confid);
  
  strlab = sbuf.toUpperCase();
  return true;
}

void NameLabelRenderer::destroyAllTextPixbuf()
{
  BOOST_FOREACH(NameLabel &nlab, *m_pdata) {
    int datasize = nlab.pixels.size();
    if ( datasize>0 ) {
      nlab.pixels.clear();
      nlab.pixels.setWidth(0);
      nlab.pixels.setHeight(0);
    }
  }
}

bool NameLabelRenderer::addLabel(MolAtomPtr patom)
{
  NameLabel newlab;
  newlab.aid = patom->getID();

  // NameLabelList::const_iterator iter = m_pdata->begin();
  //for (; iter!=m_pdata->end(); iter++) {
  // const NameLabel &n = *iter;

  BOOST_FOREACH(NameLabel &nlab, *m_pdata) {
    if (newlab.equals(nlab))
      return false; // already labeled
  }

  m_pdata->push_back(newlab);
  int nover = m_pdata->size() - m_nMax;

  for (; nover>0; nover--)
    m_pdata->pop_front();

  invalidateDisplayCache();

  return true;
}
    
bool NameLabelRenderer::addLabelByID(int aid)
{
  MolCoordPtr pobj = getClientMol();
  MB_ASSERT(!pobj.isnull());
  
  MolAtomPtr pAtom = pobj->getAtom(aid);
  return addLabel(pAtom);
}

///////////////////////

void NameLabelRenderer::propChanged(qlib::LPropEvent &ev)
{
  const LString propnm = ev.getName();
  if (propnm.equals("color")) {
    invalidateDisplayCache();
  }
  else if (propnm.startsWith("font_")) {
    destroyAllTextPixbuf();
    invalidateDisplayCache();
  }

  super_t::propChanged(ev);
}

void NameLabelRenderer::styleChanged(qsys::StyleEvent &ev)
{
  super_t::styleChanged(ev);
  destroyAllTextPixbuf();
}

void NameLabelRenderer::writeTo2(qlib::LDom2Node *pNode) const
{
  // write properties
  super_t::writeTo2(pNode);

  MolCoordPtr pobj = getClientMol();
  MB_ASSERT(!pobj.isnull());
  
  BOOST_FOREACH(NameLabel &value, *m_pdata) {

    LString said = pobj->toStrAID(value.aid);
    if (said.isEmpty())
      continue;

    qlib::LDom2Node *pChNode = pNode->appendChild("label");
    // always in child element
    pChNode->setAttrFlag(false);

    // add atom attribute
    pChNode->appendStrAttr("aid", said);
  }
}

void NameLabelRenderer::readFrom2(qlib::LDom2Node *pNode)
{
  super_t::readFrom2(pNode);

  for (pNode->firstChild(); pNode->hasMoreChild(); pNode->nextChild()) {
    qlib::LDom2Node *pChNode = pNode->getCurChild();
    LString tag = pChNode->getTagName();

    if (!tag.equals("label")) {
      // error
      continue;
    }

    if (!pChNode->findChild("aid")) {
      // error
      continue;
    }

    LString value = pChNode->getStrAttr("aid");
    if (value.isEmpty()) {
      // error
      continue;
    }
      
    NameLabel elem;
    elem.aid = -1;
    elem.strAid = value;

    m_pdata->push_back(elem);
  }
  
}

