// -*-Mode: C++;-*-
//
// Molecular surface object
//

#ifndef MOL_SURF_OBJ_HPP_INCLUDED
#define MOL_SURF_OBJ_HPP_INCLUDED

#include "surface.hpp"
#include <qlib/Vector4D.hpp>
#include <qsys/Object.hpp>
#include <qlib/LDOM2Stream.hpp>

namespace surface {

using qlib::Vector4D;
class MolSurfEditInfo;

// #define USE_VERT_TYPE_ID

class SURFACE_API MolSurfObj : public qsys::Object,
  public qlib::LDataChunk
{
  MC_SCRIPTABLE;

  friend class CutByPlane;
  friend class MolSurfEditInfo;

public:
  struct Vert {
    float x, y, z, nx, ny, nz;

#ifdef USE_VERT_TYPE_ID
    /** for debug */
    int ntype;
#endif
    
    Vert()
      : x(0), y(0), z(0), nx(0), ny(0), nz(0)
#ifdef USE_VERT_TYPE_ID
        ,ntype(0)
#endif
    {
    }

    Vert(const Vert &arg)
      : x(arg.x), y(arg.y), z(arg.z), nx(arg.nx), ny(arg.ny), nz(arg.nz)
#ifdef USE_VERT_TYPE_ID
        ,ntype(arg.ntype)
#endif
    {
    }

    Vert(const Vector4D &v, const Vector4D &n)
      : x((float)v.x()), y((float)v.y()), z((float)v.z()),
        nx((float)n.x()), ny((float)n.y()), nz((float)n.z())
#ifdef USE_VERT_TYPE_ID
        ,ntype(0)
#endif
    {
    }

    Vert(const Vector4D &v)
      : x((float)v.x()), y((float)v.y()), z((float)v.z()),
        nx(0), ny(0), nz(0)
#ifdef USE_VERT_TYPE_ID
        ,ntype(0)
#endif
    {
    }

    const Vert &operator=(const Vert &arg) {
      if(&arg!=this){
        x = arg.x;
        y = arg.y;
        z = arg.z;
        nx = arg.nx;
        ny = arg.ny;
        nz = arg.nz;
#ifdef USE_VERT_TYPE_ID
        ntype = arg.ntype;
#endif
      }
      return *this;
    }

    Vector4D v3d() const {
      return Vector4D(x, y, z);
    }

    Vector4D n3d() const {
      return Vector4D(nx, ny, nz);
    }
  };

  struct Face {
    int id1, id2, id3;

    Face() {}
    Face(int rid1, int rid2, int rid3)
         : id1(rid1), id2(rid2), id3(rid3)
      {}
  };

#ifdef USE_VERT_TYPE_ID
  /** for debug */
  enum {
    FTID_NONE = 0,
    FTID_DBG1 = 1,
    FTID_DBG2 = 2,
    FTID_DBG3 = 3,
    FTID_DBG4 = 4,
    FTID_DBG5 = 5,
    FTID_DBG6 = 6,
    FTID_DBG7 = 7
  };
#endif

private:
  //
  //  Surface Data
  //
  
  int m_nVerts;
  Vert *m_pVerts;

  int m_nFaces;
  Face *m_pFaces;

public:

  /*
  mutable std::deque<std::pair<Vector4D, LString> > m_dbg;

  void dbgmsg(const Vector4D &v, const LString &str) const
  {
    m_dbg.push_back(std::pair<Vector4D, LString>(v, str));
  }  
   */

  ////////////////////////////////////////////
public:

  MolSurfObj()
       : m_nVerts(0), m_pVerts(NULL),
         m_nFaces(0), m_pFaces(NULL)
    {
    }

  virtual ~MolSurfObj();

  ////////////////////////////////////////////
  // MbObject operations

  virtual bool isEmpty() const;
  
  virtual void deleteSelected();

  //virtual MbSelection *interpHit(Hittest *phit) const;
  //virtual void clicked(MbSelection *psel);

  ////////////////////////////////////////////
  // construction

  /// Cleanup all data
  void clean();
  
  /// Set vertex array size
  bool setVertSize(int n) {
    MB_ASSERT(n>=1);
    if (m_pVerts!=NULL)
      delete [] m_pVerts;
    m_pVerts = new Vert[n];
    if (m_pVerts==NULL) return false;
    m_nVerts = n;
    return true;
  }

  /// Load [index]th vertex element
  bool setVertex(int index, const Vector4D &vec, const Vector4D &norm) {
    MB_ASSERT(m_pVerts!=NULL);
    MB_ASSERT(index>=0 && index<m_nVerts);
    m_pVerts[index].x = (float) vec.x();
    m_pVerts[index].y = (float) vec.y();
    m_pVerts[index].z = (float) vec.z();
    m_pVerts[index].nx = (float) norm.x();
    m_pVerts[index].ny = (float) norm.y();
    m_pVerts[index].nz = (float) norm.z();
    return true;
  }

  void setVertex(int index, const Vert &v) {
    m_pVerts[index] = v;
  }

  /** set face array size */
  bool setFaceSize(int n) {
    MB_ASSERT(n>=1);
    if (m_pFaces!=NULL)
      delete [] m_pFaces;
    m_pFaces = new Face[n];

    m_nFaces = n;
    return true;
  }
  
  /** load [index]th face element */
  bool setFace(int index, int id1, int id2, int id3) {
    MB_ASSERT(index>=0 && index<m_nFaces);
    m_pFaces[index].id1 = id1;
    m_pFaces[index].id2 = id2;
    m_pFaces[index].id3 = id3;
    return true;
  }
    
  void setFace(int index, const Face &f) {
    MB_ASSERT(index>=0 && index<m_nFaces);
    m_pFaces[index] = f;
  }

  ////////////////////////////////////////
  // data extraction

  int getVertSize() const { return m_nVerts; }

  const Vert &getVertAt(int i) const {
    MB_ASSERT(m_pVerts!=NULL);
    MB_ASSERT(i>=0 && i<m_nVerts);
    return m_pVerts[i];
  }
  
  //

  int getFaceSize() const { return m_nFaces; }
  
  const Face &getFaceAt(int i) const {
    MB_ASSERT(m_pFaces!=NULL);
    MB_ASSERT(i>=0 && i<m_nFaces);
    return m_pFaces[i];
  }

  Vert *getVertPtr() const {
    return m_pVerts;
  }
  Face *getFacePtr() const {
    return m_pFaces;
  }

  //////////

  void cutByPlane(double cdiv, const Vector4D &norm, const Vector4D &pos, bool bSec);

  ////////////////////////////////////////////
  // Data chunk serialization

  virtual LString getDataChunkReaderName() const;
  // virtual LString getDataChunkWriterName() const =0;
  virtual void writeDataChunkTo(qlib::OutStream &oos) const;
  virtual void readDataChunkFrom(qlib::InStream &ois);

};

typedef MolSurfObj::Vert MSVert;
typedef MolSurfObj::Face MSFace;

}

#endif // MOL_SURF_OBJ_HPP_INCLUDED

