// -*-Mode: C++;-*-
//
// Object: base class of data object
//
// $Id: Object.cpp,v 1.42 2011/03/31 14:19:15 rishitani Exp $
//

#include <common.h>

#include "Object.hpp"
#include "SceneManager.hpp"
#include "SceneEvent.hpp"
#include "ScrEventManager.hpp"
#include "ObjLoadEditInfo.hpp"
#include "ObjExtData.hpp"
#include <qlib/LDOM2Stream.hpp>

#include "RendererFactory.hpp"
#include "UndoManager.hpp"
#include "PropEditInfo.hpp"

using namespace qsys;

Object::Object()
{
  m_pReaderOpts = NULL;

  m_uid = qlib::ObjectManager::sRegObj(this);
  m_bVisible = true;
  m_bLocked = false;
  m_bModified = false;

  m_nSceneID = qlib::invalid_uid;

  m_pEvtCaster = new ObjectEventCaster;
  addPropListener(this);

  MB_DPRINTLN("Object (%p/%d) created\n", this, m_uid);

}

Object::~Object()
{
  MB_DPRINTLN("Object(%p) destructed\n", this);
  delete m_pEvtCaster;
  qlib::ObjectManager::sUnregObj(m_uid);

  if (m_pReaderOpts!=NULL) delete m_pReaderOpts;
}

//////////

void Object::unloading()
{
}

LString Object::toString() const
{
  return LString::format("Object(name=%s)",m_name.c_str());
}

void Object::dump() const
{
  ScenePtr rscn = getScene();

  MB_DPRINT("Object: %s", toString().c_str());
  MB_DPRINT(": {");

  RendIter iter = beginRend();
  for (; iter!=endRend(); ++iter) {
    RendererPtr rrend = iter->second;
    if (!rrend.isnull()) {
      MB_DPRINT("rend %p/%d (nref=%d): ", rrend.get(), rrend->getUID(), rrend.use_count());
      //rrend->dump();
    }
    else {
      MB_DPRINTLN("(invalid rend %d)", iter->first);
    }
  }
  
  MB_DPRINTLN("}");
}

ScenePtr Object::getScene() const
{
  return SceneManager::getSceneS(m_nSceneID);
}

////////////////////////////////////////////////

RendererPtr Object::createRenderer(const LString &type_name)
{
  RendererFactory *pRF = RendererFactory::getInstance();
  RendererPtr rrend = pRF->create(type_name);
  rrend->resetAllProps();

  if (!registerRendererImpl(rrend)) {
    // error !! cannot register renderer.
    LOG_DPRINTLN("ERROR !! cannot register renderer");
    LString msg = LString::format("Cannot register renderer %s", type_name.c_str());
    MB_THROW(qlib::RuntimeException, msg);
    return RendererPtr();
  }

  return rrend;
}

RendererPtr Object::getRenderer(qlib::uid_t uid) const
{
  rendtab_t::const_iterator i = m_rendtab.find(uid);
  if (i==m_rendtab.end())
    return RendererPtr();

  return i->second;
}

RendererPtr Object::getRendererByType(const LString &type_name)
{
  rendtab_t::const_iterator i = m_rendtab.begin();
  for (; i!=m_rendtab.end(); ++i) {
    if ( type_name.equals(i->second->getTypeName()) )
      return i->second;
  }

  return RendererPtr();
}

RendererPtr Object::getRendererByIndex(int ind)
{
  rendtab_t::const_iterator i = m_rendtab.begin();
  while (ind>0) {
    ++i;
    --ind;
  }
  if (i!=m_rendtab.end())
    return i->second;

  return RendererPtr();
}

bool Object::destroyRenderer(qlib::uid_t uid)
{
  rendtab_t::iterator i = m_rendtab.find(uid);
  if (i==m_rendtab.end())
    return false;

  RendererPtr rrend = i->second;
  ScenePtr pScene = getScene();

  if (pScene.isnull() || rrend.isnull()) {
    LOG_DPRINTLN("Object::destroyRenderer> fatal error pScene or rrend is NULL!");
    return false;
  }

  // Detach the Scene from the renderer event source
  //bool res =
  //rrend->removePropListener(pScene.get());
  rrend->removeListener(pScene.get());
  //MB_ASSERT(res);

  // Fire the SCE_REND_REMOVING Event, before removing the renderer
  {
    MB_DPRINTLN("Object> Firing SCE_REND_REMOVING event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_REND_REMOVING);
    ev.setSource(getSceneID());
    ev.setTarget(rrend->getUID());
    pScene->fireSceneEvent(ev);
  }

  // remove the rend from the scene's cache list
  pScene->removeRendCache(rrend);

  qlib::uid_t objid = rrend->detachObj();
  MB_ASSERT(objid==this->m_uid);

  m_rendtab.erase(i);

  // Record undo/redo info
  UndoManager *pUM = pScene->getUndoMgr();
  if (pUM->isOK()) {
    ObjLoadEditInfo *pPEI = new ObjLoadEditInfo;
    pPEI->setupRendDestroy(getUID(), rrend);
    pUM->addEditInfo(pPEI);
  }

  return true;
}

LString Object::searchCompatibleRendererNames()
{
  LString ret;
  RendererFactory *pRF = RendererFactory::getInstance();
  std::list<LString> ls;
  int n = pRF->searchCompatibleRenderers(ObjectPtr(this), ls);
  if (n==0)
    return LString();
  return LString::join(",", ls);
}

bool Object::registerRendererImpl(RendererPtr rrend)
{
  bool res = m_rendtab.insert(rendtab_t::value_type(rrend->getUID(), rrend)).second;
  if (!res)
    return false;

  rrend->setSceneID(getSceneID());
  rrend->attachObj(this->m_uid);

  ScenePtr pScene = getScene();
  if (pScene.isnull())
    return true;
  
  // add the new rend to the scene's cache list
  pScene->addRendCache(rrend);
  
  // The scene observes events from the new renderer
  //rrend->addPropListener(pScene.get());
  rrend->addListener(pScene.get());

  // Fire the SCE_REND_ADDED Event
  {
    MB_DPRINTLN("Object> Firing SCE_REND_ADDED event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_REND_ADDED);
    ev.setSource(getSceneID());
    ev.setTarget(rrend->getUID());
    pScene->fireSceneEvent(ev);
  }
  
  // Record undo/redo info
  UndoManager *pUM = pScene->getUndoMgr();
  if (pUM->isOK()) {
    ObjLoadEditInfo *pPEI = new ObjLoadEditInfo;
    pPEI->setupRendCreate(getUID(), rrend);
    pUM->addEditInfo(pPEI);
  }

  return true;
}

qlib::LVarArray Object::getRendUIDArray() const
{
  qlib::LVarArray rval;
  if (m_rendtab.empty())
    return rval;
  
  rval.allocate(m_rendtab.size());

  rendtab_t::const_iterator i = m_rendtab.begin();
  rendtab_t::const_iterator end = m_rendtab.end();
  for (int j=0; i!=end; ++i, ++j)
    rval.setIntValue(j, i->first);

  return rval;
}

qlib::LVarArray Object::getRendArray() const
{
  qlib::LVarArray rval;
  if (m_rendtab.empty())
    return rval;
  
  rval.allocate(m_rendtab.size());

  rendtab_t::const_iterator i = m_rendtab.begin();
  rendtab_t::const_iterator end = m_rendtab.end();
  for (int j=0; i!=end; ++i, ++j) {
    RendererPtr pObj = i->second;
    rval.setObjectPtr(j, pObj.copy());
  }

  return rval;
}

////////////////////////////////

int Object::addListener(ObjectEventListener *pL)
{
  return m_pEvtCaster->add(pL);
}

bool Object::removeListener(ObjectEventListener *pL)
{
  return m_pEvtCaster->remove(pL);
}

void Object::fireObjectEvent(ObjectEvent &ev)
{
  m_pEvtCaster->replicaFire(ev);

  ScrEventManager *pSEM = ScrEventManager::getInstance();
  ev.setSource(m_nSceneID);
  pSEM->fireObjectEvent(ev);
}

qlib::uid_t Object::getRootUID() const
{
  return getUID();
}

void Object::propChanged(qlib::LPropEvent &ev)
{
  // Record undo/redo info, if the txn is active
  ScenePtr cursc = getScene();
  if (!cursc.isnull()) {

    UndoManager *pUM = cursc->getUndoMgr();
    if (pUM->isOK()) {
      PropEditInfo *pPEI = new PropEditInfo;
      pPEI->setup(getUID(), ev.getName(), ev); //ev.getOldValue(), ev.getNewValue());
      pUM->addEditInfo(pPEI);
    }
  }
  
  // propagate to object event
  {
    ObjectEvent obe;
    obe.setType(ObjectEvent::OBE_PROPCHG);
    obe.setTarget(getUID());
    obe.setDescr(ev.getName());
    obe.setPropEvent(&ev);
    fireObjectEvent(obe);
  }
  return;
}

////////////////////////////////

void Object::writeTo2(qlib::LDom2Node *pNode) const
{
  // write properties of object
  super_t::writeTo2(pNode);

  RendIter riter = beginRend();
  for (; riter!=endRend(); ++riter) {
    RendererPtr obj = riter->second;
    MB_DPRINTLN("*** writeTo2 renderer %s", obj->getTypeName());

    qlib::LDom2Node *pChNode = pNode->appendChild("renderer");

    // renderer type="nickname"
    pChNode->setTypeName( obj->getTypeName() );

    // always in child element
    pChNode->setAttrFlag(false);

    obj->writeTo2(pChNode);
  }

  // source and source type
  pNode->appendStrAttr("srctype", getSourceType());

  LString src_str = getSource();
  // LString alt_src_str = getAltSource();

  if (!src_str.startsWith("datachunk:")) {
  // convert to relative path from basedir, if possible.
    LString basedir;
    ScenePtr cursc = getScene();
    if (!cursc.isnull()) {
      basedir = cursc->getBasePath();
    }

    if (!basedir.isEmpty()) {
      src_str = qlib::makeRelativePath(src_str, basedir);
    }

    // Set path information (possibly in relative form)
    pNode->appendStrAttr("src", src_str);

    // Make an alternative path representation (in absolute form)
    if (!basedir.isEmpty()) {
      LString alt_src_str = qlib::makeAbsolutePath(src_str, basedir);
      if (! alt_src_str.equals(src_str))
        pNode->appendStrAttr("altsrc", alt_src_str);
    }

    // append reader options
    if (m_pReaderOpts!=NULL) {
      qlib::LDom2Node *pChNode = new qlib::LDom2Node(*m_pReaderOpts);
      pNode->appendChild(pChNode);
    }
  }
  else {
    // embedded --> no path name conv, reader opts, and altpath aren't required
    pNode->appendStrAttr("src", src_str);
  }
}

void Object::readFrom2(qlib::LDom2Node *pNode)
{
  LString src = pNode->getStrAttr("src");
  if (!src.isEmpty()) {
    pNode->removeChild("src");
    setSource(src);
  }
  LString altsrc = pNode->getStrAttr("altsrc");
  if (!altsrc.isEmpty()) {
    pNode->removeChild("altsrc");
    setAltSource(altsrc);
  }

  LString srctype = pNode->getStrAttr("srctype");
  if (!srctype.isEmpty()) {
    pNode->removeChild("srctype");
    setSourceType(srctype);
  }

  // reader options
  qlib::LDom2Node *pRopts = pNode->findChild("ropts");
  if (pRopts!=NULL) {
    qlib::LDom2Node *pRoptsCopy = new qlib::LDom2Node(*pRopts);
    setReaderOpts(pRoptsCopy);
  }

  super_t::readFrom2(pNode);

  RendererFactory *pRF = RendererFactory::getInstance();

  for (pNode->firstChild(); pNode->hasMoreChild(); pNode->nextChild()) {
    qlib::LDom2Node *pChNode = pNode->getCurChild();
    LString tag = pChNode->getTagName();
    LString type_name = pChNode->getTypeName();

    if (tag.equals("renderer") && !type_name.isEmpty()) {
      // TO DO: catch exception and report error here.

      //RendererPtr rrend = createRenderer(type_name);
      RendererPtr rrend = pRF->create(type_name);
      rrend->resetAllProps();

      // Renderer's properties should be built before registration to the scene,
      //   to prevent event propargation.
      rrend->readFrom2(pChNode);
      if (!pChNode->isChildrenConsumed()) {
	// TO DO: report error (unknown element)
	LOG_DPRINTLN("Object::readFrom2> Warning:"
		     " some nodes (of rend) are not consumed");
        //pChNode->dump();
      }

      // Register the built renderer
      if (!registerRendererImpl(rrend)) {
        // error !! cannot register renderer.
        LOG_DPRINTLN("ERROR !! cannot register renderer");
        LString msg = LString::format("Cannot register renderer %s", type_name.c_str());
        MB_THROW(qlib::RuntimeException, msg);
        // return RendererPtr();
      }

    }
    else {
      continue;
    }

    pChNode->setConsumed(true);
  }

}

//////////

void Object::setReaderOpts(qlib::LDom2Node *ptree)
{
  if (m_pReaderOpts!=NULL) {
    MB_DPRINTLN("Object> Warning: previous reader options is deleted");
    delete m_pReaderOpts;
  }

  m_pReaderOpts = ptree;
}

////////////////////////////////////////////////////////////
//
// Object Extension Data implementation
//

ObjExtData::ObjExtData()
{
}

ObjExtData::ObjExtData(const ObjExtData &arg)
{
}

ObjExtData::~ObjExtData()
{
}

//////////

int Object::getExtDataSize() const
{
  int n = m_extdat.size();
  return n;
}

ObjExtDataPtr Object::getExtData(const LString &name) const
{
  ObjExtDataPtr p = m_extdat.get(name);
  return p;
}

void Object::removeExtData(const LString &name)
{
  m_extdat.remove(name);
}

void Object::setExtData(const LString &name, ObjExtDataPtr p)
{
  m_extdat.forceSet(name, p);
}

void Object::forceEmbed()
{
  m_source = "";
  m_altsrc = "";
  m_sourcetype = "";
  if (m_pReaderOpts!=NULL) delete m_pReaderOpts;
  m_pReaderOpts = NULL;
}

