// -*-Mode: C++;-*-
//
//  Pov-Ray Display context implementation class
//

#ifndef POV_DISPLAY_CONTEXT_H__
#define POV_DISPLAY_CONTEXT_H__

#include "qsys.hpp"

#include <gfx/DisplayContext.hpp>
//#include "LColor.hpp"
//#include "LColorTable.hpp"
#include <qlib/LString.hpp>

namespace qlib {
class OutStream;
}

namespace qsys {

class PovIntData;

class PovDisplayContext : public gfx::DisplayContext
{

protected:
  /// output pov file
  qlib::OutStream *m_pPovOut;

  /// output inc file
  qlib::OutStream *m_pIncOut;

  /// output inc filename
  LString m_incFileName;

  /// internal data structure
  PovIntData *m_pIntData;

  double m_dZoom;
  double m_dViewDist;
  double m_dSlabDepth;

  /// bg color
  gfx::ColorPtr m_bgcolor;

  /// matrix stack
  std::deque<Matrix4D> m_matstack;

  /// unitarity of the current transformation
  bool m_bUnitary;

  /// current line width
  double m_linew;

  /// current color
  gfx::ColorPtr m_pColor;

  /// current normal vec
  Vector4D m_norm;

  /// current drawing mode
  int m_nDrawMode;

  enum {
    POV_NONE,
    POV_POINTS,
    POV_POLYGON,
    POV_LINES,
    POV_LINESTRIP,
    POV_TRIGS,
    POV_TRIGSTRIP,
    POV_TRIGFAN,
    POV_QUADS,
    POV_QUADSTRIP
  };

  bool m_fPrevPosValid;
  Vector4D m_prevPos, m_prevCol, m_prevNorm;

  int m_nTriIndex;

  // /// output texture type
  // bool m_fUseTexBlend;

  /// projection type (perspective)
  bool m_fPerspective;

  /// detail level for sphere/cone obj
  int m_nDetail;

  /// Tolerance of unitarity check
  double m_dUniTol;

  /// Z-dir slab clipping flag
  bool m_bUseClipZ;

public:
  PovDisplayContext();
  virtual ~PovDisplayContext();

  //////////////////////////////
  // generic implementation

  virtual void vertex(double x, double y, double z);
  virtual void vertex(const Vector4D &v);

  virtual void normal(double x, double y, double z);
  virtual void normal(const Vector4D &v);
  virtual void color(const gfx::ColorPtr &c);

  // virtual void color(double r, double g, double b, double a);
  
  virtual void pushMatrix();
  virtual void popMatrix();
  virtual void multMatrix(const Matrix4D &mat);
  virtual void loadMatrix(const Matrix4D &mat);
  
  ////////////////

  virtual void setLineWidth(double lw);
  virtual void setLineStipple(unsigned short pattern);
  virtual void setLighting(bool f=true);

  virtual void setPointSize(double size);

  //virtual void loadName(int nameid);
  //virtual void pushName(int nameid);
  //virtual void popName();

  ////////////////
  // line and triangles

  virtual void setPolygonMode(int id);
  virtual void startPoints();
  virtual void startPolygon();
  virtual void startLines();
  virtual void startLineStrip();
  virtual void startTriangles();
  virtual void startTriangleStrip();
  virtual void startTriangleFan();
  virtual void startQuadStrip();
  virtual void startQuads();
  virtual void end();

  ///////////////////////////////
  // higher-order objects
  
  /// Display unit sphere
  virtual void sphere();
  
  /// Display sphere with radius of r at position vec
  virtual void sphere(double r, const Vector4D &vec);

  /// Display cone (and cylinder)
  virtual void cone(double r1, double r2,const Vector4D &pos1,
                    const Vector4D &pos2,bool bCap);

  virtual void setDetail(int n);
  virtual int getDetail() const;

  /// Mesh drawing
  virtual void drawMesh(const gfx::Mesh &);

  ///////////////////////////////
  // Display List (not supported)

  virtual DisplayContext *createDisplayList() { return NULL; }
  virtual bool canCreateDL() const { return false; }
  virtual void callDisplayList(DisplayContext *pdl) {}
  virtual bool isCompatibleDL(DisplayContext *pdl) const { return false; }

  virtual bool isDisplayList() const { return false; }
  virtual bool recordStart() { return false; }
  virtual void recordEnd() {}

  ///////////////////////////////

  virtual qsys::View *getTargetView() const { return NULL; }

  void startRender();
  void endRender();

  virtual void startSection(const LString &name);
  virtual void endSection();

  virtual bool setCurrent();
  virtual bool isCurrent() const;

  // // POV doesn't support clearing img!!
  // virtual void clear(const LColor &color) {}

  ////////////////////////////////////////////////////////////
  // POV-Ray implementation

  void init(qlib::OutStream *pPovOut, qlib::OutStream *pIncOut);

  void setZoom(double z) { m_dZoom = z; }
  void setViewDist(double d) { m_dViewDist = d; }
  void setSlabDepth(double d) { m_dSlabDepth = d; }

  void setBgColor(const gfx::ColorPtr &c) { m_bgcolor = c; }

  // void setTexBlend(bool f) { m_fUseTexBlend = f; }
  void setClipZ(bool f) { m_bUseClipZ = f; }
  void setPerspective(bool f) { m_fPerspective = f; }
  void setIncFileName(const LString &name) { m_incFileName = name; }

  const gfx::ColorPtr &getCurrentColor() const { return m_pColor; }

private:
  void checkUnitary();

  void xform_vec(Vector4D &v) {
    const Matrix4D &mtop = m_matstack.front();
    v.w() = 1.0;
    mtop.xform4D(v);
  }

  void xform_norm(Vector4D &v) {
    const Matrix4D &mtop = m_matstack.front();
    v.w() = 0.0;
    mtop.xform4D(v);
  }

  void writeHeader();
  void writeTailer();

  /// Draw a single line segment from v1 to v2 to the output
  /// v1 and v2 should be transformed by matrix stack
  void drawLine(const Vector4D &v1, const Vector4D &v2);

  //void drawMesh(const Vector4D &v1, const Vector4D &v2, const Vector4D &v3,
  //const Vector4D &n1, const Vector4D &n2, const Vector4D &n3,
  //const Vector4D &c1, const Vector4D &c2, const Vector4D &c3);
};

}

#endif

