// -*-Mode: C++;-*-
//
//  Povray display context implementation
//
//  $Id: PovIntData.cpp,v 1.16 2011/04/17 10:56:39 rishitani Exp $

#include <common.h>

#include "PovIntData.hpp"
#include <qlib/PrintStream.hpp>
#include <gfx/SolidColor.hpp>
#include <gfx/Mesh.hpp>
#include "PovDisplayContext.hpp"
#include "style/StyleMgr.hpp"

using namespace qsys;

PovIntData::PovIntData(PovDisplayContext *pdc)
     : m_pPovOut(NULL), m_pIncOut(NULL)
{
  m_pdc = pdc;
  m_dClipZ = -1.0;
  m_mesh.m_pPar = this;
}

PovIntData::~PovIntData()
{
  std::for_each(m_lines.begin(), m_lines.end(), qlib::delete_ptr<Line *>());
  std::for_each(m_cylinders.begin(), m_cylinders.end(), qlib::delete_ptr<Cyl *>());
  std::for_each(m_spheres.begin(), m_spheres.end(), qlib::delete_ptr<Sph *>());
}

/////

void PovIntData::start(OutStream *fp, OutStream *ifp, const char *name)
{
  m_pPovOut = fp;
  m_pIncOut = ifp;
  m_name = name;
  m_fUseTexBlend = true;
  //false; //fTexBlend;
}

void PovIntData::meshStart()
{
  m_nMeshPivot = m_mesh.getVertexSize();
}

void PovIntData::meshEndTrigs()
{
  int nVerts = m_mesh.getVertexSize() - m_nMeshPivot;
  if (nVerts%3!=0) {
    LOG_DPRINTLN("PovIntData> Trig mesh: nVerts%3!=0 !!");
  }
  int nFaces = nVerts/3;
  int i;
  for (i=0; i<nFaces; i++) {
    m_mesh.addFace(m_nMeshPivot + i*3+0,
                   m_nMeshPivot + i*3+1,
                   m_nMeshPivot + i*3+2);
  }
}

void PovIntData::meshEndTrigStrip()
{
  int nVerts = m_mesh.getVertexSize() - m_nMeshPivot;
  if (nVerts<3) {
    LOG_DPRINTLN("PovIntData> TrigStrip mesh: nVerts<3 !!");
  }
  int i;
  for (i=2; i<nVerts; i++) {
    if (i%2==0) {
      m_mesh.addFace(m_nMeshPivot + i-2,
                  m_nMeshPivot + i-1,
                  m_nMeshPivot + i);
    }
    else {
      m_mesh.addFace(m_nMeshPivot + i,
                  m_nMeshPivot + i-1,
                  m_nMeshPivot + i-2);
    }
  }
}

void PovIntData::meshEndFan()
{
  int nVerts = m_mesh.getVertexSize() - m_nMeshPivot;
  if (nVerts<3) {
    LOG_DPRINTLN("PovIntData> TrigFan mesh: nVerts<3 !!");
  }
  int i;
  for (i=2; i<nVerts; i++) {
    m_mesh.addFace(m_nMeshPivot + i-1,
                   m_nMeshPivot + i,
                   m_nMeshPivot);
  }
}

void PovIntData::mesh(const Matrix4D &mat, const gfx::Mesh &rmesh)
{
	int i;
  const int nverts = rmesh.getVertSize();
  const int nfaces = rmesh.getFaceSize();
  const int *pinds = rmesh.getFaces();

  const int ivstart = m_mesh.getVertexSize();

  Vector4D v, n;
  ColorPtr col;
  for (i=0; i<nverts; ++i) {
    v = rmesh.getVertex(i);
    mat.xform3D(v);
    n = rmesh.getNormal(i);
    mat.xform4D(n);

    if (!rmesh.getCol(col, i))
      col = m_pdc->getCurrentColor();

    /*
      LString colmtr = col->getMaterial();
      if (colmtr.isEmpty())
        colmtr = pdc->getMaterial();
      p->c = m_clut.newColor(col, colmtr);
     */

    m_mesh.addVertex(v, n, col);
  }

  for (i=0; i<nfaces; ++i) {
    m_mesh.addFace(pinds[i*3+0] + ivstart,
                   pinds[i*3+1] + ivstart,
                   pinds[i*3+2] + ivstart);
  }

}

void PovIntData::setMeshMode(int n)
{
  if (m_nMeshMode==n)
    return;
  /*if (m_mesh.size()>0) {
    // TO DO: throw exception
    LOG_DPRINTLN("POVWriter> ERROR: mesh mode cannot be changed during rendering.");
  }*/
  
  m_nMeshMode = n;
}

/// Append line segment
void PovIntData::line(const Vector4D &v1, const Vector4D &v2, double width)
{
  Line *p = new Line;
  p->v1 = v1;
  p->v2 = v2;
  convCol(p->col);
  p->w = width;
  m_lines.push_back(p);
}

/// Append cylinder
void PovIntData::cylinder(const Vector4D &v1, const Vector4D &v2,
                          double w1, double w2,
                          int ndet, const Matrix4D *ptrf)
{
  Cyl *p = new Cyl;
  p->v1 = v1;
  p->v2 = v2;
  convCol(p->col);
  p->w1 = w1;
  p->w2 = w2;
  p->ndetail = ndet;
  if (ptrf==NULL)
    p->pTransf = NULL;
  else
    p->pTransf = new Matrix4D(*ptrf);
  m_cylinders.push_back(p);
}

void PovIntData::sphere(const Vector4D &v1, double w, int ndet)
{
  Sph *p = new Sph;
  p->v1 = v1;
  convCol(p->col);
  p->r = w;
  p->ndetail = ndet;
  m_spheres.push_back(p);
}

/// convert LColor to CLUT entry
bool PovIntData::convCol(PovIntColor &ic)
{
  const ColorPtr &pcol = m_pdc->getCurrentColor();
  LString colmtr = pcol->getMaterial();
  if (colmtr.isEmpty())
    colmtr = m_pdc->getMaterial();
  ic = m_clut.newColor(pcol, colmtr);
  return true;
}

/// Convert color to internal representation (2)
bool PovIntData::convCol(const ColorPtr &pcol, PovIntColor &ic)
{
  LString colmtr = pcol->getMaterial();
  if (colmtr.isEmpty())
    colmtr = m_pdc->getMaterial();
  ic = m_clut.newColor(pcol, colmtr);
  return true;
}

/// dump CLUT to POV file
void PovIntData::dumpClut(OutStream *fp)
{
  PrintStream ps(*fp);
  StyleMgr *pSM = StyleMgr::getInstance();

  int i;
  for (i=0; i<m_clut.size(); i++) {
    PovIntColor cind;
    cind.cid1 = i;
    cind.cid2 = -1;

    // get color
    Vector4D vc;
    m_clut.getRGBAVecColor(cind, vc);
      
    // write color
    ps.format("#declare %s_col_%d = ", m_name.c_str(), i);
    if (qlib::Util::isNear4(vc.w(), 1.0))
      ps.format("<%f,%f,%f>;\n", vc.x(), vc.y(), vc.z());
    else
      ps.format("<%f,%f,%f,%f>;\n", vc.x(), vc.y(), vc.z(), 1.0-vc.w());

    // get material
    LString mat;
    m_clut.getMaterial(cind, mat);
    if (mat.isEmpty()) mat = "default";
    LString matdef = pSM->getMaterial(mat, "pov");
    matdef = matdef.trim(" \r\n\t");
      
    // write material
    if (!m_fUseTexBlend) {
      ps.format("#declare %s_tex_%d = ", m_name.c_str(), i);
      if (!matdef.isEmpty())
        ps.println(matdef);
      else
        ps.println("texture {}");
    }
    else {
      ps.format("#declare %s_tex_%d = ", m_name.c_str(), i);

      LString colortext = LString::format("%s_col_%d",
                                          m_name.c_str(), i);
      if (!matdef.isEmpty()) {
        matdef.replace("@COLOR@", colortext);
        ps.println(matdef);
      }
      else
        ps.println("texture {pigment{color rgbt "+colortext+"}}");
    }
  }
}

void PovIntData::writeColor(const PovIntColor &ic)
{
  PrintStream ips(*m_pIncOut);

  const char *nm = m_name.c_str();
  if (ic.cid2<0) {
    //
    // Non-gradient color
    //
    if (m_fUseTexBlend) {
      ips.format("texture{%s_tex_%d}", nm, ic.cid1);
    }
    else {
      ips.format("texture{%s_tex_%d ", nm, ic.cid1);
      ips.format("pigment {color rgbt %s_col_%d}}", nm, ic.cid1);
    }
  }
  else{
    //
    // Gradient color
    //
    if (m_fUseTexBlend) {
      ips.format(
        "texture{function{%.6f}texture_map{[0 %s_tex_%d][1 %s_tex_%d]}}",
        1.0f-ic.getRhoF(), nm, ic.cid1, nm, ic.cid2);
    }
    else
    {
      ips.format("texture{%s_tex_%d ", nm, ic.cid1);
      ips.format(
        "pigment {color rgbt %s_col_%d*%f+%s_col_%d*%f}}",
        nm, ic.cid1, ic.getRhoF(), nm, ic.cid2, 1.0-ic.getRhoF());
    }
  }
}

bool PovIntData::writeLines(PrintStream &ps, PrintStream &ips)
{
  if (m_lines.size()<=0)
    return false;

  //PrintStream ps(*m_pPovOut);
  //PrintStream ips(*m_pIncOut);

  const double lw = 0.02;

  // write INC file
  BOOST_FOREACH(Line *p, m_lines) {

    Vector4D v1 = p->v1, v2 = p->v2;
    double w = p->w;

    // always keep v1.z < v2.z
    if (v1.z()>v2.z())
      std::swap(v1, v2);

    Vector4D nn = v2 - v1;
    double len = nn.length();
    if (len<=F_EPS4) {
      // ignore degenerated cylinder
      delete p;
      continue;
    }
    
    if (m_dClipZ<0 || m_dClipZ > v1.z()) {

      if (m_dClipZ < v2.z())
        v2 = nn.scale((m_dClipZ-v1.z())/(nn.z())) + v1;

      ips.format("cylinder{<%f, %f, %f>, ", v1.x(), v1.y(), v1.z());
      ips.format("<%f, %f, %f>, ", v2.x(), v2.y(), v2.z());
      ips.format("%s_lw*%f ", m_name.c_str(), w);
      writeColor(p->col);

      ips.format("}\n");
    }
    delete p;
  }

  m_lines.erase(m_lines.begin(), m_lines.end());

  // write POV file
  ps.format("#declare %s_tex = texture {\n", m_name.c_str());
  ps.format("  pigment {color rgbft <0,0,0,1,1>}\n");
  ps.format("  normal {granite 0.0 scale 1.0}\n");
  ps.format("  finish {\n");
  ps.format("   ambient 0.3\n");
  ps.format("   diffuse 1.0\n");
  ps.format("   specular 0.0\n");
  ps.format("  }\n");
  ps.format(" }\n");
  ps.format("#declare %s_lw = %.3f;\n", m_name.c_str(), lw);

  return true;
}

/// convert gradient-map index to the POV's mesh texture index
int PovIntData::convTexInd(MeshElem *p1)
{
  if ( p1->c.cid2 < 0 ) {
    return p1->c.cid1;
  }
  else {
    int gind = m_clut.getGradIndex(p1->c);
    if (gind<0) {
      LOG_DPRINTLN("FATAL ERROR: cannot convert gradient index!!");
      return 0;
    }

    return m_clut.size() + gind*256 + p1->c.getRhoI();
  }
}

void PovIntData::writeTextureList()
{
  PrintStream ips(*m_pIncOut);

  int txsize = m_clut.size() + m_clut.m_grads.size()*256;
  ips.format("texture_list{ %d", txsize);

  // write simple textures
  for (int i=0; i<m_clut.size(); i++) {
    if (i%2==0)
      ips.format(",\n");
    else
      ips.format(",");

    /*
    if (m_fUseTexBlend)
      ips.format("texture{%s_tex%d}", nm, i);
    else
      ips.format("texture{%s pigment{color rgbt %s_col%d}}",
                 txnm, nm, i);
     */
    PovIntColor pic;
    pic.cid1 = i;
    pic.cid2 = -1;
    writeColor(pic);
  }
}

void PovIntData::writeGradTexture()
{
  PrintStream ips(*m_pIncOut);

  m_clut.indexGradients();

  BOOST_FOREACH(ColorTable::grads_type::value_type &entry, m_clut.m_grads) {
    //ColorTable::grads_type::const_iterator gmi = m_clut.m_grads.begin();
    //for ( ; gmi!=m_clut.m_grads.end(); gmi++) {
    //const PovIntColor &gd = gmi->first;

    const PovIntColor &gd = entry.first;
    for (int j=0; j<256; j++) {
      double rho = double(j)/255.0f;
      ips.format(",\n");
      //if (m_fUseTexBlend)
      //ips.format(
      //"texture{function{%.6f}texture_map{[0 %s_tex%d][1 %s_tex%d]}}",
      //1.0f-rho, nm, gd.cid1, nm, gd.cid2);
      //else 
      //ips.format(
      //"texture{%s pigment {color rgbt %s_col%d*%f+%s_col%d*%f}}",
      //txnm, nm, gd.cid1, rho, nm, gd.cid2, 1.0f-rho);
      PovIntColor pic;
      pic.cid1 = gd.cid1;
      pic.cid2 = gd.cid2;
      pic.setRhoI(j);
      writeColor(pic);
    }
  }
  ips.format("}\n");
}

PovIntData::Mesh *PovIntData::calcMeshClip()
{
  int i, j;
  int nverts = m_mesh.getVertexSize();
  int nfaces = m_mesh.getFaceSize();

  std::vector<int> vidmap(nverts);
  
  PovIntData::Mesh *pMesh2 = new PovIntData::Mesh();
  pMesh2->m_pPar = this;

  std::vector<MeshElem *> verts(nverts);
  bool bNone = true;
  i=0;
  BOOST_FOREACH (MeshElem *p, m_mesh.m_verts) {
    verts[i] = p;
    if (p->v.z()>=m_dClipZ) {
      // clipped
      vidmap[i] = -1;
      bNone = false;
      ++i;
      continue;
    }

    vidmap[i] = 1;
    ++i;
  }  

  if (bNone) {
    delete pMesh2;
    return NULL;
  }

  i=0; j=0;
  BOOST_FOREACH (MeshElem *p, m_mesh.m_verts) {
    if (vidmap[i]<0) {
      ++i;
      continue;
    }
    vidmap[i] = j;
    pMesh2->copyVertex(p);
    ++j;
    ++i;
  }  

  std::deque<int>::iterator iter2 = m_mesh.m_faces.begin();
  std::deque<int>::iterator iend2 = m_mesh.m_faces.end();
  int iv[3];
  int jv[3];
  bool b[3];
  for (i=0; iter2!=iend2; iter2++, i++) {
    iv[0] = *iter2;
    iter2++; MB_ASSERT(iter2!=m_mesh.m_faces.end());
    iv[1] = *iter2;
    iter2++; MB_ASSERT(iter2!=m_mesh.m_faces.end());
    iv[2] = *iter2;

    for (j=0; j<3; ++j)
      jv[j] = vidmap[iv[j]];
    
    for (j=0; j<3; ++j)
      b[j] = (jv[j]<0);
    
    if (b[0] && b[1] && b[2]) {
      // outside clip
      continue;
    }
    else if (!b[0] && !b[1] && !b[2]) {
      // inside clip
      pMesh2->addFace(jv[0], jv[1], jv[2]);
      continue;
    }
    
    for (j=0; j<3; ++j) {

      MeshElem *pv1 = verts[ iv[j] ];
      MeshElem *pv2 = verts[ iv[(j+1)%3] ];
      MeshElem *pv3 = verts[ iv[(j+2)%3] ];
      
      // check single clipped triangles
      if (!b[j] && b[(j+1)%3] && b[(j+2)%3]) {
        int jin = jv[j];
        MeshElem *pv1 = verts[ iv[j] ];
        MeshElem *pv2 = verts[ iv[(j+1)%3] ];
        MeshElem *pv3 = verts[ iv[(j+2)%3] ];
        int inv2 = pMesh2->addVertex( cutEdge(pv1, pv2) );
        int inv3 = pMesh2->addVertex( cutEdge(pv1, pv3) );
        pMesh2->addFace(jin, inv2, inv3);
        break;
      }

      // check double clipped triangles
      // TO DO: select convex-hull triangles
      if (!b[j] && !b[(j+1)%3] && b[(j+2)%3]) {
        int jin0 = jv[j];
        int jin1 = jv[(j+1)%3];
        int ed1 = pMesh2->addVertex( cutEdge(pv3, pv1) );
        int ed2 = pMesh2->addVertex( cutEdge(pv3, pv2) );

        //if (select_trig(m_verts[nin0], m_verts[ed1], m_verts[nin1], m_verts[ed2])) {
        pMesh2->addFace(jin0, jin1, ed1);
        pMesh2->addFace(ed2, ed1, jin1);
        //}
        //else {
        //addNewFace(nin0, nin1, ed2);
        //addNewFace(ed2, ed1, nin0);
        //}
        break;
      }
      
    }
  }

  return pMesh2;
}

PovIntData::MeshElem *PovIntData::cutEdge(MeshElem *pv1, MeshElem *pv2)
{
  const Vector4D &v1 = pv1->v;
  const Vector4D &v2 = pv2->v;
  
  const Vector4D &n1 = pv1->n;
  const Vector4D &n2 = pv2->n;
  
  MeshElem *pNew = new MeshElem;
  const double t = (m_dClipZ-v1.z())/(v2.z()-v1.z());
  pNew->v = v2.scale(t) + v1.scale(1.0-t);
  
  Vector4D norm = n2.scale(t) + n1.scale(1.0-t);
  pNew->n = norm.normalize();
  
  pNew->c = pv1->c;
  return pNew;
}


bool PovIntData::writeMeshes(PrintStream &ps, PrintStream &ips)
{
  int i, j;
  const char *nm = m_name.c_str();

  if (m_mesh.getVertexSize()<=0 || m_mesh.getFaceSize()<1)
    return false;
  
  Mesh *pMesh = &m_mesh;
  bool bdel = false;
  if (m_dClipZ>0.0) {
    Mesh *pRes = calcMeshClip();
    if (pRes!=NULL) {
      pMesh = pRes;
      bdel = true;
    }
  }

  int nverts = pMesh->getVertexSize();
  int nfaces = pMesh->getFaceSize();

  // convert vertex list to array
  MeshElem **pmary = new MeshElem *[nverts];
  i=0;
  BOOST_FOREACH (MeshElem *pelem, pMesh->m_verts) {
    pmary[i] = pelem;
    ++i;
  }
  
  //
  // generate mesh2 statement
  //
  
  ips.format("mesh2{\n");
  
  // write vertex_vectors
  ips.format("vertex_vectors{ %d", nverts);
  for (i=0; i<nverts; i++) {
    MeshElem *p = pmary[i];
    if (i%6==0)
      ips.format(",\n");
    else
      ips.format(",");
    ips.format("<%f, %f, %f>", p->v.x(), p->v.y(), p->v.z());
  }
  ips.format("}\n");
  
  // write normal_vectors
  ips.format("normal_vectors{ %d", nverts);
  for (i=0; i<nverts; i++) {
    MeshElem *p = pmary[i];
    if (i%6==0)
      ips.format(",\n");
    else
      ips.format(",");
    ips.format("<%f, %f, %f>", p->n.x(), p->n.y(), p->n.z());
  }
  ips.format("}\n");
  
  //
  // write texture_list
  //
  writeTextureList();

  //
  // write gradient textures
  //
  writeGradTexture();

  //
  // write face_indices
  //
  ips.format("face_indices{ %d", nfaces);

  std::deque<int>::iterator iter2 = pMesh->m_faces.begin();
  std::deque<int>::iterator iend2 = pMesh->m_faces.end();
  for (i=0; iter2!=iend2; iter2++, i++) {
    int i1 = *iter2;
    iter2++; MB_ASSERT(iter2!=pMesh->m_faces.end());
    int i2 = *iter2;
    iter2++; MB_ASSERT(iter2!=pMesh->m_faces.end());
    int i3 = *iter2;
    
    if (i%4==0)
      ips.format(",\n");
    else
      ips.format(",");
    
    ips.format("<%d,%d,%d>,%d,%d,%d",
               i1, i2, i3,
               convTexInd(pmary[i1]),
               convTexInd(pmary[i2]),
               convTexInd(pmary[i3]));
  }
  ips.format("}\n");
  
  ips.format("}\n");
  
  //
  // clean up
  //
  if (bdel)
    delete pMesh;
  delete [] pmary;

  m_mesh.clear();

  return true;
}

bool PovIntData::writeSpheres(PrintStream &ps, PrintStream &ips)
{
  if (m_spheres.size()<=0)
    return false;

  BOOST_FOREACH (Sph *p, m_spheres) {

    if (m_dClipZ<0 || m_dClipZ > p->v1.z()) {
      ips.format("sphere{<%f, %f, %f>, ", p->v1.x(), p->v1.y(), p->v1.z());
      ips.format("%f ", p->r);
      writeColor(p->col);
      if (m_dClipZ < (p->v1.z() + p->r)) {
        ips.format("\n bounded_by {");
        ips.format("  plane {z, %f} ", m_dClipZ);
        ips.format("}");
        ips.format("clipped_by { bounded_by }");
      }
      ips.format("}\n");
    }
    
    delete p;
  }

  m_spheres.erase(m_spheres.begin(), m_spheres.end());
  return true;
}

bool PovIntData::writeCyls(PrintStream &ps, PrintStream &ips)
{
  if (m_cylinders.size()<=0)
    return false;

  BOOST_FOREACH (Cyl *p, m_cylinders) {

    Vector4D v1 = p->v1, v2 = p->v2;
    double w1 = p->w1, w2 = p->w2;
    Vector4D nn = v1 - v2;
    double len = nn.length();
    if (len<=F_EPS4) {
      // ignore the degenerated cylinder
      delete p;
      continue;
    }

    // elongate cyl. for connectivity
    nn = nn.scale(1.0/len);
    v1 += nn.scale(0.01);
    v2 -= nn.scale(0.01);

    // always keep v1.z < v2.z
    if (v1.z()>v2.z()) {
      std::swap(v1, v2);
      std::swap(w1, w2);
    }
    //const double dz = v2.z() - v1.z();
    //const double dh = qlib::abs(v2.y() - v1.y());
    //const double l = ::sqrt(dz*dz+dh*dh);
    const double delw = qlib::max(w1, w2);
    const double thr1 = v1.z() - delw;
    const double thr2 = v2.z() + delw;

    MB_ASSERT(v2.z()>v1.z());

    if (p->pTransf!=NULL || m_dClipZ<0 || m_dClipZ > thr1) {
      if (qlib::isNear4(w1, w2)) {
        // cyliner
        ips.format("cylinder{<%f,%f,%f>,", v1.x(), v1.y(), v1.z());
        ips.format("<%f,%f,%f>,", v2.x(), v2.y(), v2.z());
        ips.format("%s_lw*%f ", m_name.c_str(), w1);
      }
      else {
        // cone
        ips.format("cone{<%f,%f,%f>,", v1.x(), v1.y(), v1.z());
        ips.format("%s_lw*%f,", m_name.c_str(), w1);
        ips.format("<%f,%f,%f>,", v2.x(), v2.y(), v2.z());
        ips.format("%s_lw*%f ", m_name.c_str(), w2);
      }

      writeColor(p->col);

      if (p->pTransf!=NULL) {
        const Matrix4D &m = *p->pTransf;
        ips.format(" matrix <");
        ips.format("%f, %f, %f, ", m.aij(1,1), m.aij(2,1), m.aij(3,1) );
        ips.format("%f, %f, %f, ", m.aij(1,2), m.aij(2,2), m.aij(3,2) );
        ips.format("%f, %f, %f, ", m.aij(1,3), m.aij(2,3), m.aij(3,3) );
        ips.format("%f, %f, %f>" , m.aij(1,4), m.aij(2,4), m.aij(3,4) );
      }
      else if (m_dClipZ < thr2) {
        ips.format("\n bounded_by {");
        ips.format("  plane {z, %f} ", m_dClipZ);
        ips.format("}");
        ips.format("clipped_by { bounded_by }");
      }

      ips.format("}\n");
    }
    delete p;
  }

  m_cylinders.erase(m_cylinders.begin(), m_cylinders.end());
  return true;
}

void PovIntData::end()
{
  bool bcyl = false, bsph = false, blin = false, bmes = false;

  if (m_cylinders.size()<=0 &&
      m_spheres.size()<=0 &&
      m_mesh.getVertexSize()<=0 &&
      m_mesh.getFaceSize()<=0 &&
      m_lines.size()<=0 &&
      m_clut.size()<=0) {
    return;
  }

  PrintStream ps(*m_pPovOut);
  PrintStream ips(*m_pIncOut);

  ps.format("//\n");
  ps.format("// rendering properties for %s\n", m_name.c_str());
  ps.format("//\n");
  ps.format("#declare _show%s = 1;\n", m_name.c_str());
  
  ips.format("\n#if (_show%s)\n", m_name.c_str());

  dumpClut(m_pIncOut);

  blin = writeLines(ps, ips);

  bcyl = writeCyls(ps, ips);

  bsph = writeSpheres(ps, ips);

  bmes = writeMeshes(ps, ips);
  
  /*
  if (bcyl || bsph || bmes) {
    ps.format("#declare %s_tex = texture {\n", m_name.c_str());
    //ps.format("  pigment {color rgbft <0,0,0,1,1>}\n");
    ps.format("  normal {granite 0.0 scale 1.0}\n");
    ps.format("  finish {\n");
    ps.format("   ambient 0.10\n");
    ps.format("   diffuse 0.70  brilliance 1.00\n");
    ps.format("   specular 0.30  roughness 0.0050\n");
    ps.format("  }\n");
    ps.format(" }\n");
    //ps.format("#declare %s = material {texture {%s}}\n",
    //mat_name.c_str(), tex_name.c_str());
  }
     */

  if (bcyl)
    ps.format("#declare %s_lw = 1.00;\n", m_name.c_str());

  ips.format("\n#end\n", m_name.c_str());
  ps.format("\n");
}


