// -*-Mode: C++;-*-
//
//  Pov-Ray Display context implementation class
//

#ifndef POV_INTERNAL_DATA_HPP_INCLUDED
#define POV_INTERNAL_DATA_HPP_INCLUDED

#include "qsys.hpp"

#include <gfx/DisplayContext.hpp>
//#include "LColor.hpp"
#include <gfx/ColorTable.hpp>
#include <qlib/LString.hpp>
#include <qlib/LStream.hpp>

namespace qlib {
  class PrintStream;
}

namespace qsys {

using qlib::LString;
using qlib::Vector4D;
using qlib::OutStream;
using gfx::ColorTable;
using gfx::ColorPtr;
using qlib::PrintStream;

class PovDisplayContext;

/// Internal data structure for PovContext
class PovIntData
{
//private:
public:
  
  typedef ColorTable::elem_t PovIntColor;

  /// output pov file
  OutStream *m_pPovOut;

  /// output inc file
  OutStream *m_pIncOut;

  /// target name
  LString m_name;

  /// Output texture type
  bool m_fUseTexBlend;

  /// Color table
  ColorTable m_clut;

  /// Clipping plane (negative value: no clipping)
  double m_dClipZ;

  /// Parent display context
  PovDisplayContext *m_pdc;

  //////////

  struct Line {
    Vector4D v1, v2;
    PovIntColor col;
    double w;
  };

  std::deque<Line *> m_lines;

  //////////

  struct Cyl {
    Vector4D v1, v2;
    PovIntColor col;
    double w1, w2;
    int ndetail;
    Matrix4D *pTransf;

    ~Cyl() {
      if (pTransf!=NULL) delete pTransf;
    }
  };

  typedef std::deque<Cyl *> CylList;

  CylList m_cylinders;

  //////////

  struct Sph {
    Vector4D v1;
    PovIntColor col;
    double r;
    int ndetail;
  };

  typedef std::deque<Sph *> SphList;

  SphList m_spheres;

  //////////

  struct MeshElem {
    Vector4D v, n;
    PovIntColor c;
  };

  struct Mesh
  {
    PovIntData *m_pPar;
    std::deque<MeshElem *> m_verts;
    std::deque<int> m_faces;

    ~Mesh()
    {
      clear();
    }

    int getVertexSize() const { return m_verts.size(); }
    int getFaceSize() const { return m_faces.size()/3; }

    void addVertex(const Vector4D &v, const Vector4D &n, const ColorPtr &c)
    {
      MeshElem *p = new MeshElem;
      p->v = v;
      p->n = n;
      m_pPar->convCol(c, p->c);

      m_verts.push_back(p);
    }

    int addVertex(MeshElem *p)
    {
      int nres = m_verts.size();
      m_verts.push_back(p);
      return nres;
    }

    void copyVertex(MeshElem *pOrig)
    {
      MeshElem *p = new MeshElem;
      p->v = pOrig->v;
      p->n = pOrig->n;
      p->c = pOrig->c;
      m_verts.push_back(p);
    }

    void addFace(int iv0, int iv1, int iv2)
    {
      m_faces.push_back(iv0);
      m_faces.push_back(iv1);
      m_faces.push_back(iv2);
    }

    void clear()
    {
      std::for_each(m_verts.begin(), m_verts.end(), qlib::delete_ptr<MeshElem *>());
      m_verts.clear();
      m_faces.clear();
    }
  };
  
  Mesh m_mesh;

  int m_nMeshMode;

  /// Mesh pivot
  int m_nMeshPivot;

public:
  /** mesh modes */
  enum {
    POV_INTD_TRIGS,
    POV_INTD_TRIGSTRIP,
    POV_INTD_INTARY
  };
  

public:
  PovIntData(PovDisplayContext *pdc);
  virtual ~PovIntData();

  void start(OutStream *fp, OutStream *ifp, const char *name);
  void end();
  
  /// Append line segment
  void line(const Vector4D &v1, const Vector4D &v2, double w);

  /// Append cylinder
  void cylinder(const Vector4D &v1, const Vector4D &v2,
                double w1, double w2,
                int ndet, const Matrix4D *ptrf);

  void sphere(const Vector4D &v1, double w, int ndet);

  void meshStart();
  void meshEndTrigs();
  void meshEndTrigStrip();
  void meshEndFan();
  
  /// Mesh generation for trigs & trigstrip
  void meshVertex(const Vector4D &v1, const Vector4D &n1, const ColorPtr &col)
  {
    m_mesh.addVertex(v1, n1, col);
  }

  // / mesh generation for trigfan
  // void mesh_fan(const Vector4D &v1, const Vector4D &n1, const LColor &c1, bool bMakeTri);

  void mesh(const Matrix4D &mat, const gfx::Mesh &mesh);

  void setMeshMode(int n);
  int getMeshMode() const { return m_nMeshMode; }

public:
  /// Convert color to internal representation
  bool convCol(PovIntColor &ic);
  
  /// Convert color to internal representation (2)
  bool convCol(const ColorPtr &col, PovIntColor &ic);

  /// dump CLUT to POV file
  void dumpClut(OutStream *fp);
  //void dumpClutTex(OutStream *fp);

  // Vector4D getClutRGBA(int ind) const;

  void writeColor(const PovIntColor &id);

  bool writeLines(PrintStream &ps, PrintStream &ips);
  bool writeMeshes(PrintStream &ps, PrintStream &ips);
  int convTexInd(MeshElem *);

  bool writeSpheres(PrintStream &ps, PrintStream &ips);
  bool writeCyls(PrintStream &ps, PrintStream &ips);

  /// Mesh generation for gfx::Mesh
  //int meshVertex2(const Vector4D &v1, const Vector4D &n1, const ColorPtr &c1, PovDisplayContext *pdc);

  //int cutEdge(const Matrix4D &mat, const gfx::Mesh &rmesh, PovDisplayContext *pdc, int id1, int id2);

  //int addMeshFace(int iv0, int iv1, int iv2);

  void writeTextureList();

  void writeGradTexture();

  PovIntData::Mesh *calcMeshClip();
  PovIntData::MeshElem *cutEdge(MeshElem *pv1, MeshElem *pv2);

};

}

#endif

