// -*-Mode: C++;-*-
//
// Undo/Redo information for properties
//
// $Id: PropEditInfo.hpp,v 1.8 2010/09/26 15:17:44 rishitani Exp $

#ifndef QSYS_PROP_EDIT_INFO_HPP_INCLUDED
#define QSYS_PROP_EDIT_INFO_HPP_INCLUDED

#include "qsys.hpp"

#include "EditInfo.hpp"
#include "UndoManager.hpp"
#include "Scene.hpp"
#include <qlib/LVariant.hpp>
#include <qlib/ObjectManager.hpp>
#include <qlib/NestedPropHandler.hpp>

namespace qsys {


class PropEditInfoBase : public EditInfo
{
private:
  qlib::uid_t m_nTgtUID;
  qlib::LString m_propname;

public:
  PropEditInfoBase() : m_nTgtUID(qlib::invalid_uid)
  {
  }

  virtual ~PropEditInfoBase() {}

  void setTargetUID(qlib::uid_t uid) { m_nTgtUID = uid; }
  qlib::uid_t getTargetUID() const { return m_nTgtUID; }

  void setPropName(const LString &name) { m_propname = name; }
  const LString &getPropName() const { return m_propname; }

  template <class _RootObjType>
  bool setup(qlib::LScrObjBase *pThis) {
    qlib::uid_t rootuid = pThis->getRootUID();
    if (rootuid==qlib::invalid_uid)
      return false;
    
    _RootObjType *pTgtRoot =
      qlib::ObjectManager::sGetObj<_RootObjType>(rootuid);
    if (pTgtRoot==NULL) return false;
    
    qsys::ScenePtr scene = pTgtRoot->getScene();
    if (scene.isnull()) return false;
    
    qsys::UndoManager *pUM = scene->getUndoMgr();
    if (!pUM->isOK())
      return false;
    
    setTargetUID(rootuid);
    setPropName(pThis->getThisName());
    pUM->addEditInfo(this);
    
    return true;
  }

};

class PropEditInfo : public PropEditInfoBase
{
private:
  qlib::LVariant m_newvalue;
  qlib::LVariant m_oldvalue;

  bool m_bNewDef;
  bool m_bOldDef;

public:
  PropEditInfo() : PropEditInfoBase(), m_bNewDef(false), m_bOldDef(false)
  {
  }

  virtual ~PropEditInfo() {
    m_newvalue.cleanup();
    m_oldvalue.cleanup();
  }

  /// Setup edit info from old and new values of the property
  void setup(qlib::uid_t uid, const LString &propnm,
             const qlib::LVariant &ov, const qlib::LVariant &nv) {
    setTargetUID(uid);
    setPropName(propnm);
    m_oldvalue = ov;
    m_newvalue = nv;
    m_bNewDef = m_bOldDef = false;
  }

  /// Setup edit info from the property event
  void setup(qlib::uid_t uid, const LString &propnm,
             const qlib::LPropEvent &ev) {
    setTargetUID(uid);
    setPropName(propnm);
    m_oldvalue = ev.getOldValue();
    m_newvalue = ev.getNewValue();
    m_bOldDef = ev.isOldDefault();
    m_bNewDef = ev.isNewDefault();
  }

  /// Perform undo
  virtual bool undo() {
    qlib::LPropSupport *pTgt =
      qlib::ObjectManager::sGetObj<qlib::LPropSupport>(getTargetUID());
    if (pTgt==NULL) return false;

    qlib::NestedPropHandler nph(getPropName(), pTgt);
    if (m_bOldDef)
      return nph.apply()->resetProperty(nph.last_name());
    else
      return nph.apply()->setProperty(nph.last_name(), m_oldvalue);
  }

  /// Perform redo
  virtual bool redo() {
    qlib::LPropSupport *pTgt =
      qlib::ObjectManager::sGetObj<qlib::LPropSupport>(getTargetUID());
    if (pTgt==NULL) return false;

    qlib::NestedPropHandler nph(getPropName(), pTgt);
    if (m_bNewDef)
      return nph.apply()->resetProperty(nph.last_name());
    else
      return nph.apply()->setProperty(nph.last_name(), m_newvalue);
  }

  virtual bool isUndoable() const {
    // XXX
    // TO DO: check actually undoable
    return true;
  }
  virtual bool isRedoable() const {
    // XXX
    // TO DO: check actually redoable
    return true;
  }

};

}

#endif
