// -*-Mode: C++;-*-
//
// Renderer classes draw Object's contents to the View
//

#ifndef QSYS_RENDERER_HPP_INCLUDE_
#define QSYS_RENDERER_HPP_INCLUDE_

#include "qsys.hpp"

#include <qlib/ObjectManager.hpp>
#include <qlib/LScrSmartPtr.hpp>
#include <qlib/LScrVector4D.hpp>
#include <qlib/LPropEvent.hpp>
#include "style/StyleSupports.hpp"
#include "RendererEvent.hpp"
#include "ObjectEvent.hpp"
#include "SceneEvent.hpp"

using qlib::LString;

namespace gfx {
  class DisplayContext;
  class Hittest;
  class RawHitData;
}

namespace qsys {

  using gfx::DisplayContext;
  class StyleSheet;

  class QSYS_API Renderer :
    public StyleScrObject,
    public StyleSupports,
    public qlib::LUIDObject,
    public qlib::LPropEventListener,
    public ObjectEventListener,
    public SceneEventListener
  {
    MC_SCRIPTABLE;

  public:

    //typedef qlib::LSimpleCopyScrObject super_t;
    typedef StyleScrObject super_t;

    RendererEventCaster *m_pEvtCaster;

  private:

    /** ID of the scene to which this renderer belongs. */
    qlib::uid_t m_nSceneID;

    qlib::uid_t m_uid;
    
    qlib::uid_t m_nClientObj;

    /// Name of this renderer
    LString m_name;

    /// Visibility
    bool m_bVisible;

    /// Locked/unlocked (for user editing)
    bool m_bLocked;

    StyleSheet *m_pStyles;

  public:
    Renderer();
    Renderer(const Renderer &r);
    virtual ~Renderer();
  
    //////////

  private:
    /** = operator (for avoiding copy operation) */
    const Renderer &operator=(const Renderer &arg)
    {
      return *this;
    }

    //////////

  public:

    /// Display renderers in the scene to the frame buffer.
    virtual void display(DisplayContext *pdc) =0;

    virtual const char *getTypeName() const =0;

    virtual bool isCompatibleObj(ObjectPtr pobj) const =0;

    virtual LString toString() const;

    /// Called just before this object is unloaded
    virtual void unloading();

    virtual qlib::Vector4D getCenter() const =0;

    /// Returns true if rendering contains transparent objects.
    /// (In OGL impl, transparent objects should be rendered without depth testing.)
    virtual bool isTransp() const;

    //////////
    // Hittest

    void processHit(DisplayContext *pdc);

    virtual void displayHit(DisplayContext *pdc);

    /// Hittest support check (default is not support hittest)
    virtual bool isHitTestSupported() const;

    /// Hittest result interpretation
    virtual LString interpHit(const gfx::RawHitData &);
    
    //////////

    virtual void setSceneID(qlib::uid_t nid);

    qlib::uid_t getSceneID() const { return m_nSceneID; }
    ScenePtr getScene() const;

    qlib::uid_t getUID() const { return m_uid; }

    void setName(const LString &n) { m_name = n; }
    const LString &getName() const { return m_name; }

    void setVisible(bool b) { m_bVisible = b; }
    bool isVisible() const { return m_bVisible; }

    void setUILocked(bool b) { m_bLocked = b; }
    bool isUILocked() const { return m_bLocked; }

    qlib::LScrVector4D getCenterScr() const {
      return qlib::LScrVector4D(getCenter());
    }

    //////////////////////////////////////////////////
    // Client object management

    virtual void attachObj(qlib::uid_t obj_uid);

    virtual qlib::uid_t detachObj();

    qlib::uid_t getClientObjID() const {
      return m_nClientObj;
    }

    ObjectPtr getClientObj() const;


    //////////////////////////////////////////////////
    // Style supports

    /// Get stylesheet of this renderer
    virtual StyleSheet *getStyleSheet();

    /// Apply style sheet
    /// (name_list should be comma-separated list of style names)
    void applyStyles(const LString &name_list);

    /// Get current stylesheet names (comma-sep list)
    LString getStyleNames() const;

    /// Add style name
    /// If the style exists, the order is changed to the first pos.
    bool pushStyle(const LString &name);

    /// Remove styles that match the regex
    bool removeStyleRegex(const LString &regex);

    /// Style event listener
    virtual void styleChanged(StyleEvent &);

    /// Style context ID (==scene ID)
    virtual qlib::uid_t getStyleCtxtID() const;

  private:
    void fireStyleEvents();

    void setupStyleUndo(const LString &, const LString &);

    //////////////////////////////////////////////////
    // Default material&transparency

  private:
    LString m_defMatName;
    double m_defAlpha;

  public:
    LString getDefaultMaterial() const {
      return m_defMatName;
    }
    void setDefaultMaterial(const LString &str) {
      m_defMatName = str;
    }

    double getDefaultAlpha() const {
      return m_defAlpha;
    }
    void setDefaultAlpha(double f) {
      m_defAlpha = f;
    }

    ////////////////////////////////////////////////////////////
    // Event related operations

    int addListener(RendererEventListener *pL);
    bool removeListener(RendererEventListener *pL);
    void fireRendererEvent(RendererEvent &ev);

    /// object changed event (do nothing)
    virtual void objectChanged(ObjectEvent &ev);

    /// scene changed event (for onloaded event)
    virtual void sceneChanged(SceneEvent &ev);

    // for property event propagation
    virtual qlib::uid_t getRootUID() const;
    
    virtual void propChanged(qlib::LPropEvent &ev);

    void writeTo2(qlib::LDom2Node *pNode) const;
    void readFrom2(qlib::LDom2Node *pNode);

  };

} // namespace qsys

#endif // QSYS_RENDERER_HPP_INCLUDE_

