// -*-Mode: C++;-*-
//
// Scene: Root object of scene document
//
// $Id: Scene.cpp,v 1.101 2011/04/16 17:30:56 rishitani Exp $
//

#include <common.h>

#include "Scene.hpp"
#include "SceneEventCaster.hpp"
#include "ObjLoadEditInfo.hpp"
#include "ScrEventManager.hpp"
#include "PropEditInfo.hpp"
#include "SysConfig.hpp"
#include "StreamManager.hpp"

#include <qlib/FileStream.hpp>
#include <qlib/StringStream.hpp>
#include <qlib/LVarArgs.hpp>

// for camera loading
#include <qlib/LDOM2Stream.hpp>
#include <qlib/FileStream.hpp>

#include <gfx/DisplayContext.hpp>
#include "style/StyleMgr.hpp"
#include "style/StyleFile.hpp"

// for JS interpreter
#include <jsbr/jsbr.hpp>
#include <jsbr/Interp.hpp>

#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/path.hpp>

using namespace qsys;
namespace fs = boost::filesystem;

namespace {
class ScrScEvtLsnr : public SceneEventListener
{
private:
  //qlib::LScrCallBack *m_pCb;
  qlib::LSCBPtr m_pCb;
  
public:
  ScrScEvtLsnr(qlib::LSCBPtr pcb) : m_pCb(pcb) {}
  
  virtual ~ScrScEvtLsnr()
  {
    MB_DPRINTLN("ScrScEvtLsnr destr %p", this);
    m_pCb = qlib::LSCBPtr();
  }
  
  void set(qlib::LSCBPtr pcb) { m_pCb = pcb; }

  virtual void sceneChanged(SceneEvent &ev)
  {
    // avoid frequent event from script calling (TO DO: configurable)
    int ntype = ev.getType();
    //if (ntype==SceneEvent::SCE_VIEW_PROPCHG_DRG) return;
    //if (ntype==SceneEvent::SCE_VIEW_SIZECHG) return;
    //if (ntype==SceneEvent::SCE_VIEW_ACTIVATED) return;

    qlib::uid_t scid = ev.getSource();
    qlib::uid_t objid = ev.getTarget();
    qlib::LVarArgs args(5);
    
    // Method name
    args.at(0).setStringValue("sceneChanged");
    // type ID of the event
    args.at(1).setIntValue(ntype);
    // target scene ID
    args.at(2).setIntValue(scid);
    // target object/renderer/view ID
    args.at(3).setIntValue(objid);
    // target message/property name
    args.at(4).setStringValue(ev.getDescr());
    // // event object (in JSON format)
    // args.at(5).setStringValue(ev.getJSON());


    m_pCb->invoke(args);
  }
      
};
}

Scene::Scene()
{
  m_nUID = qlib::ObjectManager::sRegObj(this);
  m_pEvtCaster = new SceneEventCaster;
  m_pStyleMgr = StyleMgr::getInstance();
  m_pInterp = NULL;
  // m_nModified = 0;
  addPropListener(this);

  MB_DPRINTLN("Scene (%d) created.", m_nUID);
}

void Scene::init()
{
  resetAllProps();

  if (m_pInterp!=NULL) delete m_pInterp;
  ScenePtr rThis(this);

  m_pInterp = jsbr::createInterp(rThis.copy());
  if (m_pInterp==NULL)
    return;

  // setup system default path
  SysConfig *pconf = SysConfig::getInstance();
  LString scrdir = pconf->get("script_dir");
  if (!scrdir.isEmpty())
    m_pInterp->setScriptPath("system", pconf->convPathName(scrdir));
}

Scene::~Scene()
{
  removePropListener(this);
  clearAll();

  delete m_pEvtCaster;
  qlib::ObjectManager::sUnregObj(m_nUID);

  MB_DPRINTLN("Scene (%d/%p) destructed", m_nUID, this);
}

void Scene::dump() const
{
  MB_DPRINTLN("SCENE DUMP:");
  MB_DPRINTLN("Scene : {");
  MB_DPRINTLN("  name = <%s>", m_name.c_str());
  MB_DPRINTLN("  uid = <%d>", m_nUID);

  data_t::const_iterator iter = m_data.begin();
  for (; iter!=m_data.end(); ++iter) {
    MB_DPRINT("%p/%d (nref=%d): ",
	      iter->second.get(),
	      iter->second->getUID(),
	      iter->second.use_count());
    iter->second->dump();
  }

  MB_DPRINTLN("}");
  MB_DPRINTLN("SCENE DUMP END.");

}

void Scene::clearAll()
{
  clearAllData();
  m_viewtab.clear();
}

void Scene::clearAllData()
{
  m_undomgr.clearAllInfo();
  m_rendtab.clear();
  m_data.clear();
  m_camtab.clear();
  m_pStyleMgr->destroyContext(m_nUID);
}

void Scene::clearAllDataScr()
{
  clearAllData();

  {
    SceneEvent ev;
    ev.setTarget(getUID());
    ev.setType(SceneEvent::SCE_SCENE_CLEARALL);
    fireSceneEvent(ev);
  }
}

void Scene::unloading()
{
  {
    SceneEvent ev;
    ev.setTarget(getUID());
    ev.setType(SceneEvent::SCE_SCENE_REMOVING);
    fireSceneEvent(ev);
  }

  MB_DPRINTLN("Scene.unloading> *** Broadcast unloading ***");
  rendtab_t::const_iterator riter = m_rendtab.begin();
  for (; riter!=m_rendtab.end(); ++riter) {
    RendererPtr rrend = riter->second;
    MB_DPRINTLN("Scene.unloading()> rrend %d unloading().", riter->first);
    rrend->unloading();
  }
  
  data_t::const_iterator oiter = m_data.begin();
  for (; oiter!=m_data.end(); ++oiter) {
    ObjectPtr obj = oiter->second;
    MB_DPRINTLN("Scene.unloading()> robj %d unloading().", oiter->first);
    obj->unloading();
  }

  viewtab_t::const_iterator viter = m_viewtab.begin();
  for (; viter!=m_viewtab.end(); ++viter) {
    ViewPtr rvi = viter->second;
    MB_DPRINTLN("Scene.unloading()> rview %d unloading().", viter->first);
    rvi->unloading();
  }

  delete m_pInterp;
  m_pInterp = NULL;
}

/// set source path of this scene
void Scene::setSource(const LString &name)
{
  m_source = name;
  if (m_pInterp!=NULL)
    m_pInterp->setScriptPath("scene_base_path", getBasePath());
}

LString Scene::getBasePath() const
{
  fs::path srcpath(m_source);
  return srcpath.parent_path().directory_string();
}

LString Scene::resolveBasePath(const LString &aLocalFile) const
{
/*
  fs::path inpath(aLocalFile);
  if (inpath.is_complete())
    return inpath.file_string();

  fs::path base_path(getBasePath());

  // convert to the absolute representation
  inpath = fs::complete(inpath, base_path);
  return inpath.file_string();
*/

  return qlib::makeAbsolutePath(aLocalFile, getBasePath());
}

//////////////////////////////

/*ObjectPtr Scene::createObject(const LString &type_name)
{
  ObjectPtr robj(new Object());

  if (!registerObjectImpl(robj)) {
    MB_THROW(qlib::RuntimeException, "Cannot create new object");
    return ObjectPtr();
  }

  return robj;
}*/

bool Scene::addObject(ObjectPtr pobj)
{
  qlib::uid_t uid = pobj->getUID();
  data_t::const_iterator i = m_data.find(uid);
  if (i!=m_data.end()) {
    MB_DPRINTLN("Scene::addObject> object %d is already registered!!", int(uid));
    return false;
  }

  if (!registerObjectImpl(pobj)) {
    MB_THROW(qlib::RuntimeException, "Cannot add new object");
    return false;
  }

  return true;
}

bool Scene::destroyObject(qlib::uid_t uid)
{
  data_t::iterator i = m_data.find(uid);
  if (i==m_data.end()) {
    return false;
  }
  
  ObjectPtr robj = i->second;
  qlib::ensureNotNull(robj);

  // Detach this scene from the prop event source
  bool res = robj->removeListener(this);
  MB_ASSERT(res);

  //
  // purge the rendering cache of this scene
  //
  while (robj->getRendCount()>0) {
    Object::RendIter ri = robj->beginRend();
    robj->destroyRenderer(ri->first);
    //RendererPtr rrend = ri->second;
    //robj->destroyRenderer(rrend->getUID());
    //removeRendCache(rrend);
  }

  //
  // Fire the SCE_OBJ_REMOVING Event, before removing
  //
  {
    MB_DPRINTLN("Scene> Firing SCE_OBJ_REMOVING event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_OBJ_REMOVING);
    ev.setSource(getUID());
    ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }

  // Here, actually remove the item.
  m_data.erase(i);

  {
    // Record undo/redo info
    UndoManager *pUM = getUndoMgr();
    if (pUM->isOK()) {
      ObjLoadEditInfo *pPEI = new ObjLoadEditInfo;
      pPEI->setupObjDestroy(getUID(), robj);
      pUM->addEditInfo(pPEI);
    }
  }

  return true;
}

void Scene::destroyAllObjects()
{
  while (m_data.size()>0) {
    data_t::iterator oiter = m_data.begin();
    qlib::uid_t uid = oiter->first;
    destroyObject(uid);
  }
}


ObjectPtr Scene::getObject(qlib::uid_t uid) const
{
  data_t::const_iterator i = m_data.find(uid);
  if (i==m_data.end()) {
    // LString msg = LString::format("Object %d not found", uid);
    // MB_THROW(qlib::RuntimeException, msg);
    return ObjectPtr();
  }

  return i->second;
}

ObjectPtr Scene::getObjectByName(const LString &name) const
{
  BOOST_FOREACH(data_t::value_type i, m_data) {
    // if (pObj.isnull) continue;
    if (i.second->getName().equals(name))
      return i.second;
  }
  return ObjectPtr();
}

int Scene::getAllObjectUIDs(std::list<qlib::uid_t> &uids) const
{
  int nret = 0;
  data_t::const_iterator i = m_data.begin();
  for (; i!=m_data.end(); ++i) {
    MB_DPRINTLN("Scene::getAllObjectUIDs %d", i->first);
    uids.push_back(i->first);
    ++nret;
  }

  return nret;
}

qlib::LVarArray Scene::getObjUIDArray() const
{
  qlib::LVarArray rval;
  if (m_data.empty()) {
    return rval;
  }
  
  std::list<qlib::uid_t> ls;
  getAllObjectUIDs(ls);

  rval.allocate(ls.size());
  std::list<qlib::uid_t>::const_iterator iter = ls.begin();
  std::list<qlib::uid_t>::const_iterator eiter = ls.end();

  for (int i=0; iter!=eiter; ++iter, ++i)
    rval.setIntValue(i, *iter);

  return rval;
}


qlib::LVarArray Scene::getObjArray() const
{
  qlib::LVarArray rval;
  if (m_data.empty())
    return rval;
  
  rval.allocate(m_data.size());

  data_t::const_iterator i = m_data.begin();
  data_t::const_iterator end = m_data.end();
  for (int j=0; i!=end; ++i, ++j) {
    ObjectPtr pObj = i->second;
    rval.setObjectPtr(j, pObj.copy());
  }

  return rval;
}

/*
ObjectPtr Scene::loadObject(const LString &localfile, const LString &type)
{
  MB_DPRINTLN("Load object: %s (%s)", localfile.c_str(), type.c_str());
  StreamManager *pSM = StreamManager::getInstance();

  ObjectPtr pObj = pSM->loadObject(localfile, type);

  if (pObj.isnull()) {
    LOG_DPRINTLN("Load object: %s (%s) failed", localfile.c_str(), type.c_str());
    return ObjectPtr();
  }

  if (!registerObjectImpl(pObj)) {
    MB_THROW(qlib::RuntimeException, "Cannot create new object");
    return ObjectPtr();
  }

  return pObj;
}
*/

int Scene::loadObjectAsync(const LString &type)
{
#ifdef HAVE_BOOST_THREAD
  MB_DPRINTLN("Load object async: (%s)", type.c_str());
  StreamManager *pSM = StreamManager::getInstance();

  // alloc new thread
  int tid = pSM->loadObjectAsync(type);
  return tid;
#else
  return -1;
#endif
}

void Scene::supplyDataAsync(int id, const char *pbuf, int nlen)
{
#ifdef HAVE_BOOST_THREAD
  StreamManager *pSM = StreamManager::getInstance();
  pSM->supplyDataAsync(id, pbuf, nlen);
#else
#endif
}

ObjectPtr Scene::waitLoadAsync(int id)
{
#ifdef HAVE_BOOST_THREAD
  StreamManager *pSM = StreamManager::getInstance();
  ObjectPtr pObj = pSM->waitLoadAsync(id);

  if (pObj.isnull() || !registerObjectImpl(pObj)) {
    // MB_THROW(qlib::RuntimeException, "Cannot create new object");
    return ObjectPtr();
  }

  return pObj;
#else
  return ObjectPtr();
#endif
}

// private
/** Common implementation for the object registration */
bool Scene::registerObjectImpl(ObjectPtr robj)
{
  robj->setSceneID(m_nUID);
  bool res = m_data.insert(data_t::value_type(robj->getUID(), robj)).second;
  if (!res)
    return false;

  // observe object events from the new object
  robj->addListener(this);

  {
    // Fire the SCE_OBJ_ADDED Event
    MB_DPRINTLN("Scenek> Firing SCE_OBJ_ADDED event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_OBJ_ADDED);
    ev.setSource(getUID());
    ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }
  
  {
    // Append renderers to the cache
    Object::RendIter ri = robj->beginRend();
    for (; ri!=robj->endRend(); ++ri) {
      RendererPtr rrnd = ri->second;
      addRendCache(rrnd);
      // for consistency
      rrnd->setSceneID(m_nUID);
      // The scene observes events from the new renderer
      //rrnd->addPropListener(this);
      rrnd->addListener(this);
    }
  }

  {
    // Record undo/redo info
    UndoManager *pUM = getUndoMgr();
    if (!pUM->isOK())
      return true; // undo manager is disabled now.
    
    ObjLoadEditInfo *pPEI = new ObjLoadEditInfo;
    pPEI->setupObjCreate(getUID(), robj);
    pUM->addEditInfo(pPEI);
  }

  return true;
}

///////////////////////////////////////
// renderer cachelist management

bool Scene::addRendCache(RendererPtr rrend)
{
  bool res = m_rendtab.insert(rendtab_t::value_type(rrend->getUID(), rrend)).second;
  if (res) {
    // scene redrawing is required
    setUpdateFlag();
  }
  return res;
}

bool Scene::removeRendCache(RendererPtr rrend)
{
  qlib::uid_t uid = rrend->getUID();
  rendtab_t::iterator i = m_rendtab.find(uid);
  if (i==m_rendtab.end())
    return false;

  m_rendtab.erase(i);

  // scene redrawing is required
  setUpdateFlag();

  return true;
}

RendererPtr Scene::getRendByName(const LString &nm) const
{
  rendtab_t::const_iterator riter = m_rendtab.begin();
  rendtab_t::const_iterator reiter = m_rendtab.end();
  for (; riter!=reiter; ++riter) {
    RendererPtr rrend = riter->second;
    if (nm.equals(rrend->getName()))
      return rrend;
  }

  return RendererPtr();
}

///////////////////////////////////////
// Rendering of the scene

static
LString makeSectionName(ObjectPtr robj, RendererPtr rrend)
{
/*
  LString name = LString::format("_%s_%d_%s_%d",
                                 robj->getName().c_str(), robj->getUID(),
                                 rrend->getName().c_str(), rrend->getUID());
  name.replace('.', '_');
  name.replace('-', '_');
  name.replace('(', '_');
  name.replace(')', '_');
*/

  LString name = LString::format("_%d_%d", robj->getUID(), rrend->getUID());

  return name;
}

///
/// Display renderers in the scene to the frame buffer
///
void Scene::display(DisplayContext *pdc)
{
  // StyleMgr is held in the member variable (at the ctor)
  m_pStyleMgr->pushContextID(getUID());
  pdc->startRender();

  std::vector<RendererPtr> transps;
  {
    rendtab_t::const_iterator i = m_rendtab.begin();
    rendtab_t::const_iterator ie = m_rendtab.end();
    for (; i!=ie; ++i) {
      RendererPtr rrend = i->second;
      ObjectPtr robj = rrend->getClientObj();
      if (robj.isnull() || robj->isVisible()) {
        if (rrend->isVisible()) {
          if (rrend->isTransp()) {
            transps.push_back(rrend);
          }
          else {
            pdc->startSection(makeSectionName(robj, rrend));
            pdc->setMaterial(rrend->getDefaultMaterial());
            pdc->setAlpha(rrend->getDefaultAlpha());
            rrend->display(pdc);
            pdc->endSection();
          }
        }
      }
    }
  }

  // Render transparent objects
  if (transps.size()>0) {
    // pdc->enableDepthTest(false);

    std::vector<RendererPtr>::const_iterator i = transps.begin();
    std::vector<RendererPtr>::const_iterator ie = transps.end();
    for (; i!=ie; ++i) {
      RendererPtr rrend = *i;
      ObjectPtr robj = rrend->getClientObj();
      pdc->startSection(makeSectionName(robj, rrend));
      pdc->setMaterial(rrend->getDefaultMaterial());
      pdc->setAlpha(rrend->getDefaultAlpha());
      rrend->display(pdc);
      pdc->endSection();
    }
    // pdc->enableDepthTest(true);
  }

  pdc->endRender();
  m_pStyleMgr->popContextID();
}

void Scene::processHit(DisplayContext *pdc)
{
  m_pStyleMgr->pushContextID(getUID());
  pdc->startRender();

  rendtab_t::const_iterator i = m_rendtab.begin();
  for (; i!=m_rendtab.end(); ++i) {
    RendererPtr rrend = i->second;
    ObjectPtr robj = rrend->getClientObj();
    if (robj.isnull() || (robj->isVisible() && !robj->isUILocked())) {
      if (rrend->isVisible() && !rrend->isUILocked()) {
        rrend->processHit(pdc);
      }
    }
  }

  pdc->endRender();
  m_pStyleMgr->popContextID();
}

//////////////////////////////

ViewPtr Scene::createView()
{
  ViewPtr robj(View::createView());
  if (robj.isnull()) {
    LOG_DPRINTLN("Fatal error: View::createView() returned NULL!!");
    MB_THROW(qlib::NullPointerException, "");
    return ViewPtr();
  }

  robj->setSceneID(m_nUID);
  bool res = m_viewtab.insert(viewtab_t::value_type(robj->getUID(), robj)).second;
  if (!res)
    return ViewPtr();

  {
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_VIEW_ADDED);
    ev.setSource(getUID());
    ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }

  return robj;
}

ViewPtr Scene::getView(qlib::uid_t uid) const
{
  viewtab_t::const_iterator i = m_viewtab.find(uid);
  if (i==m_viewtab.end())
    return ViewPtr();

  return i->second;
}

ViewPtr Scene::getViewByName(const LString &name) const
{
  BOOST_FOREACH(viewtab_t::value_type i, m_viewtab) {
    if (i.second->getName().equals(name))
      return i.second;
  }

  return ViewPtr();
}

bool Scene::destroyView(qlib::uid_t uid)
{
  viewtab_t::iterator i = m_viewtab.find(uid);
  if (i==m_viewtab.end())
    return false;

  ViewPtr robj = i->second;
  qlib::ensureNotNull(robj);
  
  {
    MB_DPRINTLN("Scenek> Firing SCE_VIEW_REMOVING event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_VIEW_REMOVING);
    ev.setSource(getUID());
    ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }

  if (m_viewtab.size()==1) {
    // "uid" is the last view. --> prepare for destucting everything
    rendtab_t::const_iterator riter = m_rendtab.begin();
    for (; riter!=m_rendtab.end(); ++riter) {
      RendererPtr rrend = riter->second;
      rrend->unloading();
    }

    data_t::const_iterator oiter = m_data.begin();
    for (; oiter!=m_data.end(); ++oiter) {
      ObjectPtr obj = oiter->second;
      obj->unloading();
    }
  }

  robj->unloading();

  m_viewtab.erase(i);

  return true;
}

void Scene::checkAndUpdate()
{
  if (m_bUpdateRequired) {
    // Force to redraw all views
    viewtab_t::const_iterator viter = m_viewtab.begin();
    for (; viter!=m_viewtab.end(); ++viter) {
      View *pV = viter->second.get();
      if (pV->isActive())
        pV->forceRedraw();
    }
    clearUpdateFlag();
  }
  else {
    // Redraw views if required
    viewtab_t::const_iterator viter = m_viewtab.begin();
    for (; viter!=m_viewtab.end(); ++viter) {
      View *pV = viter->second.get();
      if (pV->getUpdateFlag() && pV->isActive()) {
        MB_DPRINTLN("Scene::checkAndUpdate() drawScene");
        pV->drawScene();
        pV->clearUpdateFlag();
      }
    }
  }
}

qlib::LVarArray Scene::getViewUIDArray() const
{
  qlib::LVarArray rval;
  if (m_viewtab.empty())
    return rval;
  
  rval.allocate(m_viewtab.size());

  viewtab_t::const_iterator i = m_viewtab.begin();
  viewtab_t::const_iterator end = m_viewtab.end();
  for (int j=0; i!=end; ++i, ++j)
    rval.setIntValue(j, i->first);

  return rval;
}

qlib::LVarArray Scene::getViewArray() const
{
  qlib::LVarArray rval;
  if (m_viewtab.empty())
    return rval;
  
  rval.allocate(m_viewtab.size());

  viewtab_t::const_iterator i = m_viewtab.begin();
  viewtab_t::const_iterator end = m_viewtab.end();
  for (int j=0; i!=end; ++i, ++j) {
    ViewPtr pObj = i->second;
    rval.setObjectPtr(j, pObj.copy());
  }

  return rval;
}


////////////////////////////////////////////////////////////

qlib::uid_t Scene::getRootUID() const
{
  return getUID();
}

bool Scene::isModified() const
{
  if (m_undomgr.getUndoSize()==0)
    return false;
  else
    return true;
}

bool Scene::isJustCreated() const
{
  if (isModified())
    return false;

  // scene is not modified
  
  if (getObjectCount()==0 &&
      getCameraCount()==0) {
      //getCameraCount()==1 &&
      //!getCamera("__current").isnull()) {
    // not modified but just created
    return true;
  }

  return false;
}

void Scene::setName(const LString &name)
{
  // special case; scene's name is readonly property
  // but changing by setName fires propChanged event.
  // (this is required to change UI labels of scene name...)

  // LString oldname = m_name;
  m_name = name;
  
  {
    qlib::LPropEvent ev("name");
    propChanged(ev);
  }
}

int Scene::addListener(SceneEventListener *pL)
{
  return m_pEvtCaster->add(pL);
}

int Scene::addListener(qlib::LSCBPtr scb)
{
  ScrScEvtLsnr *pLn = new ScrScEvtLsnr(scb);
  // pLn->m_pCb = scb; // .get();
  return m_pEvtCaster->add(pLn);
}

bool Scene::removeListener(SceneEventListener *pL)
{
  return m_pEvtCaster->remove(pL);
}

bool Scene::removeListener(int nid)
{
  ScrScEvtLsnr *pSLn = dynamic_cast<ScrScEvtLsnr *>(m_pEvtCaster->remove(nid));
  MB_DPRINTLN("Scene::removeListener(%d) remove %p", nid, pSLn);
  if (pSLn==NULL)
    return false;
  delete pSLn;
  return true;
}

void Scene::fireSceneEvent(SceneEvent &ev)
{
  m_pEvtCaster->replicaFire(ev);

  ScrEventManager *pSEM = ScrEventManager::getInstance();
  ev.setSource(m_nUID);
  pSEM->fireSceneEvent(ev);

  // call the script event handlers
  if (ev.getType()==SceneEvent::SCE_SCENE_ONLOADED) {
    if (m_pInterp!=NULL && !m_scrOnLoadEvent.isEmpty())
      m_pInterp->eval(m_scrOnLoadEvent);
  }
}

void Scene::propChanged(qlib::LPropEvent &ev)
{
  MB_DPRINTLN("!!! Scene::propChanged");
  /*
  qlib::LPropSupport *pTgt = ev.getTarget();
  Renderer *pRendTgt = dynamic_cast<Renderer *>(pTgt);
  if (pRendTgt!=NULL) {
    MB_ASSERT(false);
  }
   */

  // record undo/redo info
  if (!ev.isIntrDataChanged()) {
    UndoManager *pUM = getUndoMgr();
    if (pUM->isOK()) {
      // record property changed undo/redo info
      PropEditInfo *pPEI = new PropEditInfo;
      LString nm = ev.getName();
      
      if (!ev.getParentName().isEmpty())
        nm = ev.getParentName() + "." + nm;
      
      pPEI->setup(getUID(), nm, ev);
      pUM->addEditInfo(pPEI);
    }
  }

  // propagate prop event to scene event
  {
    SceneEvent sev;
    sev.setTarget(getUID());
    sev.setSource(getUID());
    sev.setType(SceneEvent::SCE_SCENE_PROPCHG);
    sev.setDescr(ev.getName());
    sev.setPropEvent(&ev);
    fireSceneEvent(sev);
  }
}

void Scene::objectChanged(ObjectEvent &ev)
{
  if (ev.getType()==ObjectEvent::OBE_CHANGED) {
    // objchg may require redrawing
    setUpdateFlag();
  }
  else if (ev.getType()==ObjectEvent::OBE_PROPCHG) {
    // propchg may require redrawing
    setUpdateFlag();
  }
}

void Scene::rendererChanged(RendererEvent &ev)
{
  if (ev.getType()==RendererEvent::RNE_CHANGED) {
    // objchg may require redrawing
    setUpdateFlag();
  }
  else if (ev.getType()==RendererEvent::RNE_PROPCHG) {
    // propchg may require redrawing
    setUpdateFlag();
  }
}

/////////////////////////////////////////////////

LString Scene::getObjectTreeJSON() const
{
  LString rval = "[\n";

  // this scene
  rval += "{";
  rval += "\"name\":\""+m_name+"\", ";
  rval += "\"parent\": -1, ";
  rval += "\"open\": \"true\", ";
  rval += "\"type\":\"\", ";
  rval += "\"rends\": [],\n";
  rval += LString::format("\"ID\": %d", getUID());
  rval += "}";

  data_t::const_iterator iter = m_data.begin();

  // int nrow = 1, nparent = -1;

  for (; iter!=m_data.end(); ++iter) {
    // if (iter!=m_data.begin())
    rval += ",\n";
    rval += "{";
    ObjectPtr obj = iter->second;
    qlib::uid_t objid = iter->first;
    
    rval += "\"name\":\""+(obj->getName())+"\", ";
    rval += "\"parent\": -1, ";
    rval += "\"open\": true, ";
    rval += LString("\"visible\": ") + LString::fromBool(obj->isVisible()) + ", ";
    rval += "\"type\":\""+LString(obj->getClassName())+"\", ";
    rval += LString::format("\"ID\": %d, ", objid);

    // nparent = nrow;
    // nrow ++;

    rval += "\"rends\": [\n";

    rendtab_t::const_iterator riter = m_rendtab.begin();
    for (bool bfirst=true; riter!=m_rendtab.end(); ++riter) {
      RendererPtr rrend = riter->second;
      if (rrend->getClientObjID()!=objid)
        continue;
      
      if (!bfirst)
        rval += ",\n";

      rval += "{";
      rval += "\"name\":\""+(rrend->getName())+"\", ";
      // rval += LString::format("\"parent\": %d,\n", objid);
      rval += "\"type\":\""+LString(rrend->getTypeName())+"\", ";
      rval += LString("\"visible\": ") + LString::fromBool(rrend->isVisible()) + ", ";
      rval += LString::format("\"ID\": %d", riter->first);
      rval += "}";
      // nrow ++;
      bfirst = false;
    }

    rval += "]\n";
    rval += "}";
  }

  rval += "\n]";

  // MB_DPRINTLN("Scene> built JSON=%s", rval.c_str());
  return rval;
}

////////////////////////////////////////////////////////////
// Camera manager

// private
bool Scene::registerCameraImpl(const LString &name, CameraPtr r)
{
  r->m_name = name;
  return m_camtab.insert(camtab_t::value_type(name, r)).second;
}

// private
CameraPtr Scene::loadCameraImpl(const LString &aLocalFile)
{
  LString path = resolveBasePath(aLocalFile);

  qlib::FileInStream fis;
  fis.open(path);
  qlib::LDom2InStream ois(fis);
  
  qlib::LDom2Tree tree;
  ois.read(tree);
  //tree.top()->dump();

  CameraPtr rcam(new Camera);
  tree.deserialize(&rcam);

  return rcam;
}

//////////
// Interfaces

bool Scene::registerCamera(const LString &name, CameraPtr r)
{
  if (!registerCameraImpl(name, r))
    return false;
  {
    // fire camera-added event
    CameraEvent ev;
    ev.setSource(m_nUID);
    ev.m_nEvtType = ScrEventManager::SEM_ADDED;
    ev.m_name = name;
    ScrEventManager *pSEM = ScrEventManager::getInstance();
    pSEM->fireObjectEvent(ev);
  }
  return true;
}

CameraPtr Scene::createCamera(const LString &name)
{
  if (!getCamera(name).isnull())
    return CameraPtr();
  CameraPtr r(new Camera);
  if (!registerCameraImpl(name, r))
    return CameraPtr();
  {
    // fire camera-added event
    CameraEvent ev;
    ev.setSource(m_nUID);
    ev.m_nEvtType = ScrEventManager::SEM_ADDED;
    ev.m_name = name;
    ScrEventManager *pSEM = ScrEventManager::getInstance();
    pSEM->fireObjectEvent(ev);
  }
  return r;
}

/// Load camera info from a local file
CameraPtr Scene::loadCameraFrom(const LString &filename, const LString &name)
{
  if (!getCamera(name).isnull())
    return CameraPtr();

  CameraPtr r;
  try {
    r = loadCameraImpl(filename);
  }
  catch (...) {
    LOG_DPRINTLN("Fatal Error: Load camera %s is failed", filename.c_str());
    return CameraPtr();
  }

  if (!registerCameraImpl(name, r))
    return CameraPtr();

  {
    // fire camera-added event
    CameraEvent ev;
    ev.setSource(m_nUID);
    ev.m_nEvtType = ScrEventManager::SEM_ADDED;
    ev.m_name = name;
    ScrEventManager *pSEM = ScrEventManager::getInstance();
    pSEM->fireObjectEvent(ev);
  }
  return r;
}

/// Save camera info to a local file
bool Scene::saveCameraTo(const LString &aLocalFile, const LString &aName)
{
  CameraPtr rcam = getCamera(aName);
  if (rcam.isnull())
    return false;

  CameraPtr r;
  try {
    LString path = resolveBasePath(aLocalFile);

    qlib::FileOutStream fis;
    fis.open(path);
    qlib::LDom2OutStream ois(fis);
  
    qlib::LDom2Tree tree("camera");
    
    // Camera obj --> LDOM2 tree
    // top node doesn't use type attribute as object's typename
    tree.serialize(rcam.get(), false);

    // LDOM2 tree --> Stream (local file)
    ois.write(&tree);
    // tree.top()->dump();
  }
  catch (...) {
    LOG_DPRINTLN("Fatal Error: Load camera %s is failed", aLocalFile.c_str());
    return false;
  }

  return true;
}

CameraPtr Scene::getCamera(const LString &nm) const
{
  camtab_t::const_iterator i = m_camtab.find(nm);
  if (i==m_camtab.end())
    return CameraPtr();
  return i->second;
}

bool Scene::destroyCamera(const LString &nm)
{
  {
    // fire camera-removing event
    CameraEvent ev;
    ev.setSource(m_nUID);
    ev.m_nEvtType = ScrEventManager::SEM_REMOVING;
    ev.m_name = nm;
    ScrEventManager *pSEM = ScrEventManager::getInstance();
    pSEM->fireObjectEvent(ev);
  }

  camtab_t::iterator i = m_camtab.find(nm);
  if (i==m_camtab.end())
    return false;
  m_camtab.erase(i);
  return true;
}

/// View --> Camera save
bool Scene::saveViewToCam(qlib::uid_t viewid, const LString &nm)
{
  ViewPtr rv = getView(viewid);
  if (rv.isnull())
    return false;
  CameraPtr rc = getCamera(nm);
  if (rc.isnull())
    rc = createCamera(nm);

  if(!rv->saveTo(rc)) return false;

  {
    // fire camera-changed event
    CameraEvent ev;
    ev.setSource(m_nUID);
    ev.m_nEvtType = ScrEventManager::SEM_CHANGED;
    ev.m_name = nm;
    ScrEventManager *pSEM = ScrEventManager::getInstance();
    pSEM->fireObjectEvent(ev);
  }
  return true;
}

/// Camera --> View restore
bool Scene::loadViewFromCam(qlib::uid_t viewid, const LString &nm)
{
  ViewPtr rv = getView(viewid);
  if (rv.isnull())
    return false;
  CameraPtr rc = getCamera(nm);
  if (rc.isnull())
    return false;

  if (!rv->loadFrom(rc))
    return false;

  // Redraw of view (rv) is required
  rv->setUpdateFlag();
  return true;
}

LString Scene::getCameraInfoJSON() const
{
  LString rval = "[";
  
  camtab_t::const_iterator viter = m_camtab.begin();
  camtab_t::const_iterator eiter = m_camtab.end();
  for (; viter!=eiter; ++viter) {
    if (viter!=m_camtab.begin())
      rval += ",";
    CameraPtr obj = viter->second;
    rval += "\""+ viter->first.escapeQuots() +"\"";
  }

  rval += "]";
  return rval;
}

////////////////////////////////////////////////////////////////////
// Undo info

void Scene::commitUndoTxn()
{
  m_undomgr.commitTxn();

  // UndoInfo event
  {
    // Fire the SCE_OBJ_ADDED Event
    MB_DPRINTLN("UndoInfo> Firing SCE_SCENE_UNDOINFO event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_SCENE_UNDOINFO);
    ev.setSource(getUID());
    ev.setTarget(getUID());
    // ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }
}

bool Scene::undo(int n)
{
  return m_undomgr.undo(n);
}

bool Scene::redo(int n)
{
  return m_undomgr.redo(n);
}

void Scene::clearUndoDataScr()
{
  if (m_undomgr.getUndoSize()==0&&
      m_undomgr.getRedoSize()==0)
    return; // --> already empty

  m_undomgr.clearAllInfo();

  // (notify changes of Undo/Redo information)
  {
    // Fire the SCE_SCENE_UNDOINFO event
    MB_DPRINTLN("UndoInfo> Firing SCE_SCENE_UNDOINFO event...");
    SceneEvent ev;
    ev.setType(SceneEvent::SCE_SCENE_UNDOINFO);
    ev.setSource(getUID());
    ev.setTarget(getUID());
    // ev.setTarget(robj->getUID());
    fireSceneEvent(ev);
  }
}

////////////////////////////////////////////////////////////////////
// Serialization

// #include <qlib/ObjStream.hpp>
#include <qlib/LDOM2Stream.hpp>

void Scene::writeTo2(qlib::LDom2Node *pNode) const
{
  // Write properties of the scene
  super_t::writeTo2(pNode);

  // Write styles of the scene
  StyleFile sfile;
  sfile.saveToNode(pNode, getUID(), getBasePath());

  // Write objects of the scene
  data_t::const_iterator oiter = m_data.begin();
  for (; oiter!=m_data.end(); ++oiter) {
    ObjectPtr obj = oiter->second;

    qlib::LDom2Node *pChNode = pNode->appendChild();
    pChNode->setTagName("object");
    pChNode->setTypeNameByObj(obj.get());
    obj->writeTo2(pChNode);
  }

  // Write camera of the scene
  camtab_t::const_iterator viter = m_camtab.begin();
  for (; viter!=m_camtab.end(); ++viter) {
    CameraPtr obj = viter->second;

    qlib::LDom2Node *pChNode = pNode->appendChild();
    pChNode->setTagName("camera");
    obj->writeTo2(pChNode);
  }

}

void Scene::readFrom2(qlib::LDom2Node *pNode)
{
  // read properties of scene
  super_t::readFrom2(pNode);

  StyleFile sfile;

  for (pNode->firstChild(); pNode->hasMoreChild(); pNode->nextChild()) {
    qlib::LDom2Node *pChNode = pNode->getCurChild();
    LString tag = pChNode->getTagName();
    LString type_name = pChNode->getTypeName();

    if (tag.equals("object") && !type_name.isEmpty()) {
      ObjectPtr robj = pChNode->createObjByTypeNameT<Object>();
      // Object's properties should be built before registration to the scene,
      //   to prevent event propargation.
      robj->readFrom2(pChNode);
      if (!pChNode->isChildrenConsumed()) {
	// TO DO: report error (unknown element)
	LOG_DPRINTLN("Scene::readFrom2> Warning:"
		     " some nodes (of obj) are not consumed");
        //pChNode->dump();
      }

      // Register the built object 
      registerObjectImpl(robj);
    }
    else if (tag.equals("camera")) {
      CameraPtr rcam;

      LString cam_src = pChNode->getStrAttr("src");
      // LString cam_name = pChNode->getStrAttr("name");
      if (!cam_src.isEmpty()) {
        rcam = loadCameraImpl(cam_src);
      }
      else {
        rcam = CameraPtr(new Camera);
      }

      // in-file setting will overwrite the external source setting
      rcam->readFrom2(pChNode);

      if (!pChNode->isChildrenConsumed()) {
	// TO DO: report error (unknown element)
	LOG_DPRINTLN("Scene::readFrom2> Warning: some Camera nodes are not consumed");
        //pChNode->dump();
      }

      const LString &nm = rcam->m_name;
      if (!registerCamera(nm, rcam)) {
	// TO DO: report error (unknown element)
	LOG_DPRINTLN("Scene::readFrom2> Error: cannot register camera \"%s\"",
		     nm.c_str());
      }
      
    }
    else if (tag.equals("styles")) {
      LString style_src = pChNode->getStrAttr("src");
      if (!style_src.isEmpty()) {
        LString path = resolveBasePath(style_src);
        sfile.loadFile(path, getUID());
      }
      else {
        // load local style nodes
        sfile.loadNodes(pChNode, getUID());
      }
    }
    else if (tag.equals("script")) {

      LString value = pChNode->getValue();
      LString contents = pChNode->getContents();
      if (value.isEmpty() && !contents.isEmpty())
        value = contents;

      // TO DO: set file/lineno info for eval()
      // TO DO: save the original script for serialization
      if (m_pInterp!=NULL)
        m_pInterp->eval(value);
    }
    else {
      continue;
    }

    pChNode->setConsumed(true);
  }
  
}

bool Scene::execJSFile(const LString &scr)
{
  LString path = resolveBasePath(scr);
  return m_pInterp->execFile(path);
}

