//
// Stream manager singleton class
//
// $Id: StreamManager.cpp,v 1.17 2010/12/15 00:19:08 rishitani Exp $
//

#include <common.h>

#include "StreamManager.hpp"
#include "ObjReader.hpp"
#include "ObjWriter.hpp"
#include <qlib/ClassRegistry.hpp>

#include <qlib/FileStream.hpp>

#ifdef HAVE_BOOST_THREAD
#define BOOST_LIB_DIAGNOSTIC 1
//#define BOOST_DYN_LINK 1
#define BOOST_ALL_DYN_LINK 1
#include <boost/thread.hpp>
#include <boost/thread/condition.hpp>
#include <boost/bind.hpp>
#include "PipeStream.hpp"
#endif

#include "SceneManager.hpp"

using namespace qsys;

SINGLETON_BASE_IMPL(qsys::StreamManager);

///////////////

StreamManager::StreamManager()
{
  MB_DPRINTLN("StreamManager(%p) created", this);
}

StreamManager::~StreamManager()
{
  MB_DPRINTLN("StreamManager(%p) destructed", this);
}

/*
ObjectPtr StreamManager::loadObject(const LString &url, const LString &ftype)
{
  MB_DPRINTLN("StreamManager.loadObject(%s,%s) called", url.c_str(), ftype.c_str());

  // Create the requested reader obj
  ObjReader *pRdr = createReader(ftype);
  if (pRdr==NULL) {
    LOG_DPRINTLN("StreamManager> Reader %s not found", ftype.c_str());
    return ObjectPtr();
  }
    
  // Construct a stream from the requested url
  qlib::FileInStream fis;
  ObjectPtr pret;
  try {
    // TO DO: handle URL
    fis.open(url);
    pret = pRdr->load(fis);
  }
  catch (...) {
    delete pRdr;
    throw;
  }

  // END
  delete pRdr;

  pret->setSource(url);
  pret->setSourceType(ftype);
  return pret;
}
 */



#ifdef HAVE_BOOST_THREAD

int StreamManager::loadObjectAsync(const LString &ftype)
{
  MB_DPRINTLN("StreamManager.loadAsyncObject(%s) called", ftype.c_str());

  // Create the requested reader obj
  ObjReader *pRdr = createReaderPtr(ftype);
  if (pRdr==NULL) {
    LOG_DPRINTLN("StreamManager> Reader %s not found", ftype.c_str());
    return -1;
  }
    
  IOThread *pThr = new IOThread;
  pThr->m_pRdr = pRdr;
  int tid = m_iotab.put(pThr);
  pThr->kick();

  return tid;
}

void StreamManager::supplyDataAsync(int id, const char *pbuf, int nlen)
{
  MB_DPRINTLN("StreamManager.supplyDataAsync(%d, size=%d) called",id, nlen);
  IOThread *pThr = m_iotab.get(id);
  if (pThr==NULL) return;
  pThr->supplyData(pbuf, nlen);
}

ObjectPtr StreamManager::waitLoadAsync(int id)
{
  IOThread *pThr = m_iotab.get(id);
  if (pThr==NULL) return ObjectPtr();
  
  pThr->notifyEos();
  pThr->waitTermination();

  ObjectPtr pret = pThr->m_pObj;
  m_iotab.remove(id);
  delete pThr;

  pret->setSource("");
  return pret;
}

#endif

///////////////

void StreamManager::regIOHImpl(const LString &abiname)
{
  LString nickname,descr,fext;
  int ntype;

  qlib::ClassRegistry *pCR = qlib::ClassRegistry::getInstance();
  qlib::LClass *pCls = pCR->getClassObjByAbiName(abiname);
  
  // Create a dummy instance to retrieve type information
  {
    InOutHandler *pIOH = dynamic_cast<InOutHandler *>(pCls->createObj());
    if (pIOH==NULL) {
      LString msg = LString::format("Class %s is not ObjReader", abiname.c_str());
      MB_THROW(qlib::InvalidCastException, msg);
      return;
    }

    nickname = pIOH->getName();
    descr = pIOH->getTypeDescr();
    fext = pIOH->getFileExt();
    ntype = pIOH->getCatID();

    delete pIOH;
  }

  ReaderInfo ri;
  ri.nickname = nickname;
  ri.descr = descr;
  ri.fext = fext;
  ri.pClass = pCls;
  ri.nCatID = ntype;

  if (!m_rdrinfotab.set(abiname, ri)) {
    LString msg = LString::format("Reader/Writer <%s> already exists", abiname.c_str());
    MB_THROW(qlib::RuntimeException, msg);
    return;
  }
}

bool StreamManager::unregistReader(const LString &abiname, bool bWriter /*= false*/)
{
  // TO DO: implementation
  return false;
}

bool StreamManager::isReaderRegistered(const LString &abiname)
{
  return m_rdrinfotab.containsKey(abiname);
}


InOutHandler *StreamManager::createHandlerPtr(const LString &nickname, int nCatID) const
{
  BOOST_FOREACH(const data_t::value_type &entry, m_rdrinfotab) {
    const LString &nknm = entry.second.nickname;
    if (!nickname.equals(nknm))
      continue;
    if (nCatID!=entry.second.nCatID)
      continue;
    qlib::LClass *pCls = entry.second.pClass;
    MB_ASSERT(pCls!=NULL);
    
    InOutHandler *pObj = dynamic_cast<InOutHandler *>(pCls->createObj());
    if (pObj==NULL) {
      // This should not happen!!
      LOG_DPRINTLN("Reader %s is not ObjReader", nknm.c_str());
      continue;
    }

    pObj->resetAllProps();
    return pObj;
  }

  // not found!!
  return NULL;
}

ObjReader *StreamManager::createReaderPtr(const LString &nickname) const
{
  return dynamic_cast<ObjReader *>(createHandlerPtr(nickname, InOutHandler::IOH_CAT_OBJREADER));
}


LString StreamManager::getReaderInfoJSON() const
{
  return getIOHInfoJSONImpl(InOutHandler::IOH_CAT_OBJREADER);
}

LString StreamManager::getWriterInfoJSON() const
{
  return getIOHInfoJSONImpl(InOutHandler::IOH_CAT_OBJWRITER);
}

LString StreamManager::getIOHInfoJSONImpl(int aCatID) const
{
  int i;
  std::list<LString> tmps;
  LString rval;
  data_t::const_iterator iter;

  rval += "({ ";

  BOOST_FOREACH(const data_t::value_type &entry, m_rdrinfotab) {
    //iter = m_rdrinfotab.begin();
    //for (i=0; iter!=m_rdrinfotab.end(); ++iter) {
    //if (iter->second.bWriter!=bWriter)
    //continue;
    if (entry.second.nCatID!=aCatID)
      continue;
    LString descr = entry.second.descr;
    tmps.push_back("\"" + descr + "\"");
  }  

  int nent = tmps.size();
  rval += LString::format("size: %d", nent);
  if (nent<=0) {
    rval += " })\n";
    return rval;
  }

  rval += ",\n";

  rval += "descrs:[ ";
  rval += LString::join(",\n", tmps);
  rval += "],\n";
  tmps.erase(tmps.begin(), tmps.end());
  
  BOOST_FOREACH(const data_t::value_type &entry, m_rdrinfotab) {
    //iter = m_rdrinfotab.begin();
    //for (i=0; iter!=m_rdrinfotab.end(); ++iter) {
    if (entry.second.nCatID!=aCatID)
      continue;
    //if (entry.second.bWriter!=bWriter)
    //continue;
    LString fext = entry.second.fext;
    tmps.push_back("\"" + fext + "\"");
  }  

  rval += "fexts:[ ";
  rval += LString::join(",\n", tmps);
  rval += "],\n";
  tmps.erase(tmps.begin(), tmps.end());

  BOOST_FOREACH(const data_t::value_type &entry, m_rdrinfotab) {
    //iter = m_rdrinfotab.begin();
    //for (i=0; iter!=m_rdrinfotab.end(); ++iter) {
    //if (iter->second.bWriter!=bWriter)
    //continue;
    if (entry.second.nCatID!=aCatID)
      continue;
    LString s = entry.second.nickname;
    tmps.push_back("\"" + s + "\"");
  }  

  rval += "nicknames:[ ";
  rval += LString::join(",\n", tmps);
  rval += "] })\n";

  return rval;
}

LString StreamManager::getInfoJSON2() const
{
  std::list<LString> tmps;
  LString rval;
  data_t::const_iterator iter;

  rval += "[";

  bool bFirst = true;
  BOOST_FOREACH(const data_t::value_type &entry, m_rdrinfotab) {

    if (!bFirst)
      rval += ",";
    bFirst = false;

    rval += "{";

    rval += LString("\"descr\": \"") + entry.second.descr + "\",";
    rval += LString("\"fext\": \"") + entry.second.fext + "\",";
    rval += LString("\"name\": \"") + entry.second.nickname + "\",";
    rval += LString::format("\"category\": %d", entry.second.nCatID);

    rval += "}";
  }  

  rval += "]";

  return rval;
}

/*
LString StreamManager::getInitRendererNames(const LString &rdrnm) const
{
  // Create the requested reader obj
  ObjReader *pRdr = createReaderPtr(rdrnm);
  if (pRdr==NULL) {
    LOG_DPRINTLN("StreamManager> Reader %s not found", rdrnm.c_str());
    return LString();
  }

  ObjectPtr pObj = pRdr->createDefaultObj();
  if (pObj.isnull()) {
    LOG_DPRINTLN("StreamManager> No default obj found for Reader %s", rdrnm.c_str());
    return LString();
  }

  LString rval;
  rval = pObj->searchCompatibleRendererNames();
  delete pRdr;

  return rval;
}
*/

LString StreamManager::findCompatibleWriterNamesForObj(qlib::uid_t objid)
{
  ObjectPtr pObj = SceneManager::getInstance()->getObject(objid);

  const int kCatID = InOutHandler::IOH_CAT_OBJWRITER;
  std::list<LString> ls;

  BOOST_FOREACH(const data_t::value_type &entry, m_rdrinfotab) {
    const LString &nknm = entry.second.nickname;
    if (kCatID!=entry.second.nCatID)
      continue;
    qlib::LClass *pCls = entry.second.pClass;
    MB_ASSERT(pCls!=NULL);
    
    ObjWriter *pObjWr = dynamic_cast<ObjWriter *>(pCls->createObj());
    if (pObjWr==NULL) {
      // This should not happen!!
      LOG_DPRINTLN("Fatal Error: Handler %s is not ObjWriter", nknm.c_str());
      continue;
    }

    if (pObjWr->canHandle(pObj)) {
      ls.push_back(nknm);
    }
    delete pObjWr;
  }

  if (ls.size()==0)
    return LString();

  return LString::join(",", ls);
}


