// -*-Mode: C++;-*-
//
// View: Generic Molecule View Class
//
// $Id: View.cpp,v 1.48 2011/03/13 12:02:45 rishitani Exp $
//

#include <common.h>

#include "View.hpp"
#include "Scene.hpp"
#include "SceneManager.hpp"
#include "ScrEventManager.hpp"
#include "DrawObj.hpp"

#include <gfx/DisplayContext.hpp>
#include <qlib/Utils.hpp>
#include <qlib/Matrix4D.hpp>
#include <qlib/LVarArgs.hpp>
#include <qlib/ClassRegistry.hpp>
#include <qlib/LClassUtils.hpp>

using namespace qsys;

View::View()
{
  m_pEvtCaster = new ViewEventCaster;
  m_uid = qlib::ObjectManager::sRegObj(this);
  MB_DPRINTLN("View (%p/%d) created\n", this, m_uid);

  // ???
  m_nWidth = 100;
  m_nHeight = 100;

  m_tkrad = 0.8f;
  // m_fViewDist = 200.0f;

  // m_ltAmbi = Vector4D(0.2f, 0.2f, 0.2f);
  // m_ltDiff = Vector4D(0.8f, 0.8f, 0.8f);
  // m_ltSpec = Vector4D(.4f, .4f, .4f);

//  m_fHitPrec = 10.0;

  addListener(this);

  m_cursorName = "default";
  m_bActive = true;
}

View::View(const View &r)
{
  m_uid = qlib::ObjectManager::sRegObj(this);
  MB_DPRINTLN("Cannot create View copy (%p/%d)\n", this, m_uid);
  MB_ASSERT(false);
}

View::~View()
{
  m_drawObjTab.clear();

  delete m_pEvtCaster;
  MB_DPRINTLN("View(%p) destructed\n", this);
  qlib::ObjectManager::sUnregObj(m_uid);
}

//////////////////////////////////

void View::unloading()
{
}

LString View::toString() const
{
  return LString::format("View(name=%s)",m_name.c_str());
}

void View::dump() const
{
  MB_DPRINT("View: %s/%d(%p)", m_name.c_str(), m_uid, this);
}

//////////////////////////////////

void View::setZoom(double f)
{
  if (!qlib::isNear4<double>(f, m_curcam.m_fZoom)) {
    m_curcam.m_fZoom = f;
    setUpdateFlag();
  }
}

void View::setSlabDepth(double d)
{
  if (d<=0.1)
    d = 0.1;
  if (d>=10000.0)
    d = 10000.0;
  if (!qlib::isNear4<double>(m_curcam.m_fSlabDepth, d)) {
    m_curcam.m_fSlabDepth = d;
    setUpdateFlag();
  }
}

void View::setViewCenter(const qlib::Vector4D &pos)
{
  m_curcam.m_center = pos;
  setUpdateFlag();

  // fire event
  {
    ViewEvent ev;
    ev.setType(ViewEvent::VWE_PROPCHG);
    ev.setDescr("center");
    fireViewEvent(ev);
  }
}

void View::setViewCenterDrag(const qlib::Vector4D &pos)
{
  m_curcam.m_center = pos;
  setUpdateFlag();

  // fire event
  {
    ViewEvent ev;
    ev.setType(ViewEvent::VWE_PROPCHG_DRG);
    ev.setDescr("center");
    fireViewEvent(ev);
  }
}

void View::setRotQuat(const qlib::LQuat &q)
{
  m_curcam.m_rotQuat = q;
  // normalization
  m_curcam.m_rotQuat.normalizeSelf();
  setUpdateFlag();
}


/// Rotate view around axes
void View::rotateView(double dax, double day, double daz)
{
  if (qlib::abs<double>(dax)>F_EPS8) {
    qlib::Vector4D xaxis(1,0,0);
    double ax = qlib::toRadian(dax)/2.0;
    m_curcam.m_rotQuat = m_curcam.m_rotQuat * qlib::LQuat(xaxis, ax);
  }

  if (qlib::abs<double>(day)>F_EPS8) {
    qlib::Vector4D yaxis(0,1,0);
    double ay = qlib::toRadian(day)/2.0;
    m_curcam.m_rotQuat = m_curcam.m_rotQuat * qlib::LQuat(yaxis, ay);
  }

  if (qlib::abs<double>(daz)>F_EPS8) {
    qlib::Vector4D zaxis(0,0,1);
    double az = qlib::toRadian(daz)/2.0;
    m_curcam.m_rotQuat = m_curcam.m_rotQuat * qlib::LQuat(zaxis, az);
  }

  //MB_DPRINTLN("QUAT len=%f", m_curcam.m_rotQuat.sqlen());

  m_curcam.m_rotQuat.normalizeSelf(F_EPS8);

  //MB_DPRINTLN("After NS QUAT len=%f", m_curcam.m_rotQuat.sqlen());

  setUpdateFlag();
}

void View::translateView(double dax, double day, double daz)
{
  Vector4D dv1, dv2;
  if (qlib::abs<double>(dax)>F_EPS8 || qlib::abs<double>(day)>F_EPS8) {
    convXYTrans(dax, day, dv1);
  }

  if (qlib::abs<double>(daz)>F_EPS8) {
    convZTrans(daz, dv2);
  }

  setViewCenterDrag(getViewCenter()-dv1-dv2);
  setUpdateFlag();
}

// create quaternion rotation (x1,y1) --> (x2,y2)
void View::trackBallMove(double curX, double curY, double prevX, double prevY)
{
  /*
  // We need to perform some transformation on X in para/cross stereo mode.
  int nSteMode = getStereoMode();
  if (nSteMode==STEREO_PARA ||
      nSteMode==STEREO_CROSS) {
    if (prevX<0.5f)
      prevX = 2.0f*prevX;
    else
      prevX = 2.0f*(prevX-0.5f);

    if (curX<0.5f)
      curX = 2.0f*curX;
    else
      curX = 2.0f*(curX-0.5f);
  }
  */
  //lock();
  
  m_curcam.m_rotQuat.mulSelf(getTrackRotQuat(curX, curY, prevX, prevY, m_tkrad));
  m_curcam.m_rotQuat.normalizeSelf(F_EPS8);

  //MB_DPRINT("Q(%f,%f,%f,%f) len %f",
  //m_curcam.m_rotQuat.Vx(), m_curcam.m_rotQuat.Vy(),
  //m_curcam.m_rotQuat.Vz(), m_curcam.m_rotQuat.a(), sql);

  //const double sql2 = ::sqrt(m_curcam.m_rotQuat.sqlen());
  //MB_DPRINTLN(" --> Q(%f,%f,%f,%f) len %f",
  //m_curcam.m_rotQuat.Vx(), m_curcam.m_rotQuat.Vy(),
  //m_curcam.m_rotQuat.Vz(), m_curcam.m_rotQuat.a(), sql2);

  //m_curcam.m_rotQuat = m_curcam.m_rotQuat * getTrackRotQuat(curX, curY, prevX, prevY, m_tkrad);
  //m_curcam.m_rotQuat.normalizeSelf();

  //unlock();

  setUpdateFlag();
}

namespace {
  void projSphere(qlib::Vector4D &vec, double tkrad)
  {
    double d = (double) vec.length();
    if (d < tkrad*0.70710678118654752440) {
      // inside sphere
      vec.z() = sqrt(tkrad*tkrad - d*d);
    } else {
      // on hyperbola
      vec.z() = (tkrad*tkrad / 2) / d;
    }
  }
}

// static
qlib::LQuat View::getTrackRotQuat(double curX, double curY,
			   double prevX, double prevY, double tkrad)
{
  double x1, y1 ,x2 , y2;

  x1 = 2.0f*prevX - 1.0;
  x2 = 2.0f*curX - 1.0;
  y1 = 1.0 - 2.0*prevY;
  y2 = 1.0 - 2.0*curY;
  
  qlib::Vector4D p1(x1, y1, 0);
  projSphere(p1, tkrad);

  qlib::Vector4D p2(x2, y2, 0);
  projSphere(p2, tkrad);

  qlib::Vector4D vaxis = p2.cross(p1);
  vaxis /= vaxis.length();
  qlib::Vector4D vdist(p1);
  vdist -= p2;
  double t = vdist.length()/(2.0*tkrad);
  if (t>1.0)
    t = 1.0;
  if (t<-1.0)
    t = -1.0;

  double phi = 2.0*asin(t);
  //MB_DPRINT("rot phi=%f\n", phi/M_PI*180);
  if (phi<1E-5) {
    //MB_DPRINT("rot phi=%f too small!!\n", phi/M_PI*180);
    return qlib::LQuat(1.0, 0.0, 0.0, 0.0);
  }

  return qlib::LQuat(vaxis, phi);
}

void View::convXYTrans(double adx, double ady, Vector4D &vec)
{
  double h = getHeight();
  double dx = adx*m_curcam.m_fZoom/h;
  double dy = -ady*m_curcam.m_fZoom/h;

  //lock();
  LQuat tmpq = m_curcam.m_rotQuat.conj();
  //unlock();
  Matrix4D rmat = Matrix4D::makeRotMat(tmpq);

  vec.x() = dx;
  vec.y() = dy;
  vec.z() = 0;
  vec.w() = 0;
  rmat.xform3D(vec);
}

void View::convZTrans(int idz, Vector4D &vec)
{
  double dz = idz;
  // XXX ???
  //dz *= m_fViewWidth;
  dz /= 4.0;

  //lock();
  LQuat tmpq = m_curcam.m_rotQuat.conj();
  //unlock();

  Matrix4D rmat = tmpq.toRotMatrix();

  vec = Vector4D(0, 0, -dz, 0);
  rmat.xform4D(vec);
}

/*
void View::setBgColor(const LColor &vec)
{
  bool fchg=false;
  lock();
  if (!m_bgColor.equals(vec)) {
    m_bgColor = vec;
    fchg = true;
  }
  unlock();
  setUpdateFlag();
}
*/

void View::setPerspec(bool b)
{
  if (m_curcam.m_fPerspec != b) {
    m_curcam.m_fPerspec = b;
    setUpdateFlag();
  }
}


void View::setStereoMode(int f)
{
  if (m_curcam.m_nStereoMode != f) {
    m_curcam.m_nStereoMode = f;
    setUpdateFlag();
  }
}


/*
void View::setStereoDist(double f)
{
  bool fchg=false;
  lock();
  if (m_fStereoDist != f) {
    m_fStereoDist = f;
    setUpdateFlag();
    fchg = true;
  }
  unlock();

  if (fchg) getDisplayMgr()->incrInvRend();
}
*/

/////////////////////////////////////////////////////
// Utility routines

bool View::safeSetCurrent()
{
  gfx::DisplayContext *pCtxt = getDisplayContext();
  if (pCtxt==NULL) return false;
  if (!pCtxt->isCurrent())
    if (!pCtxt->setCurrent()) {
      LOG_DPRINTLN("View::setup() setCurrent failed.");
      return false;
    }
  return true;
}

/////////////////////////////////////////////////////
// View size

void View::sizeChanged(int cx, int cy)
{
  m_nWidth = cx;
  m_nHeight = cy;
  setUpProjMat(-1, -1);
  
  setUpdateFlag();

  // fire event
  {
    ViewEvent ev;
    ev.setType(ViewEvent::VWE_SIZECHG);
    fireViewEvent(ev);
  }
}

int View::getWidth()
{
  return m_nWidth;
}

int View::getHeight()
{
  return m_nHeight;
}


////////////////////////////////////////////////////////
// InDevEvent message handlers

/** mouse drag start event */
bool View::mouseDragStart(InDevEvent &ev)
{
  // initialize work flags
  m_nZoomSlab = 0;
  m_nRotTranZ = 0;

  // save pre-drag state
  //m_saveViewCenter = getViewCenter();
  //m_saveRotQuat = getRotQuat();
  m_bCenterChanged = false;

  setCursor("-moz-grabbing");
  return true;
}

void View::zoomSlab(int delslab, int delzoom)
{
  // change the slab width or zoom
  if (m_nZoomSlab==0) {
    int dx = qlib::abs<int>(delslab);
    int dy = qlib::abs<int>(delzoom);
    if (dx>dy)
      m_nZoomSlab = 2;
    else if (dx<dy)
      m_nZoomSlab = 1;
  }
  else if (m_nZoomSlab==1) {
    // ZOOM mode
    double vw = getZoom();
    double dw = double(delzoom)/200.0 * vw;
    View::setZoom(vw+dw);
  }
  else {
    // SLAB mode
    double vw = getSlabDepth();
    double dw = double(delslab)/200.0 * vw;
    View::setSlabDepth(vw+dw);
    //setSlabDepth(getSlabDepth()+double(ev.getDeltaX())/20.0);
  }
  
  setUpProjMat(-1, -1);
  return;
}

void View::rotTranZ(int delrot, int deltran)
{
  Vector4D vec;
  
  // rotZ or tranZ
  if (m_nRotTranZ==0) {
    int dr = qlib::abs<int>(delrot);
    int dt = qlib::abs<int>(deltran);
    if (dr>dt)
      m_nRotTranZ = 2; // Z-rotation mode
    else if (dr<dt)
      m_nRotTranZ = 1; // Z-translation mode
  }
  else if (m_nRotTranZ==1) {
    // Z-translation mode
    convZTrans(deltran, vec);
    setViewCenterDrag(getViewCenter()-vec);
    m_bCenterChanged = true;
  }
  else {
    // Z-rotation mode
    rotateView(0.0f, 0.0f, double(delrot)/4.0f);
  }
  
  return;
}


void View::rotXY(int posx, int posy,
		 int delx, int dely,
		 double width, double height)
{
  //  MB_DPRINTLN("View::rotXY(%d,%d) (%d,%d) (%f,%f)",
  //	      posx, posy,
  //	      delx, dely,
  //	      width, height);

  // rotate view
  //double width = pview->getWidth();
  //double height = pview->getHeight();
  
  trackBallMove(double(posx)/width,
                double(posy)/height,
                double(posx-delx)/width,
                double(posy-dely)/height);
  return;
}

/** mouse drag move event */
bool View::mouseDragMove(InDevEvent &ev)
{
  // View *pview = ev.getSource();
  Vector4D vec;

  double width = (double) getWidth();
  double height = (double) getHeight();

  if (ev.isRButtonOn()) {
    if (ev.isLButtonOn()) {
      // L&B buttons (z rot/z trans)
      rotTranZ(ev.getDeltaX(), ev.getDeltaY());
    }
    else {
      convXYTrans(ev.getDeltaX(), ev.getDeltaY(), vec);
      // MB_DPRINTLN("drag move %f,%f,%f", vec.x(), vec.y(), vec.z());    
      setViewCenterDrag(getViewCenter()-vec);
      m_bCenterChanged = true;
    }
  }
  else if (ev.isLButtonOn()) {
    if (ev.isShiftOn()) {
      zoomSlab(ev.getDeltaX(), ev.getDeltaY());
    }
    else {
      rotXY(ev.getX(), ev.getY(),
	    ev.getDeltaX(), ev.getDeltaY(),
	    width, height);
    }
  }

  forceRedraw();
  //pview->update();
  //pview->drawScene();

  return true;
}

bool View::mouseWheel(InDevEvent &ev)
{
  // View *pview = ev.getSource();
  const double del = double(ev.getDeltaX())/500.0;
  double cv = getZoom();
  cv += cv*del;
  cv = qlib::max(cv, 0.1);
  cv = qlib::min(cv, 1000.0);
  //  MB_DPRINTLN("zoom: %f", cv);

  setZoom(cv);
  forceRedraw();

  return true;
}

/** mouse drag end event */
bool View::mouseDragEnd(InDevEvent &ev)
{
  if (m_bCenterChanged) {
    setViewCenter(getViewCenter());
  }

#if (GUI_ARCH==OSX)
  setCursor("auto");
#else
  setCursor("default");
#endif
  return true;
}

/** mouse click event (L,M,R button)*/
bool View::mouseClicked(InDevEvent &ev)
{
  //clickHelper(ev, false);
  return true;
}

/** mouse double click event (L,M,R button)*/
bool View::mouseDoubleClicked(InDevEvent &ev)
{
  //clickHelper(ev, true);
  return true;
}

gfx::DisplayContext *View::getSiblingCtxt()
{
  ScenePtr rsc = getScene();
  if (rsc.isnull()) return NULL;

  Scene::ViewIter viter = rsc->beginView();
  for (; viter!=rsc->endView(); ++viter) {
    if (viter->first!=m_uid) {
      qsys::ViewPtr sibvw = viter->second;
      if (sibvw.isnull())
	continue;
      return sibvw->getDisplayContext();
    }
  }
  return NULL;
}

qlib::uid_t View::getRootUID() const
{
  return getUID();
}

/////////////////////////////////////////////////////////
// Events

int View::addListener(InDevListener *p)
{
  return m_listeners.add(p);
}

bool View::removeListener(InDevListener *p)
{
  return m_listeners.remove(p);
}

bool View::removeListener(int nid)
{
  if (m_listeners.remove(nid)==NULL)
    return false;
  return true;
}

/*int View::addViewListener(ViewEventListener *pL)
{
  return m_pEvtCaster->add(pL);
}*/

/*bool View::removeViewListener(ViewEventListener *pL)
{
  return m_pEvtCaster->remove(pL);
}*/

void View::fireViewEvent(ViewEvent &ev)
{
  m_pEvtCaster->replicaFire(ev);

  ScrEventManager *pSEM = ScrEventManager::getInstance();
  ev.setTarget(getUID());
  ev.setTargetPtr(this);
  ev.setSource(m_nSceneID);
  pSEM->fireViewEvent(ev);
}

///////////////////////////////////////////

bool View::saveTo(CameraPtr rcam) const
{
  *(rcam.get()) = m_curcam;
  return true;
}

bool View::loadFrom(CameraPtr rcam)
{
  m_curcam = Camera(*(rcam.get()));
  // camera changed, so we must update proj matrix
  if (getDisplayContext()!=NULL)
    setUpProjMat(-1,-1);
  return true;
}

ScenePtr View::getScene() const
{
  return SceneManager::getSceneS(m_nSceneID);
}

void View::setCursor(const LString &cursor)
{
  m_cursorName = cursor;

  InDevEvent ev;
  ev.setType(InDevEvent::INDEV_MOUSE_ENTER);
  ev.setModifier(0);
  ev.setSource(this);
  fireInDevEvent(ev);
}

void View::fireInDevEvent(InDevEvent &ev)
{
  LString category;
  const int nev = ev.getType();
  switch (nev) {
  case InDevEvent::INDEV_LBTN_CLICK:
  case InDevEvent::INDEV_RBTN_CLICK:
  case InDevEvent::INDEV_MBTN_CLICK:
    category = "mouseClicked";
    break;
        
  case InDevEvent::INDEV_LBTN_DBLCLICK:
  case InDevEvent::INDEV_RBTN_DBLCLICK:
  case InDevEvent::INDEV_MBTN_DBLCLICK:
    category = "mouseDoubleClicked";
    break;

  case InDevEvent::INDEV_DRAG_START:
    category = "mouseDragStart";
    break;

  case InDevEvent::INDEV_DRAG_MOVE:
    category = "mouseDragMove";
    break;

  case InDevEvent::INDEV_DRAG_END:
    category = "mouseDragEnd";
    break;

  case InDevEvent::INDEV_WHEEL:
    category = "mouseWheel";
    break;

  case InDevEvent::INDEV_MOUSE_DOWN:
    category = "mouseDown";
    break;

  case InDevEvent::INDEV_MOUSE_ENTER:
    category = "mouseEnter";
    break;

  default:
    MB_ASSERT(false);
    break;
  }

  // Notify script event listeners
  ScrEventManager *pSEM = ScrEventManager::getInstance();
  bool ok = 
    pSEM->fireEventScript(category,
                          ScrEventManager::SEM_INDEV,
                          ScrEventManager::SEM_OTHER,
                          m_uid, ev);

  // !ok --> Event default propagation is canceled
  if (ok)
    return;

  // Notify native event listeners
  m_listeners.fire(ev);
}

DrawObjPtr View::getDrawObj(const LString &clsname)
{
  drawobjtab_t::const_iterator iter = m_drawObjTab.find(clsname);
  if (iter!=m_drawObjTab.end())
    return iter->second;

  qlib::ClassRegistry *pCR = qlib::ClassRegistry::getInstance();
  qlib::LClass *pCls = pCR->getClassObj(clsname);
  
  qlib::LDynamic *pObj0 = pCls->createObj();
  DrawObj *pObj = dynamic_cast<DrawObj *>(pObj0);

  if (pObj==NULL) {
    LString msg = LString::format("FatalERROR: Class %s is not DrawObj", clsname.c_str());
    MB_THROW(qlib::InvalidCastException, msg);
    return DrawObjPtr();
  }
  
  DrawObjPtr pRes(pObj);
  m_drawObjTab.insert(drawobjtab_t::value_type(clsname, pRes));

  return pRes;
}

void View::showDrawObj(DisplayContext *pdc)
{
  drawobjtab_t::const_iterator iter = m_drawObjTab.begin();
  drawobjtab_t::const_iterator eiter = m_drawObjTab.end();

  for (; iter!=eiter; ++iter) {
    if (!iter->second->isEnabled())
      continue;
    iter->second->display(pdc);
  }
}

void View::showDrawObj2D(DisplayContext *pdc)
{
  drawobjtab_t::const_iterator iter = m_drawObjTab.begin();
  drawobjtab_t::const_iterator eiter = m_drawObjTab.end();

  for (; iter!=eiter; ++iter) {
    if (!iter->second->isEnabled())
      continue;
    iter->second->display2D(pdc);
  }
}

bool View::hasHWStereo() const
{
  return false;
}

//////////////////////////////
// DrawObj

DrawObj::DrawObj()
     : m_bEnabled(false)
{
}

DrawObj::~DrawObj()
{
}

void DrawObj::setEnabled(bool f)
{
  m_bEnabled = f;
}

////////

View *View::createOffScreenView(int w, int h, int aa_depth)
{
  return NULL;
}

void View::readPixels(int x, int y, int width, int height, char *pbuf, int nbufsize, int ncomp)
{
}

void View::swapBuffers()
{
}

