// -*-Mode: C++;-*-
//
// View: Abstract object for the view
//
// $Id: View.hpp,v 1.49 2011/03/18 05:53:45 rishitani Exp $
//

#ifndef QSYS_VIEW_HPP_INCLUDE_
#define QSYS_VIEW_HPP_INCLUDE_

#include "qsys.hpp"

#include <qlib/ObjectManager.hpp>
#include <qlib/LScrObjects.hpp>
#include <qlib/LScrSmartPtr.hpp>
#include <qlib/LScrCallBack.hpp>

#include <qlib/Vector4D.hpp>
#include <qlib/LQuat.hpp>

#include "InDevEvent.hpp"
#include "ViewEvent.hpp"
#include "Camera.hpp"

using qlib::LString;
using qlib::Vector4D;
using qlib::LQuat;

namespace gfx {
  class DisplayContext;
  class Hittest;
}

namespace qsys {

  using gfx::DisplayContext;

  class QSYS_API View :
    public qlib::LNoCopyScrObject,
    public InDevListener,
    public qlib::LUIDObject
  {
    MC_SCRIPTABLE;

  private:
    qlib::uid_t m_uid;
    LString m_name;
    bool m_bActive;

  protected:
    /// Current camera for this view
    Camera m_curcam;

  private:
    /** ID of the scene to which this object belongs. */
    qlib::uid_t m_nSceneID;

    ViewEventCaster *m_pEvtCaster;

    /////////////////////////////

    /// Width of the view (in pixel)
    int m_nWidth;

    /// Height of the view (in pixel)
    int m_nHeight;

    /// view rotation trackball radius
    double m_tkrad;

    /// Input event listeners
    InDevEventCaster m_listeners;

    // for the mouse operation
    int m_nZoomSlab;
    int m_nRotTranZ;
    //Vector4D m_saveViewCenter;
    //LQuat m_saveRotQuat;
    bool m_bCenterChanged;

    /// Name of the mouse cursor type
    LString m_cursorName;

  private:

    /////////////////////////////
    View(const View &r);

    //// = operator (avoid copy operation)
    const View &operator=(const View &arg)
    {
      return *this;
    }

    //////////

  public:
    View();

    virtual ~View();
  
    //////////
  
  public:
    qlib::uid_t getUID() const { return m_uid; }

    LString getName() const { return m_name; }
    void setName(const LString &name) { m_name = name; }

    bool isActive() const { return m_bActive; }
    void setActive(bool b) { m_bActive = b; }

    void setSceneID(qlib::uid_t nid) { m_nSceneID = nid; }
    
    qlib::uid_t getSceneID() const { return m_nSceneID; }
    ScenePtr getScene() const;

    virtual LString toString() const;

    virtual void dump() const;

    virtual gfx::DisplayContext *getDisplayContext() =0;

    virtual void drawScene() =0;
    virtual void swapBuffers();

    virtual void unloading();

    //////////

    float getViewDist() const { return 200.0f; }

    // Zoom factor
    virtual void setZoom(double f);
    double getZoom() const {
      return m_curcam.m_fZoom;
    }

    // Slab depth
    virtual void setSlabDepth(double d);
    double getSlabDepth() const {
      return m_curcam.m_fSlabDepth;
    }

    // Set View center
    virtual void setViewCenter(const qlib::Vector4D &pos);

    // Set View center by mouse dragging
    virtual void setViewCenterDrag(const qlib::Vector4D &pos);

    // Get View center
    qlib::Vector4D getViewCenter() const {
      return m_curcam.m_center;
    }

    // Get View center (for scripting iface)
    qlib::LScrVector4D getViewCenterScr() const {
      return qlib::LScrVector4D(getViewCenter());
    }

    // Set View rotation quaternion
    virtual void setRotQuat(const qlib::LQuat &q);

    // Get View rotation quaternion
    qlib::LQuat getRotQuat() const {
      return m_curcam.getRotQuat();
    }

    // Get View rotation quaternion (for scripting iface)
    qlib::LScrQuat getRotQuatScr() const {
      return qlib::LScrQuat(getRotQuat());
    }

    virtual void setPerspec(bool b);
    bool isPerspec() const {
      return m_curcam.m_fPerspec;
    }
    
    int getCenterMark() const {
      return m_curcam.m_nCenterMark;
    }
    void setCenterMark(int nMode) {
      if (m_curcam.m_nCenterMark != nMode) {
	m_curcam.m_nCenterMark = nMode;
	setUpdateFlag();
      }
    }

    void setTbRad(double d) { m_tkrad = d; }
    double getTbRad() const { return m_tkrad; }

    /////////////////////////////////////////////////////////
    // Stereo View

    int getStereoMode() const {
      return m_curcam.m_nStereoMode;
    }
    virtual void setStereoMode(int nMode);

    double getStereoDist() const {
      return m_curcam.m_fStereoDist;
    }
    void setStereoDist(double d) {
      if (!qlib::isNear4(m_curcam.m_fStereoDist, d)) {
        m_curcam.m_fStereoDist = d;
	setUpdateFlag();
      }
    }

    /// Query HW stereo impl
    virtual bool hasHWStereo() const;

    // void setBgColor(const LColor &vec);

    /////////////////////////////////////////////////////////
    // Projection

    /// setup the projection matrix
    virtual void setUpProjMat(int w, int h) =0;
    
    /// ID for setUpModelMat() method
    enum {
      MM_NORMAL=0,
      MM_STEREO_LEFT=1,
      MM_STEREO_RIGHT=2
    };

    /// Setup the projection matrix for stereo
    /// @param nid==MM_NORMAL : normal mode
    ///        nid==MM_STEREO_LEFT : stereo left eye
    ///        nid==MM_STEREO_RIGHT : stereo right eye
    virtual void setUpModelMat(int nid) =0;

    /// reverse projection from view to world coord
    void convZTrans(int idz, Vector4D &vec);
    void convXYTrans(double dx, double dy, Vector4D &vec);

    /////////////////////////////////////////////////////////
    // Viewport size

    /// get view width in pixel
    int getWidth();
    
    /// get view height in pixel
    int getHeight();
    
    /// Set view size (without firing events and changing matrices)
    void setViewSize(int w, int h) {
      m_nWidth = w;
      m_nHeight = h;
    }

    /////////////////////////////////////////////////////////
    // Events

    /// Lowlevel device event: add user-input device event listener 
    int addListener(InDevListener *p);
    // int addListener(qlib::LSCBPtr scb);

    /// Lowlevel device event: remove user-input device event listener
    bool removeListener(InDevListener *p);
    bool removeListener(int nid);

    /// View size was changed to (cx,cy)
    virtual void sizeChanged(int cx, int cy);

    // /// Hilevel view event
    // int addViewListener(ViewEventListener *pL);
    // bool removeViewListener(ViewEventListener *pL);
    void fireViewEvent(ViewEvent &ev);

    /////////////////////////////////////////////////////////
    // InDevEvent message handlers
    
    /// mouse drag start event
    virtual bool mouseDragStart(InDevEvent &);
    
    /// mouse drag move event
    virtual bool mouseDragMove(InDevEvent &);
    
    /// mouse drag end event
    virtual bool mouseDragEnd(InDevEvent &);
    
    /// mouse click event (L,M,R button)
    virtual bool mouseClicked(InDevEvent &);
    
    /// mouse double click event (L,M,R button)
    virtual bool mouseDoubleClicked(InDevEvent &);
    
    /// mouse double click event (L,M,R button)
    virtual bool mouseWheel(InDevEvent &);
    
    ////////////////////////////////////////////////
    // Hit test operations
    
    /// Perform renderer hittest
    virtual LString hitTest(int x, int y) =0;
    
    /// Perform hittest by rectangle (in screen coordinate system)
    /// @param bNearest only returns the hittest result for the nearest renderer
    virtual LString hitTestRect(int x, int y, int w, int h, bool bNearest) =0;

    ////////////////////////////////////////////////
    // Framebuffer operations
    
    virtual void readPixels(int x, int y, int width, int height, char *pbuf, int nbufsize, int ncomp);
    
    /// Create a new off-screen view compatible with this view
    virtual View *createOffScreenView(int w, int h, int aa_depth);

  private:
    
    /////////////////////////////////////////////////////////////
    // helper methods for InDevEvent handling
    
    // void clickHelper(InDevEvent &ev, bool fDbl);
    void zoomSlab(int delslab, int delzoom);
    void rotTranZ(int delrot, int deltran);
    void rotXY(int posx, int posy,
	       int delx, int dely,
	       double width, double height);
    // static void projSphere(Vector3D &vec, double tkrad);

  public:
    /// Fire Input-device event (invoked by impl)
    void fireInDevEvent(InDevEvent &ev);

    /////////////////////////////////////////////////////////////
    // Utility routines

  private:
    bool m_bUpdateRequired;

  public:

    /// Rotate view around axes (ax, ay, az are in degree units)
    void rotateView(double ax, double ay, double az);

    /// Translate view
    void translateView(double x, double y, double z);

    /** Create quaternion rotation (x1,y1) --> (x2,y2) */
    void trackBallMove(double curX, double curY, double prevX, double prevY);

    /** Calculate the track-ball rotation */
    static
    qlib::LQuat getTrackRotQuat(double curX, double curY,
			 double prevX, double prevY, double tkrad);
      
    bool safeSetCurrent();

    gfx::DisplayContext *getSiblingCtxt();

    void setUpdateFlag() { m_bUpdateRequired = true; }
    bool getUpdateFlag() const { return m_bUpdateRequired; }
    // void setUpdateFlag() {}
    void clearUpdateFlag() { m_bUpdateRequired = false; }


    void checkAndUpdate() {
      if (m_bUpdateRequired) {
        drawScene();
      }
      clearUpdateFlag();
    }

    void forceRedraw() {
      drawScene();
      clearUpdateFlag();
    }

    bool saveTo(CameraPtr rcam) const;
    bool loadFrom(CameraPtr rcam);

    //////////
    // for property event propagation
    virtual qlib::uid_t getRootUID() const;

    void setCursor(const LString &cursor);
    LString getCursor() const { return m_cursorName; }

    ////////////////////////////////////////
    // Drawing object (for UI) support
  private:
    typedef std::map<LString, DrawObjPtr> drawobjtab_t;
    drawobjtab_t m_drawObjTab;

  public:
    DrawObjPtr getDrawObj(const LString &clsname);

    void showDrawObj(DisplayContext *pdc);
    void showDrawObj2D(DisplayContext *pdc);

    ////////////////////////////////////////
    // Static methods

    // static bool init(qlib::LClass *pcls);
    // static void fini(qlib::LClass *pcls);

    /// Create system-dependent view object
    static View *createView();

  };

}

#endif
