// -*-Mode: C++;-*-
//
// Style/color database
//
// $Id: StyleMgr.cpp,v 1.5 2011/04/18 14:06:23 rishitani Exp $

#include <common.h>

#include "StyleMgr.hpp"
#include "StyleSet.hpp"
#include "StyleFile.hpp"
#include "StyleSupports.hpp"

#include <gfx/SolidColor.hpp>
#include <gfx/NamedColor.hpp>

#include <qlib/FileStream.hpp>
#include <qlib/LDOM2Stream.hpp>

#define DELIM STYLEMGR_DB_DELIM

SINGLETON_BASE_IMPL(qsys::StyleMgr);

using namespace qsys;

using qlib::LDom2Node;
using gfx::SolidColor;
using gfx::SolidColorPtr;
using gfx::AbstractColor;
using gfx::NamedColor;

//////////////////////////////////////////////////
// Static initializer/finalizer (invoked upon startup/termination)

namespace {
  // Named color resolver using style manager
  class NamedColResImpl : public gfx::NamedColorResolver
  {
  public:
    StyleMgr *m_pSM;

    NamedColResImpl() {}

    /// Resolve named color
    virtual ColorPtr getColor(const LString &rkey)
    {
      return m_pSM->getColor(rkey);
    }

    /// Resolve named color with scene ID
    virtual ColorPtr getColor(const LString &rkey, qlib::uid_t nScopeID)
    {
      return m_pSM->getColor(rkey, nScopeID);
    }

    /// Get current context ID
    virtual qlib::uid_t getContextID()
    {
      return m_pSM->getContextID();
    }

  };

  NamedColResImpl *gRes;
}

//static
bool StyleMgr::init()
{
  bool res = qlib::SingletonBase<StyleMgr>::init();

  gRes = new NamedColResImpl();
  gRes->m_pSM = getInstance();
  NamedColor::setResolver(gRes);

  return res;
}

//static
void StyleMgr::fini()
{
  NamedColor::setResolver(NULL);
  delete gRes;
  qlib::SingletonBase<StyleMgr>::fini();
}

/////////////////////////

StyleMgr::StyleMgr()
{
  // global context
  m_curCtxtStack.push_front(qlib::invalid_uid);

  // create global context
  m_pGlob = getCreateStyleList(qlib::invalid_uid);

  m_pLsnrs = new StyleEventCaster;
}

StyleMgr::~StyleMgr()
{
  BOOST_FOREACH (data2_t::value_type &v, m_data2) {
    if (v.second!=NULL)
      delete v.second;
  }

  delete m_pLsnrs;

  MB_DPRINTLN("StyleMgr> m_data destructor OK.");
}

/////////////////////////

StyleList *StyleMgr::getCreateStyleList(qlib::uid_t nid)
{
  data2_t::const_iterator iter = m_data2.find(nid);
  if (iter==m_data2.end()) {
    StyleList *pNew = new StyleList;
    m_data2.insert(data2_t::value_type(nid, pNew));
    return pNew;
  }

  return iter->second;
}

void StyleMgr::destroyContext(qlib::uid_t nid)
{
  if (nid == qlib::invalid_uid) return;
  
  data2_t::iterator iter = m_data2.find(nid);
  if (iter==m_data2.end()) {
    return;
  }
  StyleList *pSet = iter->second;
  m_data2.erase(iter);
  delete pSet;
}

//////////////////////////////////////////////////////////////////////////

ColorPtr StyleMgr::getColor(const LString &rkey, qlib::uid_t nScopeID)
{
  StyleList *pSL = getCreateStyleList(nScopeID);

  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    ColorPtr rval;
    if (pSet->getColor(rkey, rval)) {
      return rval;
    }
  }  

  // check global context
  if (nScopeID!=qlib::invalid_uid)
    return getColor(rkey, qlib::invalid_uid);
  
  // not found
  return ColorPtr();
}

ColorPtr StyleMgr::getColor(const LString &rkey)
{
  //return getColor(rkey, m_nCurCtxtID);
  return getColor(rkey, getContextID());
}

Material *StyleMgr::getMaterial(const LString &mat_id, qlib::uid_t nScopeID)
{
  StyleList *pSL = getCreateStyleList(nScopeID);

  Material *pMat;
  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    pMat = pSet->getMaterial(mat_id);
    if (pMat!=NULL)
      return pMat;
  }  

  // check global context
  if (nScopeID!=qlib::invalid_uid)
    return getMaterial(mat_id, qlib::invalid_uid);
  
  // not found
  return NULL;
}

LString StyleMgr::getMaterial(const LString &mat_id, const LString &rend_type)
{
  Material *pMat = getMaterial(mat_id, getContextID());
  if (pMat==NULL)
    return LString();
  
  return pMat->getDepValue(rend_type);
}

double StyleMgr::getMaterial(const LString &mat_id, int nType)
{
  Material *pMat = getMaterial(mat_id, getContextID());
  if (pMat==NULL)
    return -1.0;
  
  return pMat->getSysValue(nType);
}

LString StyleMgr::getStrImpl(const LString &key, qlib::uid_t nScopeID)
{
  StyleList *pSL = getCreateStyleList(nScopeID);

  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    LString rval;
    if (pSet->getString(key, rval)) {
      return rval;
    }
  }  

  // check global context
  if (nScopeID!=qlib::invalid_uid)
    return getStrImpl(key, qlib::invalid_uid);
  
  // not found
  return LString();
}

LString StyleMgr::getConfig(const LString &cfg_id, const LString &rend_type)
{
  LString key = cfg_id+DELIM+rend_type+DELIM+"cfg";
  // return getStrImpl(key, m_nCurCtxtID);
  return getStrImpl(key, getContextID());
}

LString StyleMgr::getStrData(const LString &cat, const LString &key, qlib::uid_t nScopeID)
{
  LString ckey = key + DELIM + "string" +DELIM + cat;
  return getStrImpl(ckey, nScopeID);
}

ColorPtr StyleMgr::compileColor(const LString &rep, qlib::uid_t nScopeID)
{
  pushContextID(nScopeID);
  ColorPtr rval(AbstractColor::fromStringS(rep));
  popContextID();
  return rval;
}

//////////////////////////////////////////////////////////////////////////////
// Database Query methods

LString StyleMgr::getStrDataDefsJSON(const LString &cat, qlib::uid_t nScopeID)
{
  StyleList *pSL = getCreateStyleList(nScopeID);

  LString postfix = LString(DELIM) + "string" +DELIM + cat;
  int nPsLen = postfix.length();
  LString rval = "[";

  bool bfirst = true;
  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    StyleSet::strdata_iterator iter = pSet->strBegin();
    StyleSet::strdata_iterator eiter = pSet->strEnd();
    for (; iter!=eiter; ++iter) {
      const LString &fkey = iter->first;
      if (!fkey.endsWith(postfix)) continue;
      LString key = fkey.substr(0, fkey.length()-nPsLen);

      if (!bfirst)
        rval += ",";
      else
        bfirst = false;

      rval += "\""+key.escapeQuots()+"\"";
    }
  }  

  rval += "]";
  return rval;
}

LString StyleMgr::getColorDefsJSON(qlib::uid_t nScopeID)
{
  StyleList *pSL = getCreateStyleList(nScopeID);

  LString rval = "[";

  bool bfirst = true;
  BOOST_FOREACH(StyleList::value_type pSet, *pSL) {
    StyleSet::coldata_iterator iter = pSet->colBegin();
    StyleSet::coldata_iterator eiter = pSet->colEnd();
    for (; iter!=eiter; ++iter) {
      const LString &fkey = iter->first;
  
      if (!bfirst)
        rval += ",";
      else
        bfirst = false;

      rval += "\""+fkey.escapeQuots()+"\"";
    }
  }  

  rval += "]";
  return rval;
}

bool StyleMgr::scrLoadFile(const LString &path, qlib::uid_t scope)
{
  StyleFile sfile;
  return sfile.loadFile(path, scope);
}

/////////////////////

void StyleMgr::addListener(StyleEventListener *pLsnr)
{
  // TO DO: listen for related styleset's event
  m_pLsnrs->add(pLsnr);
}

void StyleMgr::removeListener(StyleEventListener *pLsnr)
{
  m_pLsnrs->remove(pLsnr);
}

void StyleMgr::fireEvent(StyleEvent &ev)
{
  m_pLsnrs->replicaFire(ev);
}

LString StyleMgr::getStyleNamesJSON(qlib::uid_t nSceneID)
{
  LString rval = "[";
  LString postfix = LString(DELIM) + "style";
  int nPsLen = postfix.length();

  StyleList *pSL = getCreateStyleList(nSceneID);
  MB_ASSERT(pSL!=NULL);

  bool bfirst = true;
  BOOST_FOREACH (StyleSet *pSet, *pSL) {
    StyleSet::data_iterator iter = pSet->dataBegin();
    StyleSet::data_iterator eiter = pSet->dataEnd();

    for (; iter!=eiter; ++iter) {
      const LString &fkey = iter->first;
      if (!fkey.endsWith(postfix)) continue;
      LString key = fkey.substr(0, fkey.length()-nPsLen);

      LDom2Node *pNode = iter->second;
      LString desc = pNode->getStrAttr("desc");

      if (!bfirst)
        rval += ",";
      else
        bfirst = false;

      rval += "{\"name\":";
      rval += "\""+key.escapeQuots()+"\",";
      rval += "\"desc\":";
      rval += "\""+desc.escapeQuots()+"\"}";
    }
  }

  rval += "]";
  return rval;
}

