/*
 * Cueplot: a GUI front-end to gnuplot
 * Copyright (C) 2007, 2008 Muneyuki Noguchi
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation, 
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#ifndef PLOT_DATA_WIDGET_INCLUDED
#define PLOT_DATA_WIDGET_INCLUDED

class Plot;
class PlotListModel;
class PlotListView;
class PlotOptionDialog;

#include <QtGui/QWidget>
class QItemSelectionModel;
class QModelIndex;
class QPushButton;
class QStringList;

//! プロットする項目を扱うウィジェット
class PlotDataWidget : public QWidget
{
	Q_OBJECT

	public:
		explicit PlotDataWidget(QWidget *parent = 0);

		/*!
		 * \brief リスト中に項目があるかどうかを返す
		 * \return 項目があれば true
		 */
		bool hasItem() const;
		/*!
		 * \brief プロットする項目に対応するコマンドを返す
		 * \return プロットする項目に対応するコマンド
		 */
		QStringList commandList() const;

		//! ダイアログをすべて閉じる
		void closeDialog();
		
	public slots:
		/*!
		 * \brief 項目を追加する
		 * \param plot 追加する項目
		 */
		void addData(const Plot &plot);
		/*!
		 * \brief 項目が追加できるかどうかを設定する
		 * \param enable 追加できるようにするときは true
		 */
		void setAdditionEnabled(bool enable);

	private slots:
		//! 項目に対するオプションを設定するダイアログを表示する
		void setPlotOptions();
		//! 選択されている項目を上に移動させる
		void setDataUp();
		//! 選択されている項目を下に移動させる
		void setDataDown();
		//! 選択されている項目を削除する
		void deleteData();
		/*!
		 * \brief 項目の変更をダイアログに適用する
		 * \param topLeft 変更対象の項目のうち最も左上の項目
		 * \param bottomRight 変更対象の項目のうち最も右下の項目
		 */
		void changeDialog(const QModelIndex &topLeft, 
				const QModelIndex &bottomRight);
		/*!
		 * \brief 挿入した項目に対応するダイアログを挿入する
		 * \param start 最初の項目
		 * \param end 最後の項目
		 * addData() が呼ばれたときやファイルがドロップされたときに
		 * この関数が呼ばれる。
		 * \sa addData()
		 */
		void insertDialog(const QModelIndex &parent, int start, int end);
		/*!
		 * \brief 削除した項目に対応するダイアログを削除する
		 * \param start 最初の項目
		 * \param end 最後の項目
		 * \sa deleteData()
		 */
		void removeDialog(const QModelIndex &parent, int start, int end);
		/*!
		 * \brief 移動された項目に対応するダイアログの順序を移動させる
		 * \param from 項目の移動前の位置
		 * \param to 項目の移動後の位置
		 * moveData() が呼ばれたときやユーザがドラッグドロップしたときに
		 * この関数が呼ばれる。
		 * \sa moveData()
		 */
		void moveDialog(int from, int to);
		//! リストの状態に応じてボタンの有効/無効を設定する
		void setPushButton();
		/*!
		 * \brief 指定されたダイアログに対応する項目のオプションを変える
		 * \param dialog オプションを変えたダイアログ
		 * \param plot オプション
		 */
		void changeOption(const PlotOptionDialog &dialog, const Plot &plot);

	signals:
		/*!
		 * \brief このシグナルはプロット可能かどうかが変わったときに発信される
		 * \param enabled 可能になったときは true
		 */
		void setPlotEnabled(bool enabled);
		/*!
		 * \brief このシグナルはコマンドを実行するときに発信される
		 * \param command 実行するコマンド
		 */
		void executed(const QString &command);
		//! このシグナルは項目が追加されるときに発信される
		void dataAdded();

	private:
		// don't implement these
		PlotDataWidget(const PlotDataWidget &);
		PlotDataWidget &operator=(const PlotDataWidget &);

		/*!
		 * \brief 項目を移動させる
		 * \param up 項目を上に移動させるときは true
		 */
		void moveData(bool up);

		//! リスト中の項目の状態に応じて、プロットできるかどうかを設定する
		void setAction();
		QList<PlotOptionDialog *> dialogList;
		QPushButton *addPushButton;
		QPushButton *optionPushButton;
		QPushButton *upPushButton;
		QPushButton *downPushButton;
		QPushButton *deletePushButton;
		PlotListModel *plotListModel;
		PlotListView *plotListView;
		QItemSelectionModel *selectionModel;
};
#endif // PLOT_DATA_WIDGET_INCLUDED
