/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.account;

import java.util.ArrayList;
import java.util.List;

import org.apache.cayenne.access.DataContext;
import org.apache.cayenne.access.Transaction;
import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.eip.cayenne.om.account.AvzMPostPosition;
import com.aimluck.eip.cayenne.om.account.EipMPost;
import com.aimluck.eip.common.ALAbstractCheckList;
import com.aimluck.eip.common.ALEipManager;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 部署を複数削除するためのクラス． <BR>
 * このとき部署に関連づけられているグループも削除する．
 * 
 */
public class AccountPostMultiDelete extends ALAbstractCheckList {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(AccountPostMultiDelete.class.getName());

  /**
   * 一括削除処理
   * 
   * @param action
   *            アクションクラス
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @param values
   *            部署IDリスト
   * @param msgList
   *            エラーメッセージリスト
   */
  // @Override
  // protected boolean action(RunData rundata, Context context,
  // List<String> values, List<String> msgList) {
  // try {
  // // オブジェクトモデルを取得
  // List<EipMPost> list = getEipMPosts(rundata, context, values);
  // if (list == null || list.size() == 0) {
  // return false;
  // }
  //
  // int listsize = list.size();
  // for (int i = 0; i < listsize; i++) {
  // EipMPost record = list.get(i);
  // // グループからユーザーを削除
  // List<ALEipUser> users = ALEipUtils.getUsers(record.getGroupName());
  // int size = users.size();
  // for (int j = 0; j < size; j++) {
  // JetspeedSecurity.unjoinGroup(
  // users.get(j).getName().getValue(),
  // record.getGroupName());
  // }
  //
  // // グループを削除
  // JetspeedSecurity.removeGroup(record.getGroupName());
  //
  // // 部署を削除
  // Database.deleteAll(record);
  // }
  //
  // Database.commit();
  //
  // // singletonオブジェクトのリフレッシュ
  // ALEipManager.getInstance().reloadPost();
  //
  // } catch (Exception ex) {
  // Database.rollback();
  // logger.error("部署情報の一括削除に失敗しました。", ex);
  // return false;
  // }
  // return true;
  // }
  @Override
  protected boolean action(RunData rundata, Context context,
      List<String> values, List<String> msgList) {

    // 標準のトランザクション（オートコミット）
    Transaction baseTx = Transaction.getThreadTransaction();

    DataContext dataContext = DataContext.getThreadDataContext();

    // 自己管理トランザクション
    Transaction tx =
      Transaction.internalTransaction(DataContext
        .getThreadDataContext()
        .getParentDataDomain()
        .getTransactionDelegate());

    // 標準のトランザクションを自己管理トランザクションに置き換えます。
    Transaction.bindThreadTransaction(tx);

    try {

      // トランザクション開始
      tx.begin();

      // オブジェクトモデルを取得
      List<EipMPost> list = getEipMPosts(rundata, context, values);
      if (list == null || list.size() == 0) {
        return false;
      }

      List<EipMPost> deletePostList = new ArrayList<EipMPost>();
      deletePostList.addAll(list);

      List<EipMPost> tmpDeleteChildPostList = new ArrayList<EipMPost>();
      for (EipMPost e : deletePostList) {
        if (e.getParentPostId() == 0) {
          // 部署１の場合は、部署２を検索してあればリストに含める。
          Expression exp =
            ExpressionFactory.matchExp(EipMPost.PARENT_POST_ID_PROPERTY, e
              .getPostId());
          List<EipMPost> childList =
            Database.query(EipMPost.class, exp).fetchList();
          if (childList != null && childList.size() > 0) {
            tmpDeleteChildPostList.addAll(childList);
          }
        }
      }
      // 部署2のリストを追加する。
      deletePostList.addAll(tmpDeleteChildPostList);

      // 削除対象部署リストから重複を除く
      deletePostList = removeDuplicates(deletePostList);

      // 削除対象部署リストの部署を順に削除する。
      for (EipMPost e : deletePostList) {

        // グループからユーザーを削除
        List<ALEipUser> users = ALEipUtils.getUsers(e.getGroupName());
        int size = users.size();
        for (int i = 0; i < size; i++) {
          JetspeedSecurity.unjoinGroup(users.get(i).getName().getValue(), e
            .getGroupName());
        }
        // グループを削除
        JetspeedSecurity.removeGroup(e.getGroupName());

        // 部署表示順位を削除
        Database.delete(e.getAvzMPostPositionArray());

        // 部署を削除
        Database.delete(e);
      }

      dataContext.commitChanges();

      // add by motegi 他の部署の順番を変更する．
      SelectQuery<AvzMPostPosition> p_query =
        Database.query(AvzMPostPosition.class);
      p_query.orderAscending(AvzMPostPosition.POSITION_PROPERTY);
      List<AvzMPostPosition> entityPositions = p_query.fetchList();

      if (entityPositions != null && entityPositions.size() > 0) {
        int size = entityPositions.size();
        for (int i = 0; i < size; i++) {
          AvzMPostPosition entityPosition = entityPositions.get(i);
          entityPosition.setPosition(Integer.valueOf(i + 1));
        }
      }

      Database.commit();

      // singletonオブジェクトのリフレッシュ
      ALEipManager.getInstance().reloadPost();

    } catch (Exception ex) {
      Database.rollback();
      String msg = "部署情報の一括削除に失敗しました。";
      logger.error(msg, ex);
      msgList.add(msg);
      return false;
    } finally {
      // 自己管理トランザクションを標準のトランザクションに戻します。
      Transaction.bindThreadTransaction(baseTx);
    }
    return true;
  }

  /**
   * 部署情報リスト取得
   * 
   * @param rundata
   *            JetSpeedランデータ
   * @param context
   *            JetSpeedコンテキスト
   * @return 部署情報リスト
   * @throws Exception
   */
  private List<EipMPost> getEipMPosts(RunData rundata, Context context,
      List<String> values) throws Exception {
    List<EipMPost> list = null;

    try {
      if (values == null || values.size() == 0) {
        logger.debug("values are empty...");
        return null;
      }

      Expression exp =
        ExpressionFactory.inDbExp(EipMPost.POST_ID_PK_COLUMN, values);
      SelectQuery<EipMPost> query = Database.query(EipMPost.class, exp);
      list = query.fetchList();
      if (list == null || list.size() == 0) {
        logger.debug("Not found ID...");
        return null;
      }
    } catch (Exception ex) {
      // change start
      // logger.error("Exception", ex);
      // list = null;
      throw ex;
      // change end
    }
    return list;
  }

  /**
   * 削除対象部署リストから重複を除く
   * 
   * @param srcList
   *            削除対象部署リスト
   * @return 重複除去後の削除対象部署リスト
   */
  private List<EipMPost> removeDuplicates(List<EipMPost> srcList) {
    List<EipMPost> destList = new ArrayList<EipMPost>();
    for (EipMPost s : srcList) {
      boolean found = false;
      for (EipMPost d : destList) {
        if (s.getPostId().intValue() == d.getPostId().intValue()) {
          found = true;
          break;
        }
      }
      if (!found) {
        destList.add(s);
      }
    }
    return destList;
  }
}
